// CmOrdArr.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Ordered array definition.
// -----------------------------------------------------------------

#ifndef _CMORDARR_H
#define _CMORDARR_H

#include <cm/include/cmcont.h>

class CmOrderedArrayIterator;                  // Iterator class stub.

class CmOrderedArray : public CmContainer {    // Ordered array definition.
public:
  CmOrderedArray(unsigned = 0, unsigned = 0);  // Default array constructor.
  CmOrderedArray(const CmOrderedArray&);       // Array copy constructor.
 ~CmOrderedArray();                            // Array destructor.

  CmOrderedArray& operator=(const CmOrderedArray&);  // Assignment operator.

  CmObject* operator[](int) const;             // Indexing operator.
                                               // (Cannot be used as lvalue).

  void        delta      (unsigned);           // Set delta value.
  unsigned    delta      () const;             // Get delta value.
  int         total      () const;             // Return number of objects.
  CmObject*   at         (int) const;          // Get object at index.
  Bool        add        (CmObject*);          // Add object to array in order.
  Bool        remove     (CmObject*);          // Remove equal object.
  Bool        removeAt   (int);                // Remove object at index.
  int         index      (CmObject*) const;    // Get index of object.
  int         shouldGo   (CmObject*) const;    // Get index for potential obj.
  CmObject*   lookup     (CmObject*) const;    // Look for equal object.
  Bool        contains   (CmObject*) const;    // Object is in array?
  unsigned    occurrences(CmObject*) const;    // How many occurrences?
  void        removeAll  ();                   // Remove all objects.
  Bool        resize     (unsigned);           // Resize the array.
  Bool        isEmpty    () const;             // Is array empty?
  CmIterator* newIterator() const;             // Get array iterator.

  Bool write(CmReserveFile&) const;            // Write to reserve file.
  Bool read (CmReserveFile&);                  // Read from reserve file.

  CMOBJECT_DEFINE(CmOrderedArray, CmContainer) // Define object funcs.

protected:
  unsigned   _delta;                           // Delta value.
  unsigned   _total;                           // Number of objects.
  CmObject **_entries;                         // Array of object pointers.
  friend     CmOrderedArrayIterator;           // Iterator can access,
};

class CmOrderedArrayIterator : public CmIterator {  // Iterator definition.
public:
  CmOrderedArrayIterator(const CmOrderedArray& A)   // Iterator constructor.
                 : _array(A), _index(0) {}

  Bool      done    () const;                  // Check if done iterating.
  CmObject* next    ();                        // Return and advance.
  CmObject* previous();                        // Return and backup.
  CmObject* current () const;                  // Return current object.
  void      first   ();                        // Move to first object.
  void      last    ();                        // Move to last object.

  CMOBJECT_DEFINE(CmOrderedArrayIterator, CmIterator) // Define object funcs.

protected:
  const CmOrderedArray& _array;                // Array being iterated.
  int                   _index;                // Current array index.
  friend                CmOrderedArray;        // Array class can access.
};

// "delta" sets a new delta value for automatic growing.
inline void CmOrderedArray::delta(unsigned dt)
{ _delta = dt; }

// "delta" returns the current delta value.
inline unsigned CmOrderedArray::delta() const
{ return _delta; }

// "total" returns the number of objects added to this array.
inline int CmOrderedArray::total() const
{ return _total; }

// "at" returns the object at the specified index.
inline CmObject* CmOrderedArray::at(int idx) const
{ return (idx >= 0 && idx < _total) ? _entries[idx] : NULL; }

// "[]" returns the object at the specified index.  Cannot be used as lvalue.
inline CmObject* CmOrderedArray::operator[](int idx) const
{ return at(idx); }

// "contains" checks if the array contains an object equal to the input.
inline Bool CmOrderedArray::contains(CmObject* pObj) const
{ return (index(pObj) > -1) ? TRUE : FALSE; }

// "done" checks to see if we can iterate any further.
inline Bool CmOrderedArrayIterator::done() const
{ return (_index >= _array._total || _index < 0); }

// "next" returns the current object and advances the iterator.
inline CmObject* CmOrderedArrayIterator::next()
{ return (_index < _array._total) ? _array._entries[_index++] : NULL; }

// "previous" returns the current object and decrements the iterator.
inline CmObject* CmOrderedArrayIterator::previous()
{ return (_index >= 0) ? _array._entries[_index--] : NULL; }

// "current" returns the current object pointed to by the iterator.
inline CmObject* CmOrderedArrayIterator::current() const
{ return (_index < _array._total) ? _array._entries[_index] : NULL; }

// "first" moves the iterator to the first object.
inline void CmOrderedArrayIterator::first()
{ _index = 0; }

// "last" moves the iterator to the last object.
inline void CmOrderedArrayIterator::last()
{ _index = _array._total-1; }

#endif
