// CmList.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Singly linked list definition.
// -----------------------------------------------------------------

#ifndef _CMLIST_H
#define _CMLIST_H

#include <cm/include/cmcont.h>

class CmLink;                                     // List link class stub.
class CmLinkedListIterator;                       // Iterator class stub.

class CmLinkedList : public CmContainer {         // Linked list definition.
public:
  CmLinkedList() : _first(NULL), _last(NULL) {}   // Construct list.
  CmLinkedList(const CmLinkedList&);              // List copy constructor.
 ~CmLinkedList();                                 // List destructor.

  CmLinkedList& operator= (const CmLinkedList&);  // Assignment operator.
  CmObject*     operator[](int) const;            // Get object by index.

  Bool        add         (CmObject*);            // Add to end of list.
  Bool        append      (CmObject*);            // Add to end of list.
  Bool        prepend     (CmObject*);            // Add to start of list.
  Bool        insertAfter (CmObject*, CmObject*); // Insert after another.
  Bool        insertBefore(CmObject*, CmObject*); // Insert before another.
  Bool        remove      (CmObject*);            // Remove object from list.
  CmObject*   removeFirst ();                     // Remove first object.
  CmObject*   removeLast  ();                     // Remove last object.
  CmObject*   lookup      (CmObject*) const;      // Find equal object.
  CmObject*   first       () const;               // Get first object in list.
  CmObject*   last        () const;               // Get last object in list.
  Bool        contains    (CmObject*) const;      // See if list contains.
  unsigned    occurrences (CmObject*) const;      // Count occurrences of obj.
  void        removeAll   ();                     // Clear the list.
  CmIterator* newIterator () const;               // Create list iterator.

  CMOBJECT_DEFINE(CmLinkedList, CmContainer)      // Define object funcs.

protected:
  CmLink *_first;                                 // First link in list.
  CmLink *_last;                                  // Last link in list.
  friend  CmLinkedListIterator;                   // Iterator can access.
};

class CmLinkedListIterator : public CmIterator {  // Iterator definition.
public:
  CmLinkedListIterator(const CmLinkedList& L)     // Iterator construct.
                      : _list(L), _link(L._first) {}

  Bool      done    () const;                     // Check if end.
  CmObject* next    ();                           // Return and advance.
  CmObject* previous();                           // Return and backup.
  CmObject* current () const;                     // Get current object.
  void      first   ();                           // Move to first object.
  void      last    ();                           // Move to last object.

  CMOBJECT_DEFINE(CmLinkedListIterator, CmIterator)  // Define object funcs.

protected:
  CmLink             *_link;                      // Current list link.
  const CmLinkedList& _list;                      // List being iterated.
  friend              CmLinkedList;               // List class can access.
};

class CmLink {                                    // List link class definition.
protected:
  CmLink(CmObject *O) : _next(NULL), _data(O) {}  // Link constructor.

  CmLink   *_next;                                // Pointer to next link.
  CmObject *_data;                                // Pointer to object.
  friend    CmLinkedList;                         // List class can access.
  friend    CmLinkedListIterator;                 // Iterator class can access.
};

// "first" returns the first object in the list.
inline CmObject* CmLinkedList::first() const
{ return (_first) ? _first->_data : NULL; }

// "last" returns the last object in the list.
inline CmObject* CmLinkedList::last() const
{ return (_last) ? _last->_data : NULL; }

// "done" checks if the iterator can advance any further.
inline Bool CmLinkedListIterator::done() const
{ return _link == NULL; }

// "current" returns the current object in the list.
inline CmObject* CmLinkedListIterator::current() const
{ return (_link) ? _link->_data : NULL; }

// "first" resets the iterator to the start of the list.
inline void CmLinkedListIterator::first()
{ _link = _list._first; }

// "last" moves the iterator to the last object.
inline void CmLinkedListIterator::last()
{ _link = _list._last; }

#endif
