// CmIter.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Abstract container iterator definition.
// -----------------------------------------------------------------

#ifndef _CMITER_H
#define _CMITER_H

#include <cm/include/cmobject.h>

class CmIterator : public CmObject {                  // Iterator definition.
public:
  virtual Bool      done    () const = 0;             // Check if end of cont.
  virtual CmObject* next    ()       = 0;             // Return and advance.
  virtual CmObject* previous()       = 0;             // Return and backup.
  virtual CmObject* current () const = 0;             // Return current object.
  virtual void      first   ()       = 0;             // Set to first object.
  virtual void      last    ()       = 0;             // Set to last object.

  CmObject* nextOccurrence    (CmObject*);            // Go to next equal.
  CmObject* previousOccurrence(CmObject*);            // Go to prevous equal.

  operator int() const;                               // Check if not end.

  CmObject* operator++();                             // Advance and return.
  CmObject* operator++(int);                          // Return and advance.
  CmObject* operator+=(int);                          // Advance n places.
  CmObject* operator--();                             // Backup and return.
  CmObject* operator--(int);                          // Return and backup.
  CmObject* operator-=(int);                          // Backup n places.
  CmObject* operator()() const;                       // Return current object.

  CMOBJECT_ABSTRACT(CmIterator, CmObject)             // Define object funcs.
};

// "int" conversion operator returns zero if at the end.
inline CmIterator::operator int() const
{ return !done(); }

// "++" advances the iterator to the next object and returns it.
inline CmObject* CmIterator::operator++()
{ next(); return current(); }

// "++" returns the current object and advances the iterator to the next.
inline CmObject* CmIterator::operator++(int)
{ return next(); }

// "--" decrements the iterator by one and returns the object.
inline CmObject* CmIterator::operator--()
{ previous(); return current(); }

// "--" returns the current object and decrements the iterator by one.
inline CmObject* CmIterator::operator--(int)
{ return previous(); }

// "()" returns the current iterator object.
inline CmObject* CmIterator::operator()() const
{ return current(); }

#endif
