// CmHash.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Hash table definition.
// -----------------------------------------------------------------

#ifndef _CMHASH_H
#define _CMHASH_H

#include <cm/include/cmcont.h>

class CmHashBucket;                            // Hash bucket class stub.
class CmHashNode;                              // Hash node class stub.
class CmHashTableIterator;                     // Iterator class stub.

class CmHashTable : public CmContainer {       // Hash table definition.
public:
  CmHashTable(unsigned = 10);                  // Default table constructor.
  CmHashTable(const CmHashTable&);             // Table copy constructor.
 ~CmHashTable();                               // Hash table destructor.

  CmHashTable& operator=(const CmHashTable&);  // Assignment operator.

  int         total      () const;             // Number of object in table.
  Bool        add        (CmObject*);          // Add object to container.
  Bool        remove     (CmObject*);          // Remove object.
  CmObject*   lookup     (CmObject*) const;    // Find equal object.
  Bool        contains   (CmObject*) const;    // Table has object?
  unsigned    occurrences(CmObject*) const;    // Count occurrences of object.
  void        removeAll  ();                   // Remove all objects.
  Bool        resize     (unsigned);           // Resize the hash table.
  Bool        isEmpty    () const;             // Is table empty?
  CmIterator* newIterator() const;             // Get hash table iterator.

  Bool write(CmReserveFile&) const;            // Write to reserve file.
  Bool read (CmReserveFile&);                  // Read from reserve file.

  CMOBJECT_DEFINE(CmHashTable, CmContainer)    // Define object funcs.

protected:
  int           _total;                        // Number of objects in table.
  CmHashBucket *_buckets;                      // Array of hash buckets.
  friend        CmHashTableIterator;           // Iterator can access.
};

class CmHashBucket {                           // Bucket class definition.
protected:
  CmHashBucket() : _size(0), _first(NULL), _last(NULL) {}  // Bucket construct.
 ~CmHashBucket() { removeAll(FALSE); }         // Bucket destructor.

  unsigned  size     () const;                 // Return list size.
  Bool      append   (CmObject*);              // Add object to end of list.
  Bool      contains (CmObject*) const;        // See if object is in list.
  Bool      remove   (CmObject*, Bool);        // Remove object from list.
  CmObject* lookup   (CmObject*) const;        // Find equal object in list.
  void      removeAll(Bool);                   // Remove all objects.

  unsigned    _size;                           // List size.
  CmHashNode *_first;                          // First object in list.
  CmHashNode *_last;                           // Last object in list.
  friend      CmHashTable;                     // Table can access.
  friend      CmHashTableIterator;             // Iterator can access.
};

class CmHashNode {                             // Hash bucket node definition.
protected:
  CmHashNode(CmObject* O)                      // Node constructor.
            : _next(NULL), _data(O) {}

  CmHashNode *_next;                           // Next node in list.
  CmObject   *_data;                           // Object pointer.
  friend      CmHashBucket;                    // Bucket can access.
  friend      CmHashTableIterator;             // Iterator can access.
};

class CmHashTableIterator : public CmIterator {  // Iterator definition.
public:
  CmHashTableIterator(const CmHashTable&);     // Iterator constructor.

  Bool      done    () const;                  // Check if done iterating.
  CmObject* next    ();                        // Return and advance.
  CmObject* previous();                        // Return and backup.
  CmObject* current () const;                  // Return current object.
  void      first   ();                        // Move to first object.
  void      last    ();                        // Move to last object.

  CMOBJECT_DEFINE(CmHashTableIterator, CmIterator)  // Define object funcs.

protected:
  const CmHashTable& _table;                   // Table being iterated.
  int                _bucket;                  // Current bucket index.
  CmHashNode        *_node;                    // Current bucket node.
  friend             CmHashTable;              // Table class can access.
};

// "total" returns the number of object contained in the table.
inline int CmHashTable::total() const
{ return _total; }

// "size" returns the hash bucket size.
inline unsigned CmHashBucket::size() const
{ return _size; }

// "done" checks to see if the iterator can move any farther.
inline Bool CmHashTableIterator::done() const
{ return (_node) ? FALSE : TRUE; }

// "current" returns the current object pointed to by the iterator.
inline CmObject* CmHashTableIterator::current() const
{ return (_node) ? _node->_data : NULL; }

#endif
