// CmBTree.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// BTree tree definition.
// -----------------------------------------------------------------

#ifndef _CMBTREE_H
#define _CMBTREE_H

#include <cm/include/cmcont.h>

class CmBTreeNode;                             // Tree node class stub.
class CmBTreeIterator;                         // Tree iterator class stub.

class CmBTree : public CmContainer {           // BTree class definition.
public:
  CmBTree(unsigned = 3);                       // Default tree constructor.
  CmBTree(const CmBTree&);                     // Tree copy constructor.
 ~CmBTree();                                   // Tree destructor.

  CmBTree& operator=(const CmBTree&);          // Assignment operator.

  unsigned order () const;                     // Return tree order.
  unsigned height() const;                     // Return tree height.

  Bool        add        (CmObject*);          // Add object to tree.
  Bool        remove     (CmObject*);          // Remove equal object.
  CmObject*   lookup     (CmObject*) const;    // Find equal object in tree.
  Bool        contains   (CmObject*) const;    // See if tree contains equal.
  unsigned    occurrences(CmObject*) const;    // Count occurrences of equal.
  void        removeAll  ();                   // Remove all objects.
  CmIterator* newIterator() const;             // Create tree iterator.

  CMOBJECT_DEFINE(CmBTree, CmContainer)        // Define object funcs.

protected:
  CmBTreeNode* nodeSearch(CmObject*) const;    // Search for insertion node.
  CmBTreeNode* lookupNode(CmObject*) const;    // Get node where object is.
  CmBTreeNode* lookupAbs (CmObject*) const;    // Get node of exact object.

  static void removeNodes(CmBTreeNode*, Bool); // Remove node children.
  static unsigned minKeys(unsigned);           // Get minimum keys allowed.

  unsigned     _order;                         // Tree order.
  CmBTreeNode *_root;                          // Root tree node.
  friend       CmBTreeIterator;                // Iterator class can access.
};

class CmBTreeNode {                            // Tree node definition.
protected:
  CmBTreeNode(CmBTreeNode*, unsigned);         // Node constructor.
 ~CmBTreeNode();                               // Node destructor.

  int  shouldGo(CmObject*) const;              // Where should object insert.
  int  index   (CmObject*) const;              // Index of equal object.
  int  indexAbs(CmObject*) const;              // Index of absolute object.
  int  index   (CmBTreeNode*) const;           // Index where child is located.
  Bool addAt   (int, CmObject*, CmBTreeNode*, int); // Add object at index.
  Bool removeAt(int);                          // Remove object at index.
  void clear   (int);                          // Clear arrays.

  CmBTreeNode* split(unsigned);                // Split the node.

  CmBTreeNode  *_parent;                       // Parent node.
  CmObject    **_data;                         // (order  ) object pointers.
  CmBTreeNode **_children;                     // (order+1) child nodes.
  int           _total;                        // Number of objects in node.
  friend        CmBTree;                       // Tree class can access.
  friend        CmBTreeIterator;               // Iterator class can access.
};

class CmBTreeIterator : public CmIterator {    // Iterator class definition.
public:
  CmBTreeIterator(const CmBTree &Tr);          // Iterator constructor.

  Bool      done    () const;                  // Check if at end.
  CmObject* next    ();                        // Return and advance.
  CmObject* previous();                        // Return and backup.
  CmObject* current () const;                  // Get current object.
  void      first   ();                        // Move to first object.
  void      last    ();                        // Move to last object.

  CMOBJECT_DEFINE(CmBTreeIterator, CmIterator) // Define object funcs.

protected:
  const CmBTree          &_tree;               // Tree being iterated.
  CmBTreeNode            *_node;               // Current tree node.
  int                     _index;              // Current node object.
  friend                  CmBTree;             // Tree class can access.
};

// "order" returns the current tree order.
inline unsigned CmBTree::order () const
{ return _order; }

// "minKeys" returns the minimum number of keys allowed in a non-
// root node of the tree.
inline unsigned CmBTree::minKeys(unsigned ordr)
{ return (((ordr / 2) * 2) == ordr) ? (ordr / 2) - 1 : (ordr / 2); }

// "done" returns whether or not we are done iterating.
inline Bool CmBTreeIterator::done() const
{ return (!_node); }

// "current" returns the current object pointed to by this iterator.
inline CmObject* CmBTreeIterator::current() const
{ return (_node && _node->_data[_index]) ? _node->_data[_index] : NULL; }

#endif
