// CmArray.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Array definition.
// -----------------------------------------------------------------

#ifndef _CMARRAY_H
#define _CMARRAY_H

#include <cm/include/cmcont.h>

class CmArrayIterator;                       // Iterator class stub.

class CmArray : public CmContainer {         // Array class definition.
public:
  CmArray(unsigned = 0, unsigned = 0);       // Default array constructor.
  CmArray(const CmArray&);                   // Array copy constructor.
 ~CmArray();                                 // Array destructor.

  CmArray&  operator= (const CmArray&);      // Assignment operator.
  CmObject* operator[](int) const;           // Indexing operator.
                                             // (Cannot be used as lvalue).

  void        delta      (unsigned);         // Set delta value.
  unsigned    delta      () const;           // Get delta value.
  int         total      () const;           // Return number of objects.
  CmObject*   at         (int) const;        // Get object at index.
  Bool        add        (CmObject*);        // Append object to array.
  Bool        insertAt   (int, CmObject*);   // Insert object at index.
  Bool        replaceAt  (int, CmObject*);   // Replace object at index.
  Bool        remove     (CmObject*);        // Remove equal object.
  Bool        removeAt   (int);              // Remove object at index.
  int         index      (CmObject*) const;  // Get index of object.
  CmObject*   lookup     (CmObject*) const;  // Look for equal object.
  Bool        contains   (CmObject*) const;  // Object is in array?
  unsigned    occurrences(CmObject*) const;  // How many occurrences?
  void        removeAll  ();                 // Remove all objects.
  Bool        resize     (unsigned);         // Resize the array.
  Bool        isEmpty    () const;           // Is array empty?
  void        quickSort  ();                 // Quick sort the array.
  CmIterator* newIterator() const;           // Get array iterator.
  
  Bool write(CmReserveFile&) const;          // Write to reserve file.
  Bool read (CmReserveFile&);                // Read from reserve file.

  CMOBJECT_DEFINE(CmArray, CmContainer)      // Define object funcs.

protected:
  static int cmpObjs(const void*, const void*); // Quick sort compare func.

  unsigned   _delta;                         // Delta value.
  unsigned   _total;                         // Number of objects.
  CmObject **_entries;                       // Array of object pointers.
  friend     CmArrayIterator;                // Iterator can access,
};

class CmArrayIterator : public CmIterator {  // Iterator definition.
public:
  CmArrayIterator(const CmArray& A)          // Iterator constructor.
                 : _array(A), _index(0) {}

  Bool      done    () const;                // Check if done iterating.
  CmObject* next    ();                      // Return and advance.
  CmObject* previous();                      // Return and backup.
  CmObject* current () const;                // Return current object.
  void      first   ();                      // Move to first object.
  void      last    ();                      // Move to last object.

  CMOBJECT_DEFINE(CmArrayIterator, CmIterator)  // Define object funcs.

protected:
  const CmArray& _array;                     // Array being iterated.
  int            _index;                     // Current array index.
  friend         CmArray;                    // Array class can access.
};

// "delta" sets a new delta value for automatic growing.
inline void CmArray::delta(unsigned dt)
{ _delta = dt; }

// "delta" returns the current delta value.
inline unsigned CmArray::delta() const
{ return _delta; }

// "total" returns the number of objects added to this array.
inline int CmArray::total() const
{ return _total; }

// "at" returns the object at the specified index.
inline CmObject* CmArray::at(int idx) const
{ return (idx >= 0 && idx < _total) ? _entries[idx] : NULL; }

// "[]" returns the object at the specified index.  Cannot be used as lvalue.
inline CmObject* CmArray::operator[](int idx) const
{ return at(idx); }

// "contains" checks if the array contains an object equal to the input.
inline Bool CmArray::contains(CmObject* pObj) const
{ return (index(pObj) > -1) ? TRUE : FALSE; }

// "done" checks to see if we can iterate any further.
inline Bool CmArrayIterator::done() const
{ return (_index >= _array._total || _index < 0); }

// "next" returns the current object and advances the iterator.
inline CmObject* CmArrayIterator::next()
{ return (_index < (_array._total)) ? _array._entries[_index++] : NULL; }

// "previous" returns the current object and decrements the iterator.
inline CmObject* CmArrayIterator::previous()
{ return (_index >= 0) ? _array._entries[_index--] : NULL; }

// "current" returns the current object pointed to by the iterator.
inline CmObject* CmArrayIterator::current() const
{ return (_index < _array._total) ? _array._entries[_index] : NULL; }

// "first" moves the iterator to the first object.
inline void CmArrayIterator::first()
{ _index = 0; }

// "last" moves the iterator to the last object.
inline void CmArrayIterator::last()
{ _index = _array._total-1; }

#endif
