/*  C K U U S X --  "User Interface" common functions. */
 
/*
 Author: Frank da Cruz (fdc@columbia.edu, FDCCU@CUVMA.BITNET),
 Columbia University Center for Computing Activities.
 First released January 1985.
 Copyright (C) 1985, 1990, Trustees of Columbia University in the City of New 
 York.  Permission is granted to any individual or institution to use, copy, or
 redistribute this software so long as it is not sold for profit, provided this
 copyright notice is retained.
*/

/* Includes */

#include <stdio.h>
#include <ctype.h>
#include <signal.h>

#include "ckcdeb.h"
#include "ckcasc.h"
#include "ckcker.h"
#include "ckcfil.h"
#include "ckuusr.h"

/* Variables declared here */

#ifdef DEBUG
char debfil[50];			/* Debugging log file name */
#endif

char pktfil[50];			/* Packet log file name */
char sesfil[50];			/* Session log file name */

#ifdef TLOG
char trafil[50];			/* Transaction log file name */
#endif

char optbuf[50];			/* Options for MAIL or REMOTE PRINT */
char cmdstr[256];			/* Place to build generic command */

static int n;				/* General purpose int */

int success = 1,			/* Command success/failure flag */
    cmdlvl = 0,				/* Command level */
    action,				/* Action selected on command line*/
    sessft = 0,				/* Session log file type, 0 = text */
    pflag = 1;				/* Print prompt & messages */

#ifndef NOMSEND				/* Multiple SEND */
char *msfiles[MSENDMAX];
#endif

/* External variables */

extern int local, quiet, binary, bctu, rptflg, ebqflg, network,
  what, spsiz, urpsiz, wmax, czseen, cxseen, winlo, displa, pflag, timint,
  npad, ebq, ebqflg, bctr, rptq, atcapu, lpcapu, swcapu, wslotn, wslotr, rtimo,
  mypadn, sq, capas, rpsiz, tsecs, dfloc, deblog, pktlog, seslog, xitsta,
  escape, tlevel, bgset, backgrd, wslots;

#ifdef TLOG
extern int tralog;
#endif

extern long speed, filcnt, tfc, tlci, tlco, ffc, flci, flco, rptn, fsize;

extern CHAR *rdatap, padch, seol, ctlq, mypadc, eol;

extern char ttname[], *dftty, *cmarg, **cmlist;

#ifdef NOICP
char *cmerrp = "Kermit: ";
#else
extern char cmerrp[];
#endif

#ifndef NOCCTRAP
#include <setjmp.h>			/* Control-C trap */
jmp_buf cmjbuf;
#endif

/*  T R A P  --  Terminal interrupt handler */
 
SIGTYP
trap(sig,code) int sig, code; {
#ifdef VMS
    int i; FILE *f;
#endif
    debug(F101,"^C trap() caught signal","",sig);
    debug(F101," code","",code);
    zclose(ZIFILE);			/* If we were transferring a file, */
    zclose(ZOFILE);			/* close it. */
#ifdef VMS
/*
  Fix terminal.
*/  
    conres();
    i = printf("^C...\n");		/* Echo ^C to standard output */
    if (i < 1 && ferror(stdout)) {	/* If there was an error */
	fclose(stdout);			/* close standard output */
	f = fopen(dftty, "w");		/* open the controlling terminal */
	if (f) stdout = f;		/* and make it standard output */
	printf("^C...\n");		/* and echo the ^C again. */
    }
#else
    printf("^C...\n");			/* Not VMS, no problem... */
#endif
#ifndef NOCCTRAP
#ifdef UNIX
    ttimoff();				/* Turn off any timer interrupts */
#endif
    longjmp(cmjbuf,1);			/* Jump back to parser */
#else
    doexit(BAD_EXIT);			/* Exit poorly */
#endif
}
/*  S T P T R A P -- Handle SIGTSTP signals */

SIGTYP
stptrap(sig,code) int sig, code; {
    int x;
    debug(F101,"stptrap() caught signal","",sig);
    debug(F101," code","",code);
    conres();				/* Reset the console */
    x = psuspend();			/* Try to suspend. */
    if (x < 0)
      printf("Job control not supported\r\n"); 
    debug(F100,"stptrap back from suspend","",0);
#ifndef NOICP
    if (tlevel < 0)
      concb(escape);			/* Put console back in Kermit mode */
    if (pflag) {
	extern char cmdbuf[];
	prompt();			/* Reissue prompt when fg'd */
	printf("%s",cmdbuf);
    }
#endif
}

/*  C H K I N T  --  Check for console interrupts  */
 
chkint() {
    int ch, cn; long zz;
 
    if ((!local) || (quiet)) return(0);	/* Only do this if local & not quiet */
#ifdef datageneral
    cn = (con_reads_mt) ? 1 : conchk();	/* Any input waiting? */
#else
    cn = conchk();			/* Any input waiting? */
#endif
    debug(F101,"conchk","",cn);
 
    if (cn < 1) return(0);

#ifdef datageneral
    /* We must be careful to just print out one result for each character
     * read.  The flag, conint_avl, controls duplication of characters.
     * Only one character is handled at a time, which is a reasonable
     * limit.  More complicated schemes could handle a buffer.
     */
    if (con_reads_mt) {
	if ((ch = conint_ch) <= 0) return(0);   /* I/O error, or no data */
	else if (conint_avl == 0) return(0);    /* Char already read */
	else conint_avl = 0;                    /* Flag char as read */
    }
    else { if ((ch = coninc(5)) < 0) return(0);  }
#else
    if ((ch = coninc(5)) < 0) return(0);
#endif
    switch (ch & 0177) {
      case 'A': case 'a': case 0001:	/* Status report */
	screen(SCR_TN,0,0l,"^A  Status report:");
	screen(SCR_TN,0,0l," file type: ");
	if (binary) {
#ifdef VMS
	    if (binary == XYFT_I)
	      screen(SCR_TZ,0,0l,"image");
	    else if (binary == XYFT_L)
	      screen(SCR_TZ,0,0l,"labeled");
	    else screen(SCR_TZ,0,0l,"binary");
#else
	    screen(SCR_TZ,0,0l,"binary");
#endif /* VMS */
	} else {
	    screen(SCR_TZ,0,0l,"text");
	}
	screen(SCR_QE,0,(long)filcnt," file number");
	if (fsize) screen(SCR_QE,0,fsize," size");
	screen(SCR_QE,0,(long)ffc,   " characters so far");
	if (fsize) {
	    zz = ( ffc * 100L ) / fsize;
	    screen(SCR_QE,0,zz,      " percent done");
	}
	screen(SCR_QE,0,(long)bctu,  " block check");
	screen(SCR_QE,0,(long)rptflg," compression");
	screen(SCR_QE,0,(long)ebqflg," 8th-bit prefixing");
	if (!network)
	  screen(SCR_QE,0, speed, " speed");
	if (what == W_SEND)
	  screen(SCR_QE,0,(long)spsiz, " packet length");
	else if (what == W_RECV || what == W_REMO)
	  screen(SCR_QE,0,(long)urpsiz," packet length");
	screen(SCR_QE,0,(long)wslots,  " window slots");
	return(0);

      case 'B': case 'b': case 0002:	/* Cancel batch */
      case 'Z': case 'z': case 0032:
	screen(SCR_TN,0,0l,"Cancelling Batch ");
	czseen = 1;
	return(0);

      case 'F': case 'f': case 0006:	/* Cancel file */
      case 'X': case 'x': case 0030:
	screen(SCR_TN,0,0l,"Cancelling File ");
	cxseen = 1;
	return(0);

      case 'R': case 'r': case 0022:	/* Resend */
      case 0015: case 0012:
	screen(SCR_TN,0,0l,"Resending packet ");
	resend(winlo);
	return(1);

      case 'E': case 'e':		/* Send error packet */
      case 0005:
	return(-1);

      default:				/* Anything else, print message */
	intmsg(1);
	return(0);
    }
}

/*  I N T M S G  --  Issue message about terminal interrupts  */
 
intmsg(n) long n; {
    extern char *chstr();
    char buf[80];

    if ((!displa) || (quiet)) return;
    buf[0] = NUL;			/* Keep compilers happy */
#ifdef UXIII
    conchk();				/* Clear out pending escape-signals */
#endif
#ifdef VMS
    conres();				/* So Ctrl-C will work */
#endif
    if (n == 1) {

#ifdef UXIII				/* We need to signal before kb input */
#ifndef aegis
#ifndef datageneral
	sprintf(buf,"Type escape character (%s) followed by:",chstr(escape));
	screen(SCR_TN,0,0l,buf);
#endif
#endif
#endif

 screen(SCR_TN,0,0l,"X to cancel file,  CR to resend current packet");
 screen(SCR_TN,0,0l,"Z to cancel group, A for status report");
 screen(SCR_TN,0,0l,"E to send Error packet, Ctrl-C to quit immediately: ");
    }
    else screen(SCR_TU,0,0l," ");
}

/*  S C R E E N  --  Screen display function  */
 
/*  screen(f,c,n,s)
      f - argument descriptor
      c - a character or small integer
      n - a long integer
      s - a string.
 Fill in this routine with the appropriate display update for the system.
 This version is for a dumb tty.
*/
screen(f,c,n,s) int f; char c; long n; char *s; {
    static int p = 0;			/* Screen position */
    int len;				/* Length of string */
    char buf[80];			/* Output buffer */
    len = strlen(s);			/* Length of string */
    if ((f != SCR_WM) && (f != SCR_EM)) /* Always update warning & errors */
      if (!displa || quiet)
	return;
 
    switch (f) {
 
case SCR_FN:    			/* filename */
    conoll(""); conol(s); conoc(SP); p = len + 1; return;
 
case SCR_AN:    			/* as-name */
    if (p + len > 75) { conoll(""); p = 0; }
    conol("=> "); conol(s); if ((p += (len + 3)) > 78) { conoll(""); p = 0; }
    return;
 
case SCR_FS: 				/* file-size */
    sprintf(buf,", Size: %ld",n);  conoll(buf);  p = 0; return;
 
case SCR_XD:    			/* x-packet data */
    conoll(""); conoll(s); p = 0; return;
    
case SCR_ST:      			/* File status */
    switch (c) {
	case ST_OK:   	   		/*  Transferred OK */
	    if ((p += 5) > 78) { conoll(""); p = 0; }
	    conoll(" [OK]"); p += 5; return;
 
	case ST_DISC: 			/*  Discarded */
	    if ((p += 12) > 78) { conoll(""); p = 0; }
	    conoll(" [discarded]"); p += 12; return;
 
	case ST_INT:       		/*  Interrupted */
	    if ((p += 14) > 78) { conoll(""); p = 0; }
	    conoll(" [interrupted]"); p += 14; return;
 
	case ST_SKIP: 			/*  Skipped */
	    conoll("");
	    conol("Skipping "); conoll(s); p = 0;
	    return;
 
        case ST_ERR:
	    conoll("");
	    conol("Error "); conoll(s); p = 0;
	    return;

        default:
	    conoll("*** screen() called with bad status ***"); p = 0; return;
    }

case SCR_PN:    			/* Packet number */
    sprintf(buf,"%s: %ld",s,n); conol(buf); p += strlen(buf); return;
 
case SCR_PT:    			/* Packet type or pseudotype */
    if (c == 'Y') return;		/* Don't bother with ACKs */
    if (c == 'D') {			/* Only show every 4th data packet */
	if (n % 4) return;
	c = '.';
    }
#ifndef AMIGA
    if (p++ > 77) {			/* If near right margin, */
	conoll("");			/* Start new line */
	p = 0;				/* and reset counter. */
    }
#endif
    conoc(c);				/* Display the character. */
#ifdef AMIGA
    if (c == 'G') conoll("");           /* new line after G packets */
#endif
    return;
 
case SCR_TC:    			/* transaction complete */
    conoc(BEL); return;
 
case SCR_EM:				/* Error message */
    conoll(""); conoc('?'); conoll(s); p = 0; return;		/* +1	*/
 
case SCR_WM:				/* Warning message */
    conoll(""); conoll(s); p = 0; return;
 
case SCR_TU:				/* Undelimited text */
    if ((p += len) > 77) { conoll(""); p = len; }
    conol(s); return;
 
case SCR_TN:				/* Text delimited at beginning */
    conoll(""); conol(s); p = len; return;
 
case SCR_TZ:				/* Text delimited at end */
    if ((p += len) > 77) { conoll(""); p = len; }
    conoll(s); return;
    
case SCR_QE:				/* Quantity equals */
    sprintf(buf,"%s: %ld",s,n);
    conoll(buf); p = 0; return;
 
default:
    conoll("*** screen() called with bad object ***"); p = 0; return;
    }
}

/*  S D E B U  -- Record spar results in debugging log  */

sdebu(len) int len; {
    debug(F111,"spar: data",rdatap,len);
    debug(F101," spsiz ","", spsiz);
    debug(F101," timint","",timint);
    debug(F101," npad  ","",  npad);
    debug(F101," padch ","", padch);
    debug(F101," seol  ","",  seol);
    debug(F101," ctlq  ","",  ctlq);
    debug(F101," ebq   ","",   ebq);
    debug(F101," ebqflg","",ebqflg);
    debug(F101," bctr  ","",  bctr);
    debug(F101," rptq  ","",  rptq);
    debug(F101," rptflg","",rptflg);
    debug(F101," atcapu","",atcapu);
    debug(F101," lpcapu","",lpcapu);
    debug(F101," swcapu","",swcapu);
    debug(F101," wslotn","", wslotn);
}
/*  R D E B U -- Debugging display of rpar() values  */

rdebu(d,len) char *d; int len; {
    debug(F111,"rpar: data",d,len);
    debug(F101," rpsiz ","",xunchar(d[0]));
    debug(F101," rtimo ","", rtimo);
    debug(F101," mypadn","",mypadn);
    debug(F101," mypadc","",mypadc);
    debug(F101," eol   ","",   eol);
    debug(F101," ctlq  ","",  ctlq);
    debug(F101," sq    ","",    sq);
    debug(F101," ebq   ","",   ebq);
    debug(F101," ebqflg","",ebqflg);
    debug(F101," bctr  ","",  bctr);
    debug(F101," rptq  ","",  d[8]);
    debug(F101," rptflg","",rptflg);
    debug(F101," capas ","", capas);
    debug(F101," bits  ","",d[capas]);
    debug(F101," atcapu","",atcapu);
    debug(F101," lpcapu","",lpcapu);
    debug(F101," swcapu","",swcapu);
    debug(F101," wslotr","", wslotr);
    debug(F101," rpsiz(extended)","",rpsiz);
}

/*  E R M S G  --  Nonfatal error message  */

ermsg(msg) char *msg; {			/* Print error message */
#ifdef OSK
    if (!quiet) printf("\n%s - %s\n",cmerrp,msg);
#else
    if (!quiet) printf("\r\n%s - %s\n",cmerrp,msg);
#endif /* OSK */
    tlog(F110,"Error -",msg,0L);
}

/*  F A T A L  --  Fatal error message */

fatal(msg) char *msg; {
#ifdef OSK
    printf("\nFatal: %s\n",msg);
#else
    printf("\r\nFatal: %s\n",msg);
#endif /* OSK */
    tlog(F110,"Fatal:",msg,0L);
    doexit(BAD_EXIT);			/* Exit indicating failure */
}

/*  D O E X I T  --  Exit from the program.  */

doexit(exitstat) int exitstat; {

#ifndef NOICP
    extern struct mtab mactab[];
    extern int nmac;
#endif /* NOICP */

    ttclos();				/* Close external line, if any */    
    if (local) {
	strcpy(ttname,dftty);		/* Restore default tty */
	local = dfloc;			/* And default remote/local status */
    }
#ifdef COMMENT
    if (!quiet) conres();		/* Restore console terminal. */
    if (!quiet) connoi();		/* Turn off console interrupt traps. */
#else
    conres();				/* Restore console terminal. */
    connoi();				/* Turn off console interrupt traps. */
#endif
 
    if (deblog) {			/* Close any open logs. */
	debug(F100,"Debug Log Closed","",0);
	*debfil = '\0';
	deblog = 0;
	zclose(ZDFILE);
    }
    if (pktlog) {
	*pktfil = '\0';
	pktlog = 0;
	zclose(ZPFILE);
    }
    if (seslog) {
    	*sesfil = '\0';
	seslog = 0;
	zclose(ZSFILE);
    }
#ifdef TLOG
    if (tralog) {
	tlog(F100,"Transaction Log Closed","",0L);
	*trafil = '\0';
	tralog = 0;
	zclose(ZTFILE);
    }
#endif
    zclose(ZRFILE);
    zclose(ZWFILE);

    syscleanup();

#ifndef NOICP
/* If a macro named "on_exit" is defined, execute it. */
    if (nmac) {				/* Any macros defined? */
	int k;				/* Yes */
	k = mlook(mactab,"on_exit",nmac); /* Look up "on_exit" */
	if (k >= 0) {			/* If found, */
	    if (dodo(k,"") > -1)	/* set it up */
	    parser(1);			/* and execute it */
        }
    }
#endif /* NOICP */

#ifdef VMS
    if (exitstat == 0 || exitstat == 0x2C /* || other values ??? */)
        exitstat |= 0x10000000;		/* avoid displaying error message */
#else
    if (exitstat != GOOD_EXIT || xitsta != GOOD_EXIT) exitstat = BAD_EXIT;
#endif
    exit(exitstat);				/* Exit from the program. */
}

/*  B L D L E N  --  Make length-encoded copy of string  */
 
char *
bldlen(str,dest) char *str, *dest; {
    int len;
    len = strlen(str);
    *dest = tochar(len);
    strcpy(dest+1,str);
    return(dest+len+1);
}
 
 
/*  S E T G E N  --  Construct a generic command  */
 
extern char cmdstr[];

setgen(type,arg1,arg2,arg3) char type, *arg1, *arg2, *arg3; {
    char *upstr, *cp;
 
    cp = cmdstr;
    *cp++ = type;
    *cp = NUL;
    if (*arg1 != NUL) {
	upstr = bldlen(arg1,cp);
	if (*arg2 != NUL) {
	    upstr = bldlen(arg2,upstr);
	    if (*arg3 != NUL) bldlen(arg3,upstr);
	}
    }
    cmarg = cmdstr;
    debug(F110,"setgen",cmarg,0);
 
    return('g');
}

/* Set up interrupts */

setint() {
    conint(trap,stptrap);       /* Turn on console terminal interrupts. */
    bgchk();	    	        /* Check background status */
}

bgchk() {				/* Check background status */
    if (bgset < 0) pflag = !backgrd;
    else pflag = (bgset == 0 ? 1 : 0);
}

#ifdef DEBUG
/*  D E B U G  --  Enter a record in the debugging log  */
 
/*
 Call with a format, two strings, and a number:
   f  - Format, a bit string in range 0-7.
        If bit x is on, then argument number x is printed.
   s1 - String, argument number 1.  If selected, printed as is.
   s2 - String, argument number 2.  If selected, printed in brackets.
   n  - Int, argument 3.  If selected, printed preceded by equals sign.
 
   f=0 is special: print s1,s2, and interpret n as a char.
*/
#define DBUFL 2300
debug(f,s1,s2,n) int f, n; char *s1, *s2; {
    static char s[DBUFL];
    char *sp = s;
 
    if (!deblog) return;		/* If no debug log, don't */
    switch (f) {
    	case F000:			/* 0, print both strings, */
	    if (strlen(s1) + strlen(s2) + 5 > DBUFL) { /* and n as a char */
		sprintf(sp,"DEBUG string too long\n");
	    } else {
		if (n > 31 && n < 127)
		  sprintf(sp,"%s%s:%c\n",s1,s2,n);
		else if (n < 32 || n == 127)
		  sprintf(sp,"%s%s:^%c\n",s1,s2,(n+64) & 0x7F);
		else if (n > 127 && n < 160)
		  sprintf(sp,"%s%s:~^%c\n",s1,s2,(n-64) & 0x7F);
		else if (n > 159 && n < 256)
		  sprintf(sp,"%s%s:~%c\n",s1,s2,n & 0x7F);
		else sprintf(sp,"%s%s:%d\n",s1,s2,n);
	    }
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
	    break;
    	case F001:			/* 1, "=n" */
	    sprintf(sp,"=%d\n",n);
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
	    break;
    	case F010:			/* 2, "[s2]" */
	    if (strlen(s2) + 4 > DBUFL)
	      sprintf(sp,"DEBUG string too long\n");
	    else sprintf(sp,"[%s]\n",s2);
	    if (zsout(ZDFILE,"") < 0) deblog = 0;
	    break;
    	case F011:			/* 3, "[s2]=n" */
	    if (strlen(s2) + 15 > DBUFL)
	      sprintf(sp,"DEBUG string too long\n");
	    else sprintf(sp,"[%s]=%d\n",s2,n);
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
	    break;
    	case F100:			/* 4, "s1" */
	    if (zsoutl(ZDFILE,s1) < 0) deblog = 0;
	    break;
    	case F101:			/* 5, "s1=n" */
	    if (strlen(s1) + 15 > DBUFL)
	      sprintf(sp,"DEBUG string too long\n");
	    else sprintf(sp,"%s=%d\n",s1,n);
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
	    break;
    	case F110:			/* 6, "s1[s2]" */
	    if (strlen(s1) + strlen(s2) + 4 > DBUFL)
	      sprintf(sp,"DEBUG string too long\n");
	    else sprintf(sp,"%s[%s]\n",s1,s2);
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
	    break;
    	case F111:			/* 7, "s1[s2]=n" */
	    if (strlen(s1) + strlen(s2) + 15 > DBUFL)
	      sprintf(sp,"DEBUG string too long\n");
	    else sprintf(sp,"%s[%s]=%d\n",s1,s2,n);
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
	    break;
	default:
	    sprintf(sp,"\n?Invalid format for debug() - %d\n",n);
	    if (zsout(ZDFILE,s) < 0) deblog = 0;
    }
}
#endif

#ifdef TLOG
#define TBUFL 300
/*  T L O G  --  Log a record in the transaction file  */
/*
 Call with a format and 3 arguments: two strings and a number:
   f  - Format, a bit string in range 0-7, bit x is on, arg #x is printed.
   s1,s2 - String arguments 1 and 2.
   n  - Int, argument 3.
*/
tlog(f,s1,s2,n) int f; long n; char *s1, *s2; {
    static char s[TBUFL];
    char *sp = s; int x;
    
    if (!tralog) return;		/* If no transaction log, don't */
    switch (f) {
    	case F000:			/* 0 (special) "s1 n s2"  */
	    if (strlen(s1) + strlen(s2) + 15 > TBUFL)
	      sprintf(sp,"?T-Log string too long\n");
	    else sprintf(sp,"%s %ld %s\n",s1,n,s2);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
	    break;
    	case F001:			/* 1, " n" */
	    sprintf(sp," %ld\n",n);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
	    break;
    	case F010:			/* 2, "[s2]" */
	    x = strlen(s2);
	    if (s2[x] == '\n') s2[x] = '\0';
	    if (x + 6 > TBUFL)
	      sprintf(sp,"?T-Log string too long\n");
	    else sprintf(sp,"[%s]\n",s2);
	    if (zsout(ZTFILE,"") < 0) tralog = 0;
	    break;
    	case F011:			/* 3, "[s2] n" */
	    x = strlen(s2);
	    if (s2[x] == '\n') s2[x] = '\0';
	    if (x + 6 > TBUFL)
	      sprintf(sp,"?T-Log string too long\n");
	    else sprintf(sp,"[%s] %ld\n",s2,n);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
	    break;
    	case F100:			/* 4, "s1" */
	    if (zsoutl(ZTFILE,s1) < 0) tralog = 0;
	    break;
    	case F101:			/* 5, "s1: n" */
	    if (strlen(s1) + 15 > TBUFL)
	      sprintf(sp,"?T-Log string too long\n");
	    else sprintf(sp,"%s: %ld\n",s1,n);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
	    break;
    	case F110:			/* 6, "s1 s2" */
	    x = strlen(s2);
	    if (s2[x] == '\n') s2[x] = '\0';
	    if (strlen(s1) + x + 4 > TBUFL)
	      sprintf(sp,"?T-Log string too long\n");
	    else sprintf(sp,"%s %s\n",s1,s2);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
	    break;
    	case F111:			/* 7, "s1 s2: n" */
	    x = strlen(s2);
	    if (s2[x] == '\n') s2[x] = '\0';
	    if (strlen(s1) + x + 15 > TBUFL)
	      sprintf(sp,"?T-Log string too long\n");
	    else sprintf(sp,"%s %s: %ld\n",s1,s2,n);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
	    break;
	default:
	    sprintf(sp,"\n?Invalid format for tlog() - %ld\n",n);
	    if (zsout(ZTFILE,s) < 0) tralog = 0;
    }
}
#endif /* TLOG */


#ifndef NOMGET
static char *mgbufp = NULL;
extern char *malloc();

/*  F N P A R S E  --  */

/*
  Argument is a character string containing one or more filespecs.
  This function breaks the string apart into an array of pointers, one
  to each filespec, and returns the number of filespecs.  Used by server
  when it receives a GET command to allow it to process multiple file
  specifications in one transaction.  Sets cmlist to point to a list of
  file pointers, exactly as if they were command line arguments.

  This version of fnparse treats spaces as filename separators.  If your
  operating system allows spaces in filenames, you'll need a different 
  separator.  

  This version of fnparse mallocs a string buffer to contain the names.  It
  cannot assume that the string that is pointed to by the argument is safe.
*/
#ifdef MAC				/* Filename separator */
#define FNSEP ','
#else
#define FNSEP SP
#endif
fnparse(string) char *string; {
    char *p, *s, *q;
    int r = 0;				/* Return code */

    if (mgbufp) free(mgbufp);		/* Free this from last time. */
    mgbufp = malloc(strlen(string)+2);
    if (!mgbufp) {
	debug(F100,"fnparse malloc error","",0);
	return(0);
    }	
    strcpy(mgbufp,string);		/* Make a safe copy */
    p = s = mgbufp;			/* Point to the copy */
    r = 0;				/* Initialize our return code */
    while (*p == SP) p++,s++;		/* Skip leading spaces */
    while (1) {				/* Loop through rest of string */
	if (*s == FNSEP || *s == NUL) {	/* Look for separator or terminator */
	    msfiles[r] = p;		/* Add this filename to the list */
	    debug(F111,"fnparse",msfiles[r],r);
	    r++;			/* Count it */
	    if (*s == NUL) break;	/* End of string? */
	    *s++ = NUL;			/* No, turn space to NUL */
	    while (*s == SP) s++;	/* Skip repeated spaces */
	    p = s;			/* Start of next name */
	    continue;
	}
	s++;				/* Otherwise keep scanning */
    }
    debug(F101,"fnparse r","",r);
    cmlist = msfiles;
    return(r);
}
#endif /* NOMGET */
