
//***************************************************************************
//
// Class:					CFileSystem
//
// Purpose:				To encapsulate access to the file system.
//
// Copyright:			Copyright 1993, 1994 Scott P. Leslie, All Rights Reserved.
//
// Version:				Version 1.1 for MS Windows 3.1 and MSVC 1.0.
//
// Important:			Read the README.TXT file included with this distribution
//								before using this class.
//
// Shareware:			This class is ShareWare.  If you use it, you should register
//								it with the author.  If you do not register it after 14 days
//								of use, you must discontinue using it.
//
//								Contact the author at ScotLeslie@aol.com for more info.
//
// Distribution:	You may distribute this class unmodified as long as all
//								accompanying documentation and notes are also distributed.
//								Object code generated from this class (or any derivation
//								of this class) can only be distributed by registered users.
//
// Disclaimer:		This class is provided as is with no implied or express
//								warranties.  You should test this class for your particular
//								use on non-critical data before using it in a production
//								system.
//
//***************************************************************************

//***************************************************************************
//	Include Files
//***************************************************************************

#include "stdafx.h"

#include "filesys.h"


//***************************************************************************
//
// Name:		CFileSystem
//
// Purpose:	Constructor.
//
// Notes:		None.
//
//***************************************************************************
CFileSystem::CFileSystem()
{
	MaxFileNameLength = 256;
} // CFileSystem


//***************************************************************************
//
// Name:		~CFileSystem
//
// Purpose:	Destructor.
//
// Notes:		None.
//
//***************************************************************************
CFileSystem::~CFileSystem()
{
	// No code.
} // ~CFileSystem


//***************************************************************************
//
// Name:		GetDirectoryEntry
//
// Purpose:	To return the next directory entry based on a wildcard, etc...
//
// Example:	CString *pFileName = fs.GetDirectoryEntry("*.txt", hidden);
//
// Notes:		None.
//
//***************************************************************************
CString *
CFileSystem::GetDirectoryEntry(const CString& Wildcard /* = "" */, const Attribute eFileAttrib /* = normal */)
{
	int nRetVal;

	// If they passed in a Wildcard, we want to lookup the first entry.
	if (Wildcard != "")
	{
		nRetVal = _dos_findfirst((const char *) Wildcard, eFileAttrib, &m_FileInfo);
	} // if
	else
	{
		nRetVal = _dos_findnext(&m_FileInfo);
	} // else

	if (nRetVal != 0)
	{
		// Error occurred, return NULL.
		return NULL;
	} // if

	// Create a string and copy the name to it.
	CString *pString = new CString();
	memcpy(pString->GetBufferSetLength(MaxFileNameLength), m_FileInfo.name, 13);
	pString->ReleaseBuffer(-1);

	return pString;
} // GetDirectoryEntry


//***************************************************************************
//
// Name:		GetCurrentDirectory
//
// Purpose:	To get the current working directory.
//
// Example:	CString CurrentDir = fs.GetCurrentDirecory();
//
// Notes:		None
//
//***************************************************************************
CString
CFileSystem::GetCurrentDirectory()
{
	CString String;
	char *pRetVal = _getcwd(String.GetBufferSetLength(MaxFileNameLength), MaxFileNameLength);
	String.ReleaseBuffer(-1);

	// If an error occured, clean up and return NULL.
	if (pRetVal == 0)
	{
		String = "";
	} // if

	return String;
} // GetCurrentDirectory


//***************************************************************************
//
// Name:		ChangeDirectory
//
// Purpose:	To change the current working directory.
//
// Example:	BOOL bRetVal = fs.ChangeDirectory("c:\\foo\\bar");
//
// Notes:		None.
//
//***************************************************************************
BOOL
CFileSystem::ChangeDirectory(const CString& NewDirectory)
{
	int nRetVal = _chdir((const char *) NewDirectory);
	if (nRetVal == -1)
	{
		return FALSE;
	} // if

	return TRUE;
} // ChangeDirectory


//***************************************************************************
//
// Name:		MakeDirectory
//
// Purpose:	To make a directory.
//
// Example:	BOOL bRetVal = fs.MakeDirectory("c:\\foo\\bar");
//
// Notes:		None
//
//***************************************************************************
BOOL
CFileSystem::MakeDirectory(const CString& NewDirectory)
{
#ifdef _WINNT_
	ASSERT(FALSE);	// Untested
	return CreateDirectory(NewDirectory);

#else // _WINNT_

	int nRetVal = _mkdir((const char *) NewDirectory);
	if (nRetVal == -1)
	{
		return FALSE;
	} // if

	return TRUE;

#endif // _WINNT_
} // MakeDirectory


//***************************************************************************
//
// Name:		MakePath
//
// Purpose:	To make all the directories in a given path.  If any of the
//					directories exist, the creation continues with lower level
//					directories.
//
// Ret Val:	TRUE : No error.
//					FALSE : Unable to create path, or path already exists.
//
// Example:	BOOL bRetVal = fs.MakePath("c:\\foo\\bar\\what");
//
// Notes:		None
//
//***************************************************************************
BOOL
CFileSystem::MakePath(const CString& NewDirectory)
{
	CString NewDir = NewDirectory;	// Copy string for manipulation
	CString	DirName;
	BOOL		bRetVal = TRUE;

	// Make sure the directory name ends in a slash
	if (NewDir[NewDir.GetLength() - 1] != '\\')
	{
		NewDir = NewDir + '\\';
	} // if

	// Create each directory in the path
	UINT	nIndex = 0;
	BOOL	bDone = FALSE;
	while (!bDone)
	{
		// Extract one directory
		nIndex = NewDir.Find('\\');
		if (nIndex != -1)
		{
			DirName = DirName + NewDir.Left(nIndex);
			NewDir = NewDir.Right(NewDir.GetLength() - nIndex - 1);

			// The first time through, we might have a drive name
			if (DirName.GetLength() >= 1  &&  DirName[DirName.GetLength() - 1] != ':')
			{
				bRetVal = MakeDirectory(DirName);
			} // if
			DirName = DirName + '\\';
		} // if
		else
		{
			// We're finished
			bDone = TRUE;
		} // else
	} // while

	// Return the last MakeDirectory() return value.
	return bRetVal;
} // MakePath


//***************************************************************************
//
// Name:		DeleteDirectory
//
// Purpose:	To delete a directory.  Optionally, all lower level files
//					and directories can be deleted.
//
// Ret Val:	TRUE : Successfully deleted directory.
//					FALSE : Failed to delete directory.
//
// Example:	BOOL bRetVal = fs.DeleteDirectory("c:\\foo\\bar", TRUE);
//
// Notes:		If bDeleteFilesAndDirs == FALSE, directory must be empty or
//					an error will occur and FALSE will be returned.
//
//***************************************************************************
BOOL
CFileSystem::DeleteDirectory(const CString& Directory, const BOOL bDeleteFilesAndDirs /* = FALSE */)
{
	if (bDeleteFilesAndDirs)
	{
		CStringList *pSubDirs = GetSubdirList(Directory);
		for (POSITION pos = pSubDirs->GetHeadPosition(); pos != NULL; )
		{
			CString DirName = pSubDirs->GetNext(pos);
			DeleteDirectory(DirName, bDeleteFilesAndDirs);
		} // for

		delete pSubDirs;
		pSubDirs = NULL;

		DeleteFiles(AppendWildcard(Directory, "*.*"));
	} // if

	int nRetVal = _rmdir((const char *) Directory);
	if (nRetVal == -1)
	{
		return FALSE;
	} // if

	return TRUE;
} // DeleteDirectory


//***************************************************************************
//
// Name:		GetDirectorySize
//
// Purpose:	To return the size (in bytes) of the files in the specified
//					directory.
//
// Example:	LONG lSize = fs.GetDirectorySize("c:\\foo\\bar", "*.*", TRUE);
//
// Notes:		If an error occurs in reading the status of a file, that
//					file is skipped and not counted as part of the size.
//					There is currently not way to find out if an error of this
//					kind occured.
//
//***************************************************************************
LONG
CFileSystem::GetDirectorySize(const CString& Directory /* = "" */, const CString& WildCard /* = "*.*" */, const BOOL bRecurseSubdirs /* = FALSE */)
{
	LONG lSize = 0;

	// Do all the subdirectories first...
	if (bRecurseSubdirs)
	{
		CStringList *pSubDirs = GetSubdirList(Directory);
		for (POSITION pos = pSubDirs->GetHeadPosition(); pos != NULL; )
		{
			CString DirName = pSubDirs->GetNext(pos);
			lSize += GetDirectorySize(DirName, WildCard, bRecurseSubdirs);
		} // for

		delete pSubDirs;
		pSubDirs = NULL;
	} // if

	// Find the sizes of all the files in the specified directory.
	CStringList *pFileList = GetFileList(AppendWildcard(Directory, WildCard), allfiles);
	for (POSITION pos = pFileList->GetHeadPosition(); pos != NULL; )
	{
		CFileStatus status;
		CString FileName = pFileList->GetNext(pos);
		if (CFile::GetStatus(FileName, status))
		{
			lSize += status.m_size;
		} // if
	} // for

	delete pFileList;
	pFileList = NULL;
	
	return lSize;
} // GetDirectorySize


//***************************************************************************
//
// Name:		GetCurrentFileSystem
//
// Purpose:	To return a string containing the current file system name.
//
// Example:	CString FileSystem = fs.GetCurrentFileSystem();
//
// Notes:		None
//
//***************************************************************************
CString
CFileSystem::GetCurrentFileSystem()
{
	unsigned int	nDrive = 0;
	char					cDrive = 'A';

	_dos_getdrive(&nDrive);

	cDrive = (char) ('A' + nDrive - 1);
	CString String = cDrive;
	String += ":\\";

	return String;
} // GetCurrentFileSystem


//***************************************************************************
//
// Name:		ChangeFileSystem
//
// Purpose:	To change the current file system.
//
// Example:	BOOL bRetVal = fs.ChangeFileSystem('c');
//
// Notes:		Obsolete.  The version that takes a CString parameter
//					should be used instead.  This change was made in preparation
//					for the addition of Windows NT support.
//
//***************************************************************************
BOOL
CFileSystem::ChangeFileSystem(const char cFileSystem)
{
	unsigned int nNumDrives;		// ignored return value
	unsigned int nNewDrive;

	nNewDrive = toupper(cFileSystem) - 'A' + 1;

	if (nNewDrive >= 0  &&  nNewDrive <= 26)
	{
		_dos_setdrive(nNewDrive, &nNumDrives);
		return TRUE;
	} // if

	return FALSE;
} // ChangeFileSystem


//***************************************************************************
//
// Name:		ChangeFileSystem
//
// Purpose:	To change the current file system.
//
// Example:	BOOL bRetVal = fs.ChangeFileSystem("c:\\");
//
// Notes:		Expects a filesystem name of the form "X:\".
//
//***************************************************************************
BOOL
CFileSystem::ChangeFileSystem(const CString& FileSystem)
{
	BOOL bRetVal = FALSE;

	if (FileSystem.GetLength() > 0)
	{
		char cFileSystem = FileSystem[0];
		bRetVal = ChangeFileSystem(cFileSystem);
	} // if

	return bRetVal;
} // ChangeFileSystem


//***************************************************************************
//
// Name:		GetFileSystemList
//
// Purpose:	To return a list of available file systems (ie. drives).
//
// Example:	CStringList *pFSList = fs.GetFileSystemList();
//
// Notes:		None
//
//***************************************************************************
CStringList	*
CFileSystem::GetFileSystemList()
{
	CStringList *pStringList = new CStringList();

	for (int nDriveNum = 0; nDriveNum < 26; nDriveNum++)
	{
		if (GetDriveType(nDriveNum) != 0)
		{
			CString DriveName = (char)('A' + nDriveNum);
			DriveName = DriveName + ":\\";
			pStringList->AddTail((const char *)DriveName);
		} // if
	} // for

	return pStringList;
} // GetFileSystemList


//***************************************************************************
//
// Name:		GetVolumeLabel
//
// Purpose:	To get the volume label for a filesystem.
//
// Example:	CString VolumeLabel = fs.GetVolumneLabel("c:\\");
//
// Notes:		None
//
//***************************************************************************
CString
CFileSystem::GetVolumeLabel(const CString& FileSystem)
{
	CString VolumeLabel = "";
	CString *pVolLabel = GetDirectoryEntry(AppendWildcard(FileSystem, "*.*"), volume);
	if (pVolLabel != NULL)
	{
		VolumeLabel = *pVolLabel;

		delete pVolLabel;
		pVolLabel = NULL;
	} // if

	return VolumeLabel;
} // GetVolumeLabel


//***************************************************************************
//
// Name:		GetFileSystemType
//
// Purpose:	To get the type (removable, network, etc...) of the specified
//					filesystem.
//
// Ret Val:	0 = Error.
//					DRIVE_REMOVABLE : Removable Disk (ie. floppy)
//					DRIVE_FIXED : Non-Removable Disk (ie. hard drive)
//					DRIVE_REMOTE : Network Disk (ie. NFS mounted, Netware volume)
//
// Example:	LONG lType = fs.GetFileSystemType("c:\\");
//					ASSERT(lType == DRIVE_FIXED);
//
// Notes:		None
//
//***************************************************************************
LONG
CFileSystem::GetFileSystemType(const CString& FileSystem)
{
	LONG lType = DRIVE_UNDETERMINED;

	if (FileSystem.GetLength() > 0)
	{
		char cFileSystem = (char) toupper(FileSystem[0]);

		int nDriveNum = toupper(cFileSystem) - 'A';
	
		if (nDriveNum >= 0  &&  nDriveNum <= 26)
		{
			lType = GetDriveType(nDriveNum);
		} // if
	} // if

	return lType;
} // GetFileSystemType


//***************************************************************************
//
// Name:		RenameFile
//
// Purpose:	To rename a file.
//
// Example:	BOOL bRetVal = fs.RenameFile("c:\\foo.txt", "c:\\bar.doc");
//
// Notes:		None
//
//***************************************************************************
BOOL
CFileSystem::RenameFile(const CString& OldFileName, const CString& NewFileName)
{
	TRY
	{
		CFile::Rename((const char *) OldFileName, (const char *) NewFileName);
	} // TRY
	CATCH(CFileException, Exception)
	{
		return FALSE;
	} // CATCH
	END_CATCH

	return TRUE;
} // RenameFile


//***************************************************************************
//
// Name:		DeleteFiles
//
// Purpose:	To delete a set of files based on a wildcard file specification.
//
// Example:	BOOL bRetVal = fs.DeleteFiles("c:\\foo\\bar\\*.txt");
//
// Notes:		Only normal files (not hidden or system) files can be deleted
//					with this function.
//
//***************************************************************************
BOOL
CFileSystem::DeleteFiles(const CString& FileSpec)
{
	BOOL bRetVal = TRUE;
	CStringList *pDir = GetDirectory(FileSpec, normal);

	for (POSITION pos = pDir->GetHeadPosition(); pos != NULL; )
	{
		CString FileName = pDir->GetNext(pos);
		if (DeleteFile(FileName) == FALSE)
		{
			bRetVal = FALSE;
		} // if
	} // for

	// Clean up.
	delete pDir;
	pDir = NULL;

	return bRetVal;
} // DeleteFiles


//***************************************************************************
//
// Name:		DeleteFile
//
// Purpose:	To delete a file.
//
// Example:
//					BOOL bRetVal = fs.DeleteFile("c:\foo.txt");
//
// Notes:		None
//
//***************************************************************************
BOOL
CFileSystem::DeleteFile(const CString& FileName)
{
	TRY
	{
		CFile::Remove((const char *) FileName);
	} // TRY
	CATCH(CFileException, Exception)
	{
		return FALSE;
	} // CATCH
	END_CATCH

	return TRUE;
} // DeleteFile


//***************************************************************************
//
// Name:		CloseFile
//
// Purpose:	To close a file.
//
// Notes:		None
//
//***************************************************************************
BOOL
CFileSystem::CloseFile(CFile *pFile) const
{
	BOOL bRetVal = TRUE;

	TRY
	{
		pFile->Close();
	} // TRY
	CATCH(CFileException, e)
	{
		bRetVal = FALSE;
	} // CATCH
	END_CATCH

	return bRetVal;
} // CloseFile


//***************************************************************************
//
// Name:		CopyFile
//
// Purpose:	To copy a file.
//
// Ret Val:	TRUE : File was copied successfully.
//					FALSE : Error occured.
//
// Example:	BOOL bRetVal = fs.CopyFile("foo.txt", "a:\\bar.txt", 20480);
//
// Notes:		If the destination file exists, an error occurs.
//
//***************************************************************************
BOOL
CFileSystem::CopyFile(const CString& SourceFileName, const CString& DestFileName, const unsigned long lBuffSize /* = 10240 */)
{
	BOOL		bRetVal	= TRUE;
	char *	pBuff		= new char[lBuffSize];
	CFile Source;
	CFile Dest;

	// Check for exitance of the destination file.
	CFileStatus destStatus;
	if ((CFile::GetStatus(DestFileName, destStatus)))
	{
		delete [] pBuff;
		pBuff = NULL;

		return FALSE;
	} // if

	// Open the files, creating the destination.
	if (Source.Open((const char *) SourceFileName, CFile::modeRead))
	{
		if (Dest.Open((const char *) DestFileName, CFile::modeCreate | CFile::modeWrite))
		{
			DWORD	dwLength = Source.GetLength();
		
			// Copy the data in the file.
			while (dwLength > 0)
			{
				UINT nRead = Source.Read(pBuff, (UINT) lBuffSize);
				if (nRead)
				{
					Dest.Write(pBuff, nRead);
				} // if
			
				dwLength -= nRead;
			} // while
		
			CloseFile(&Source);
			CloseFile(&Dest);
		} // if
		else
		{
			CloseFile(&Source);

			bRetVal = FALSE;
		} // else
	} // if
	else
	{
		bRetVal = FALSE;
	} // else

	delete [] pBuff;
	pBuff = NULL;

	return bRetVal;
} // CopyFile


//***************************************************************************
//
// Name:		CopyFiles
//
// Purpose:	To copy a set of files based on a wildcard file specification.
//
// Example:	BOOL bRetVal = fs.CopyFiles("c:\\foo\\bar\\*.txt", "c:\\foo2");
//
// Notes:		Only normal files (not hidden or system) files can be copied
//					with this function.
//
//***************************************************************************
BOOL
CFileSystem::CopyFiles(const CString& FileSpec, const CString& DestPath)
{
	BOOL bRetVal = TRUE;
	CStringList *pDir = GetDirectory(FileSpec, normal);

	for (POSITION pos = pDir->GetHeadPosition(); pos != NULL; )
	{
		CString FileName = pDir->GetNext(pos);
		CString DestFileName = AppendWildcard(DestPath, GetFileName(FileName));

		if (CopyFile(FileName, DestFileName) == FALSE)
		{
			bRetVal = FALSE;
		} // if
	} // for

	// Clean up.
	delete pDir;
	pDir = NULL;

	return bRetVal;
} // CopyFiles


//***************************************************************************
//
// Name:		CompareFiles
//
// Purpose:	To compare the contents of two files to see if they are the
//					same.
//
// Ret Val:	TRUE, if the files are the same.
//					FALSE, if the files are different, or an error occurs.
//
// Example:	BOOL bRetVal = fs.CompareFiles("foo.txt", "bar.txt", 20480);
//
// Notes:		None
//
//***************************************************************************
BOOL
CFileSystem::CompareFiles(const CString& FileName1, const CString& FileName2, const unsigned long lBuffSize /* = 10240 */)
{
	BOOL		bRetVal	= TRUE;
	char *	pBuff1	= new char[lBuffSize];
	char *	pBuff2	= new char[lBuffSize];
	CFile		File1;
	CFile		File2;

	// Make sure we allocated the buffers
	if (!pBuff1  ||  !pBuff2)
	{
		if (pBuff1)
		{
			delete [] pBuff1;
		} // if

		if (pBuff2)
		{
			delete [] pBuff2;
		} // if

		return FALSE;
	} // if

	// Open the files.
	if (File1.Open((const char *) FileName1, CFile::modeRead))
	{
		if (File2.Open((const char *) FileName2, CFile::modeRead))
		{
			DWORD	dwLength1 = File1.GetLength();
			DWORD	dwLength2 = File2.GetLength();
			if (dwLength1 != dwLength2)
			{
				bRetVal = FALSE;
			} // if

			// Read the data in the file.
			while (bRetVal == TRUE  &&  dwLength1 > 0)
			{
				UINT nRead1 = File1.Read(pBuff1, (UINT) lBuffSize);
				UINT nRead2 = File2.Read(pBuff2, (UINT) lBuffSize);

				if (nRead1 != nRead2)
				{
					bRetVal = FALSE;
					break;							// break out of the while loop
				} // if

				if (memcmp(pBuff1, pBuff2, nRead1) != 0)
				{
					bRetVal = FALSE;
				} // if
			
				dwLength1 -= nRead2;
			} // while

			CloseFile(&File1);
			CloseFile(&File2);
		} // if
		else
		{
			CloseFile(&File1);

			bRetVal = FALSE;
		} // else
	} // if
	else
	{
		bRetVal = FALSE;
	} // else

	delete [] pBuff1;
	delete [] pBuff2;

	return bRetVal;
} // CompareFiles


//***************************************************************************
//
// Name:		GetFileName
//
// Purpose:	Extract a file name from a path.
//
// Example:	CString FileName = fs.GetFileName("c:\\foo\\bar\\what.txt");
//					ASSERT(FileName == "what.txt");
//
// Notes:		None
//
//***************************************************************************
CString
CFileSystem::GetFileName(const CString& PathAndFileName)
{
	CString FileName = PathAndFileName;	// Copy to make modifications.

	// Find the last "\" in the string and return everything after it.
	int nIndex = FileName.Find('\\');
	while(nIndex != -1)
	{
		FileName = FileName.Right(FileName.GetLength() - nIndex - 1);

		nIndex = FileName.Find('\\');
	} // while

	return FileName;
} // GetFileName


//***************************************************************************
//
// Name:		GetPath
//
// Purpose:	To return the directory from a path.
//
// Example:	CString Path = fs.GetPath("c:\\foo\\bar\\what.txt");
//					ASSERT(Path == "c:\\foo\\bar");
//
// Notes:		None
//
//***************************************************************************
CString
CFileSystem::GetPath(const CString& PathAndFileName)
{
	CString FileName = PathAndFileName; // Copy to modify;
	CString	Path = "";

	// Find the last "\" in the string and return everything up to and including it.
	int nIndex = FileName.Find('\\');
	while(nIndex != -1)
	{
		Path = Path + FileName.Left(nIndex + 1);
		FileName = FileName.Right(FileName.GetLength() - nIndex - 1);

		nIndex = FileName.Find('\\');
	} // while

	return Path;
} // GetPath


//***************************************************************************
//
// Name:		GetDirectory
//
// Purpose:	To return a list of files based on a search string, etc...
//
// Example:	CStringList *pDir = fs.GetDirectory("*.txt", normal, TRUE);
//					(void) fs.GetDirectory("*.doc", normal, TRUE, pDir);
//
// Notes:		The filenames include the path.
//
//***************************************************************************
CStringList	*
CFileSystem::GetDirectory(const CString& SearchString, const Attribute eFileAttrib, const BOOL bRecurseSubDirs /* = FALSE */, CStringList *pStringList /* = NULL */)
{
	// If they don't pass in a list, create one.
	if (pStringList == NULL)
	{
		pStringList = new CStringList();
	} // if

	// Read the file list.
	CStringList *pFileList = GetFileList(SearchString, eFileAttrib);
	pStringList->AddTail(pFileList);
	delete pFileList;
	pFileList = NULL;

	if (bRecurseSubDirs)
	{
		CString				CurDir = GetPath(SearchString);
		CStringList *	pDirList = GetSubdirList(CurDir);

		// Go through the directories we just got and recurse through them too.
		for (POSITION pos=pDirList->GetHeadPosition(); pos != 0; )
		{
			CString String = pDirList->GetNext(pos);

			// Get file name part of search path
			CString	SearchSpec = GetFileName(SearchString);

			// Do the recursion.
			GetDirectory(String + "\\" + SearchSpec, eFileAttrib, bRecurseSubDirs, pStringList);
		} // for

		delete pDirList;
		pDirList = NULL;
	} // if

	return pStringList;
} // GetDirectory


//***************************************************************************
//
// Name:		GetSubdirList
//
// Purpose:	To return a list of subdirectories of another directory.
//
// Example:
//					CStringList *pDirs = fs.GetSubdirList("c:\\foo", FALSE);
//
// Notes:		None
//
//***************************************************************************
CStringList *
CFileSystem::GetSubdirList(const CString& SearchDir, const BOOL bPathInName /* = TRUE */)
{
	// Read the directory list
	CStringList *	pDirList = new CStringList();

	CString SearchPath = AppendWildcard(SearchDir, "*.*");
	CString *	pString = GetDirectoryEntry(SearchPath, directory);
	while (pString != NULL)
	{
		CString	String;
		CString	FullPath;
		CString	CurDir = GetPath(SearchPath);
		sprintf(FullPath.GetBufferSetLength(1024), "%s%s", (const char *)CurDir, (const char *)(*pString));
		FullPath.ReleaseBuffer(-1);
		if (bPathInName)
		{
			sprintf(String.GetBufferSetLength(1024), "%s%s", (const char *)CurDir, (const char *)(*pString));
			String.ReleaseBuffer(-1);
		} // if
		else
		{
			sprintf(String.GetBufferSetLength(1024), "%s", (const char *)(*pString));
			String.ReleaseBuffer(-1);
		} // else

		// If it's not one of the special directories.
		if (*pString != "."  &&  *pString != "..")
		{
			// Get the file type and make sure it's a directory before we add it to the list.
			CFileStatus FileStatus;
			CFile::GetStatus((const char *) FullPath, FileStatus);
			if (FileStatus.m_attribute == directory)
			{
				pDirList->AddTail((const char *) String);
			} // if
		} // if

		// Delete the string we got back from GetDirectoryEntry
		delete pString;
		pString = NULL;
	
		pString = GetDirectoryEntry();
	} // while

	return pDirList;
} // GetSubdirList


//***************************************************************************
//
// Name:		GetFileList
//
// Purpose:	Return a list of files given a search path and attribute.
//
// Notes:		This only searches the specified directory.
//					Use GetDirectory() to recurse subdirectories.
//					The filenames include the path.
//
// Example:
//					CStringList *pList = GetFileList("c:\\foo\\bar\\*.cpp", normal);
//
//***************************************************************************
CStringList *
CFileSystem::GetFileList(const CString& SearchString, const Attribute eFileAttrib)
{
	CStringList *pDirList = new CStringList();

	// Read the file list
	CString		CurDir = GetPath(SearchString);
	CString *	pString = GetDirectoryEntry(SearchString, eFileAttrib);
	CString	String;
	while (pString != NULL)
	{
		sprintf(String.GetBufferSetLength(1024), "%s%s", (const char *)CurDir, (const char *)(*pString));
		String.ReleaseBuffer(-1);
		pDirList->AddTail((const char *) String);

		// Delete the string we got back from GetDirectoryEntry
		delete pString;
		pString = NULL;

		pString = GetDirectoryEntry();
	} // while

	return pDirList;
} // GetFileList


//***************************************************************************
//
// Name:		Sort
//
// Purpose:	To sort the give string list.
//
// Exmaple:
//					fs.Sort(pDirList);
//
// Notes:		None
//
//***************************************************************************
void
CFileSystem::Sort(CStringList *pStringList)
{
	int nListSize = pStringList->GetCount();
	for (int i = 0; i < nListSize-1; i++)
	{
		CString String1 = pStringList->GetAt(pStringList->FindIndex(i));
		for (int j = i+1; j < nListSize; j++)
		{
			CString String2 = pStringList->GetAt(pStringList->FindIndex(j));
			if (String1 > String2)
			{
				// Swap.
				pStringList->SetAt(pStringList->FindIndex(i), String2);
				pStringList->SetAt(pStringList->FindIndex(j), String1);

				String1 = String2;
			} // if
		} // for
	} // for
} // Sort


//***************************************************************************
//
// Name:		LoadListBox
//
// Purpose:	To load a list box with the given string list.
//
// Exmaple:
//					fs.LoadListBox(pListBox, pDirList);
//
// Notes:		None
//
//***************************************************************************
void
CFileSystem::LoadListBox(CListBox *pListBox, const CStringList * pStringList)
{
	for (POSITION pos=pStringList->GetHeadPosition(); pos != 0; )
	{
		CString String = pStringList->GetNext(pos);
		pListBox->AddString(String);
	} // for
} // LoadListBox


//***************************************************************************
//
// Name:		LoadComboBox
//
// Purpose:	To load a combo box with the given string list.
//
// Example:
//					fs.LoadListBox(pComboBox1, pDirList);
//
// Notes:		None
//
//***************************************************************************
void
CFileSystem::LoadComboBox(CComboBox *pComboBox, const CStringList * pStringList)
{
	for (POSITION pos=pStringList->GetHeadPosition(); pos != 0; )
	{
		CString String = pStringList->GetNext(pos);
		pComboBox->AddString(String);
	} // for
} // LoadComboBox


//***************************************************************************
//
// Name:		AppendWildcard
//
// Purpose:	To append a wildcard to a path.  It takes into account
//					whether the path has a backslash at the end.
//
// Example:
//					CString foo = fs.AppendWildcard("c:\\foo\\bar", "*.txt");
//					ASSERT(foo == "c:\\foo\\bar\\*.txt");
//					CString foo = fs.AppendWildcard("c:\\foo\\bar\\", "*.txt");
//					ASSERT(foo == "c:\\foo\\bar\\*.txt");
//
// Notes:		If the path is the empty string, the Wildcard is returned.
//
//***************************************************************************
CString
CFileSystem::AppendWildcard(const CString& Path, const CString& Wildcard)
{
	long lLength = Path.GetLength();
	CString RetVal;

	if (lLength == 0  ||  Path.GetAt((int)(lLength - 1)) == '\\')
	{
		RetVal = Path + Wildcard;
	} // if
	else
	{
		RetVal = Path + "\\" + Wildcard;
	} // else

	return RetVal;
} // AppendWildcard

