/***************************** ATI DRIVER *******************************/
/*	ATI VGA Wonder BGI driver.  This program acts as a BGI driver	*/
/*  for the ATI VGA Wonder in the 256 colour modes.  In these modes	*/
/*  the memory is mapped as follows.  One pixel per byte, laid out 	*/
/*  linearly in 64K pages (banks).  Only one bank is accesible at a	*/
/*  time.  Up to 8 banks of memory are used for the highest resolution	*/
/*  mode.								*/
/*									*/
/*  640 x 400 mode memory layout:					*/
/*									*/
/*  A0000:0 -> --------- <========================================\	*/
/*	      |         |---------                                |	*/
/*	      |         |         |----------                     |	*/
/*	      |         |         |          |---------- <==\=====|	*/
/*	      |         |         |          |          |   |     |	*/
/*	       ---------|         |          |          |  64K    |	*/
/*	        Page 0   ---------|          |          |   |   256,000	*/
/*			   Page 1  ----------|          |   |     |	*/
/*				      Page 2  ---------- <==/=====/	*/
/*					        Page 3			*/
/*									*/
/*									*/
/*  640 x 480 mode memory layout:					*/
/*									*/
/*  A0000:0 -> -------- <=========================================\	*/
/*            |        |--------                                  |	*/
/*	      |        |        |--------                         |	*/
/*	      |        |        |        |--------                |	*/
/*	      |        |        |        |        |------- <=\    |	*/
/*	       --------|        |        |        |       |  |    |	*/
/*	        Page 0  --------|        |        |       | 64K   |	*/
/*		         Page 1  --------|        |       |  | 307,200	*/
/*			          Page 2  --------|       |  |    |	*/
/*					   Page 3  ------- <=/====/	*/
/*						    Page 4		*/
/*									*/
/*  640 x 800 mode memory layout:					*/
/*									*/
/*  A0000:0 -> -------- <========================================\	*/
/*	      |        |--------                                 |	*/
/*	      |        |        |--------                        |	*/
/*	      |        |        |        | \                     |	*/
/*	      |        |        |        |  \                    |	*/
/*	       --------|        |        |   \                   |	*/
/*	        Page 0  --------|        |    \                  |	*/
/*		     	 Page 1  --------       -------- <=\  480,000	*/
/*				  Page 3  \    |        |  |     |	*/
/*					   \   |        | 64K    |	*/
/*					    \  |        |  |     |	*/
/*					     \ |        |  |     |	*/
/*						-------- <=/=====/	*/
/*						 Page 7			*/
/*									*/
/*									*/
/* 	V 0.90	01/04/89  Robert Adsett Original.			*/
/* 	V 0.95	17/04/89  Robert Adsett Recode basic block operations	*/
/*				in assembler to increase the speed. 	*/
/*				The effect is quite dramatic.		*/
/*	V 1.00  18/05/90  Robert Adsett Clean up the comments some. 	*/
/*				This is the release version.  Some	*/
/*				features are not fully implemented yet	*/
/*				but it seems solid enough.		*/
/*	V 1.10	30/06/90  Robert Adsett Add colour setting code so that	*/
/*				colour commands other than 		*/
/*				'setrgbcolor()' will work.		*/
/*	V 1.11	11/07/90  Robert Adsett Was copying one too many rows	*/
/*					in save/restore bitmap.		*/
/*									*/
/*  LIMITATIONS:							*/
/*	Flood fill is unimplemented at present.				*/
/*	Error checking & reporting are almost non-existant.		*/
/*	Everything that can be emulated is.  This seems to be 		*/
/*   reasonably fast so changing this is not a high priority.		*/
/*									*/
/*  KNOWN BUGS:								*/
/*	Line width is currently ignored.  This seems to be taken care	*/
/*   of by the kernal but should be changed.				*/
/************************************************************************/

#include <stdlib.h>
#include <string.h>
#include "bgi.h"
#include "ati.h"

/*	Generic driver global variables.  Should be present in almost	*/
/*  every driver.							*/

const CHAR_TABLE_ENTRY far * const char_def = 
			(CHAR_TABLE_ENTRY far *)MK_FP( 0xf000, 0xfa6e);
				/* Pointer to character definition	*/
				/*  table in ROM.			*/

				/* Fill pattern definitions.		*/
const FILLPATTERN def_patterns[12] = {
		{ 0, 0, 0, 0, 0, 0, 0, 0 }, 	/* No Fill.		*/
		{ 0xff, 0xff, 0xff, 0xff, 
		  0xff, 0xff, 0xff, 0xff },	/* Solid Fill.		*/
		{ 0xff, 0xff, 0, 0, 0xff, 
		  0xff, 0, 0},			/* Line Fill.		*/
		{ 1, 2, 4, 8, 0x10, 0x20,
		  0x40, 0x80},			/* Lt Slash fill.	*/
		{ 0xE0, 0xC1, 0x83, 0x07, 
		  0x0E, 0x1C, 0x38, 0x70 },	/* Slash Fill.		*/
		{ 0xF0, 0x78, 0x3C, 0x1E, 
		  0x0F, 0x87, 0xC3, 0xE1 },	/* Backslash Fill.	*/
		{ 0xA5, 0xD2, 0x69, 0xB4, 
		  0x5A, 0x2D, 0x96, 0x4B },	/* Lt Backslash Fill.	*/
		{ 0xFF, 0x88, 0x88, 0x88, 
		  0xFF, 0x88, 0x88, 0x88 },	/* Hatch Fill.		*/
		{ 0x81, 0x42, 0x24, 0x18, 
		  0x18, 0x24, 0x42, 0x81 },	/* XHatch Fill.		*/
		{ 0xCC, 0x33, 0xCC, 0x33, 
		  0xCC, 0x33, 0xCC, 0x33 },	/* Interleave Fill.	*/
		{ 0x80, 0x00, 0x08, 0x00, 
		  0x80, 0x00, 0x08, 0x00 }, 	/* Wide Dot Fill.	*/
		{ 0x88, 0x00, 0x22, 0x00, 
		  0x88, 0x00, 0x22, 0x00 }	/* Close Dot Fill.	*/
	       };
/*									*/
/*	The following structure defines the Bit Manipulation Utility	*/
/*	function table.							*/
/*									*/

const UTILITIES Utility_Table = {	/* Bit Utilities Function Table */
  (NRFPTR) dispatch_enter_graphics,	/* Enter graphics mode function */
  (NRFPTR) dispatch_leave_graphics,	/* Leave graphics mode function */
  (NRFPTR) dispatch_putpix,		/* Write a pixel function	*/
  (NRFPTR) dispatch_getpix,		/* Read a pixel function	*/
  (NRFPTR) dispatch_bits_per_pixel,	/* Bits per pixel value		*/
  (NRFPTR) dispatch_set_page,		/* Set the active drawing page	*/
  (NRFPTR) dispatch_set_visual,		/* Set the active display page	*/
  (NRFPTR) dispatch_set_write_mode	/* Set the current write mode	*/
  };

int MAXY = 399, MAXX = 639, MINY = 0, 	/* Clipping limits.		*/
    MINX = 0;
int CURRENT_MODE, TEXT_MODE;
unsigned int current_line_style = 0xffff;	/* Current line drawing	*/
						/*  style.		*/
int current_write_mode = COPY;		/* Current drawing mode.	*/
int current_line_width = 1;		/* Current line width.		*/
unsigned char current_colour = 0xff, 	/* Current drawing,		*/
	      fill_colour = 0xff,	/*  filling,			*/
              background_colour = 0;	/*  and background colour.	*/

unsigned int	CP_X = 0, CP_Y = 0;	/* Current Drawing Pointer CP.	*/
unsigned int	char_size, char_path;	/* Current character size and	*/
					/*  drawing path.		*/

const unsigned int line_style_mask[16] = {	/* A set of bit masks	*/
					0x8000,	/*  used to mask bits	*/
					0x4000,	/*  from the current	*/
					0x2000,	/*  line style.		*/
					0x1000,
					0x0800,
					0x0400,
					0x0200,
					0x0100,
					0x0080,
					0x0040,
					0x0020,
					0x0010,
					0x0008,
					0x0004,
					0x0002,
					0x0001
					};

STATUS	Stat_Block = {		/* Device status block.			*/
  0,				/* Current device status.		*/
  0,				/* Device Type Identifier.		*/
  639,				/* Device Full Resolution in X		*/
  399,				/* Device Full Resolution in Y		*/
  639,				/* Device Effective X Resolution	*/
  399,				/* Device Effective Y Resolution	*/
  9000,				/* Device X Size in inches*1000		*/
  7000,				/* Device Y Size in inches*1000		*/
  10000,			/* Aspect Ratio * 10000			*/
  				/* For compatibility the other fields 	*/
				/*  set so.				*/
  8,
  8,
  0x90,
  0x90
  };

ATIPALETTE Default_Palette = {	/* Default palette.  			*/
  16, { 0x00, 0x00, 0x00,	/* In this case set during startup.	*/
  	0x00, 0x00, 0x0f,
	0x00, 0x0f, 0x00,
	0x00, 0x00, 0x03,
	0x00, 0x00, 0x04,
	0x00, 0x00, 0x05,
	0x00, 0x00, 0x07,
	0x00, 0x00, 0x14,
	0x00, 0x00, 0x38,
	0x00, 0x00, 0x39,
	0x00, 0x00, 0x3A,
	0x00, 0x00, 0x3B,
	0x00, 0x00, 0x3C,
	0x00, 0x00, 0x3D,
	0x00, 0x00, 0x3E,
	0x00, 0x00, 0x3F,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0x00, 0x00, 0x00,
	0xff, 0xff, 0xff}
  };

/* Global variables specific to this driver.				*/

int rowsize;			/* Number of bytes in a row.		*/
struct BANK bank[8];		/* Information about what each bank 	*/
				/*   contains (up to 8 banks).		*/
unsigned char current_bank = 0,	/* Current bank.			*/
    maximum_bank;		/* Maximum bank for this mode.		*/

SCREEN_PTR const screen_buffer = (SCREEN_PTR)MK_FP( ATI_SEGMENT, 0);
				/* Pointer to the base of video memory.	*/
unsigned char current_pattern[8][8];	/* Current fill pattern stored	*/
					/*  as pixels.			*/
SCREEN_PTR current_address,	/* Current address into video memory.	*/
	   old_address;		/* Previous address.  Used in 		*/
	 			/*   incrementing.			*/

/*									*/
/*	Function Protoypes local to the ATI driver.			*/
/*									*/

void copy_image( 
	unsigned char const far *from, 
	int from_xsize, 
        unsigned char far *to, 
	int to_xsize,
	int mode
	);
void char_draw( unsigned char c);
void set_pattern( unsigned char *current_pattern, 
		  unsigned char const far *pattern);
void update_pattern( void);


/******************************* INSTALL ********************************/
/*	The Install function is used to prepare the driver to use.  The	*/
/*  calls to this function allow the kernal to inquire the mode 	*/
/*  information, and allow the kernal to install the mode infomation.	*/
/*  sets the error code to 'grInvalidMode' for invalid mode numbers	*/
/*  and 'grError' for an unrecognized command.				*/
/************************************************************************/

const char *CopyRight = 
           "ATI VGA Wonder BGI driver (V 1.10) Copyright R. Adsett 1990.";

				/* Names of the drivers modes.  The 	*/
				/*  first character is the length of 	*/
				/*  the string.				*/
char *Name[] = {
		"\x21 ATI VGA Wonder (640 x 400 x 256)",
		"\x21 ATI VGA Wonder (640 x 480 x 256)",
		"\x21 ATI VGA Wonder (800 x 600 x 256)"
		};

long install(
	unsigned int mode,	/* Mode to use.				*/
	char command		/* Install sub command.			*/
	)
{
long ret_code;

switch( command )			/* Determine the command to use */
    {
    case 0:				/* Install Device Command.	*/
         if((mode & 0xff) >= MAX_MODES)	/* Is the mode requested valid? */
              {				/*  No set an error code.	*/
              Stat_Block.stat = grInvalidMode;
	      }
	 CURRENT_MODE = mode & 0xff;	/* Get mode to use.		*/
	 switch( CURRENT_MODE)		/* Set up mode specific info.	*/
	      {
	      case ATI256LO:
	           Stat_Block.xres = 639;	/* Resolution in this	*/
	           Stat_Block.yres = 399;	/*   mode.		*/
	           Stat_Block.xefres = 639;
	           Stat_Block.yefres = 399;
		   break;

	      case ATI256MED:
	           Stat_Block.xres = 639;	/* Resolution in this	*/
	           Stat_Block.yres = 479;	/*   mode.		*/
	           Stat_Block.xefres = 639;
	           Stat_Block.yefres = 479;
		   break;

	      case ATI256HI:
	           Stat_Block.xres = 799;	/* Resolution in this	*/
	           Stat_Block.yres = 599;	/*   mode.		*/
	           Stat_Block.xefres = 799;
	           Stat_Block.yefres = 599;
		   break;
	      }
         ret_code = (unsigned int)&Stat_Block;	/* Return pointer to 	*/
         break;					/*  the status block.	*/

    case 1:				/* Mode Query Command.		*/
         ret_code = (long)MAX_MODES << 16; /* Return number of modes.	*/
         break;

    case 2:				/* Mode Name Command.		*/
         if( (mode & 0xff) > MAX_MODES)	/* Is the mode requested valid? */
              {				/*  No set an error code.	*/
              Stat_Block.stat = grInvalidMode;
	      }
	      				/* Return pointer to the name.	*/
         ret_code = (unsigned int)Name[mode & 0xff];
         break;

    default:				/* Unknown Install Call.	*/
         Stat_Block.stat = grError;	/* General error.		*/
         break;
    }					/* End of Install command case.	*/
return( ret_code);			/* Return pointer, mode numbers */
}

/******************************* INITIALIZE *****************************/
/*	The initialize function is used to enter the graphics mode.	*/
/*  Save the current mode so that it can be restored later.		*/
/************************************************************************/

void init( unsigned int dit_offset, unsigned int dit_segment)
{
struct DIT far *Dev_info;
struct REGPACK mode_regs;
void *deftbl;
static int mode_table[] = { 0x61, 0x62, 0x63 };
static struct BANK bankmed[] = { {0, 102, 0, 255},
				 { 102, 204, 256, 511},
				 { 204, 307, 512, 127},
				 { 307, 409, 128, 383},
				 { 409, 511, 384, 639}
			       };
static struct BANK bankhi[] = { { 0, 81, 0, 735},
				{ 81, 163, 736, 671},
				{ 163, 245, 672, 607},
				{ 245, 327, 608, 543},
				{ 327, 409, 544, 479},
				{ 409, 491, 480, 415},
				{ 491, 573, 416, 351},
				{ 573, 655, 352, 287}
			      };

Dev_info = (struct DIT far *)MK_FP( dit_segment, dit_offset);

background_colour = Dev_info->background;
					/* Set the background colour.	*/
update_pattern();			/* Set up the pattern for use.	*/
deftbl = &(Default_Palette.colora[0]); 	/* Load table offset.		*/

mode_regs.r_ax = 0x1200;		/* Tell the bios to load the 	*/
mode_regs.r_bx = 0x31;			/*  default palette when it 	*/
intr( 0x10, &mode_regs);		/*  switches modes.		*/

if( Dev_info->init != 0xA5)		/* Don't ask me!		*/
    {
    mode_regs.r_ax = 0xf00;		/* Get current mode.		*/
    intr( 0x10, &mode_regs);
    TEXT_MODE = mode_regs.r_ax & 0xff;

    					/* Set asked for mode.		*/
    mode_regs.r_ax = mode_table[ CURRENT_MODE] & 0xff;
    intr( 0x10, &mode_regs);

    switch( CURRENT_MODE)		/* Set up mode specific info.	*/
         {
	 case ATI256LO:
	      rowsize = 640;		/* Bytes per row.		*/
	      maximum_bank = 4;		/* Number of banks of memory.	*/
	      				/* Bank info for this mode.	*/
	      memcpy( bank, bankmed, sizeof( bankmed));
	      break;

	 case ATI256MED:
	      rowsize = 640;   		/* Bytes per row.		*/
	      maximum_bank = 5;		/* Number of banks of memory.	*/
	      				/* Bank info for this mode.	*/
	      memcpy( bank, bankmed, sizeof( bankmed));
	      break;

	 case ATI256HI:
	      rowsize = 800;   		/* Bytes per row.		*/
	      maximum_bank = 8;		/* Number of banks of memory.	*/
	      				/* Bank info for this mode.	*/
	      memcpy( bank, bankhi, sizeof( bankhi));
	      break;

	 }
    }
palette( 0x80ff, 0xff,  0xff, 0xff);
mode_regs.r_es = _DS;			/* Load colour table segment.	*/
mode_regs.r_dx = (unsigned int)deftbl;	/* Load table offset.		*/
mode_regs.r_bx = 0;			/* Beginning of the table.	*/
mode_regs.r_cx = 255;			/* Size of the table.		*/
mode_regs.r_ax = 0x1017;		/* Request read all palette	*/
intr( 0x10, &mode_regs);		/*  service.			*/
}

/******************************* CLEAR **********************************/
/*	Clear the screen.  Write zero everywhere.  			*/
/************************************************************************/

void clear( void )
{

for( current_bank = 0; current_bank < maximum_bank; current_bank++)
    {
    BANK_SELECT();
    set_mem( screen_buffer, 0, 0);
    }
}

/********************************** POST ********************************/
/*	Go to text mode.  Use mode stored previously.			*/
/************************************************************************/

void post( void )
{
struct REGPACK mode_regs;

mode_regs.r_ax = TEXT_MODE;
intr( 0x10, &mode_regs);
}

/********************************* MOVE *********************************/
/*	This function is used to move the current pointer (CP). This is	*/
/*  a pretty generic routine, library later?  				*/
/************************************************************************/

void move( int x, int y)
{

CP_X = x;				/* Update the current pointer.	*/
CP_Y = y;
}

/******************************** DRAW **********************************/
/*	Draw a line vector from the CP to the specified coordinate.	*/
/*  Update the CP to the new coordinate.				*/
/************************************************************************/

void draw( int x, int y)
{

line( CP_X, CP_Y, x, y);		/* Draw the line.		*/
CP_X = x;				/* Update the current pointer	*/
CP_Y = y;
}

/********************************* VECT *********************************/
/*	Draw a line between the two specified points.			*/
/************************************************************************/

void vect( int x1, int y1, int x2, int y2)
{

line( x1, y1, x2, y2);		/* Draw the line.			*/
}

/********************************* PATBAR *******************************/
/*	Fill a rectangle with the current filling pattern.  Do not 	*/
/*  outline.  The coordinates passed are the upper left corner and the	*/
/*  lower right corner.							*/
/************************************************************************/

void patbar( int x1, int y1, int x2, int y2)
{
int x3, y3;

if( x2 < x1)			/* Sometimes the emulation routines get	*/
    {				/*   the y coords. backwards.  We'll	*/
    x3 = x2; x2 = x1; x1 = x3;	/*   check the x coords. as well just 	*/
    }				/*   to be sure.			*/
if( y2 < y1)
    {
    y3 = y2; y2 = y1; y1 = y3;
    }
for( ; y1 <= y2; y1++)		/* For each line.			*/
    {
    int pre_done;

    x3 = x1;
    y3 = y1;
    pre_done = 0;
    CALC_ADDR( x3, y3);		/* Calculate start address....		*/
    if( (x3 & 7) != 0)
         {
	 int size;

	 pre_done = 8 - x3 & 7;
	 size = ((x2-x1+1) < pre_done) ? (x2-x1+1) : pre_done;
         if( current_address + size < current_address)
	      {
	      int size2;

	      size2 = -(int)current_address;
              copy_image( &current_pattern[y1&7][x3&7], pre_done, 
	             current_address, size2, COPY);
    				/* ... and copy appropriate line of the	*/
				/*  current pattern in.			*/
              CALC_ADDR( x3 + size2, y3);   /* Find the beginning.	*/
              copy_image( &current_pattern[y1&7][x3&7 + size2], pre_done,
	             current_address, size - size2, COPY);
	      }
	 else
	      {
              copy_image( &current_pattern[y1&7][x3&7], pre_done, 
	             current_address, size, COPY);
	      }
         CALC_ADDR( x3 + pre_done, y3);	/* Calculate start address....	*/
	 x3 += pre_done;
	 }
    if( pre_done < x2-x1+1)
         {
	 int size;

	 size = x2 - x1 + 1 - pre_done;
         if( current_address + size < current_address)
              {
	      int size2;

	      size2 = -(int)current_address;
              copy_image( current_pattern[y1&7], 8, current_address, 
	                  size2, COPY);
    				/* ... and copy appropriate line of the	*/
				/*  current pattern in.			*/
              CALC_ADDR( x3 + size2, y3);   /* Find the beginning.	*/
              copy_image( current_pattern[y1&7], 8, current_address, 
	                  size - size2, COPY);
    				/* ... and copy appropriate line of the	*/
				/*  current pattern in.			*/
	      }
	 else
	      {
              copy_image( current_pattern[y1&7], 8, current_address, 
	                  size, COPY);
    				/* ... and copy appropriate line of the	*/
				/*  current pattern in.			*/
	      }
	 }
    }
}

/******************************* PALETTE ********************************/
/*	Set the colour palette entries.  Only the set RGB colour method	*/
/*  is supported.							*/
/************************************************************************/

void palette( int flag_index, int red_colour,  int blue, int green)
{
int index;
char r, g, b;
struct REGPACK mode_regs;

index = flag_index & 0x3fff;		/* Grab colour table index.	*/
r = red_colour & 0x3f;			/* Grab colours.		*/
g = green & 0x3f;
b = blue & 0x3f;
switch(flag_index & 0xc000)
    {
    case 0x4000:
    	 /*	unused.							*/
	 break;

    case 0x0000:			/* Load colour table with colour*/
	 b = r & 3;			/*  in red_colour.  It's mapped	*/
	 g = (r >> 2) & 3;		/*  as 2 bits each for red, 	*/
	 r = (r >> 4) & 3;		/*  green, and blue.  Set up	*/
	 /*FALLTHROUGH*/		/*  and let setrgb do the work.	*/

    case 0x8000:			/* Load rgb.			*/
	 mode_regs.r_bx = index;	/* Entry to load.		*/
	 mode_regs.r_dx = r << 8;	/* Colour values to set it to.	*/
	 mode_regs.r_cx = (g << 8) | b;
	 mode_regs.r_ax = 0x1010;	/* Request set palette service.	*/
	 intr( 0x10, &mode_regs);	/* Call the Bios.		*/
	 break;
    	 
    case 0xc000:			/*  Load background.		*/
         background_colour = r;
	 break;
    }
}

/****************************** ALLPALETTE ******************************/
/*	The BGI kernal only passes the first 16 entries.  Therefore 	*/
/*  this function ignores the passed argument and simply restores the	*/
/*  default palette.  (At least that way the results are consistent). 	*/
/************************************************************************/

void allpalette( unsigned int pptr_offset, unsigned int pptr_segment)
{
struct REGPACK mode_regs;

mode_regs.r_es = _DS;			/* Load colour table segment.	*/
mode_regs.r_dx = (unsigned int)&Default_Palette; /* Load table offset.	*/
mode_regs.r_dx++;
mode_regs.r_bx = 0;			/* Beginning of the table.	*/
mode_regs.r_cx = 255;			/* Size of the table.		*/
mode_regs.r_ax = 0x1012;		/* Request set all palette	*/
intr( 0x10, &mode_regs);		/*  service.			*/
}

/******************************* COLOR **********************************/
/*	Sets new foreground (drawing) and fill colours.			*/
/************************************************************************/

void color( char new_fill_colour, char new_draw_colour )
{

new_draw_colour &= 0xff;	/* Mask colours to possible range.	*/
new_fill_colour &= 0xff;
current_colour = new_draw_colour;
fill_colour = new_fill_colour;
update_pattern();		/* Update current pattern to reflect	*/
				/*   the new colours.			*/
}

/******************************* FILLSTYLE ******************************/
/*	Set the current fillstyle.					*/
/************************************************************************/

static int current_pattern_no = 0, user_pattern[8];
				/* Current pattern and user pattern.	*/

void fillstyle( unsigned char pattern, unsigned int pptr_offset,
	unsigned int pptr_segment)
{
unsigned char far *pptr;
int i;

pptr = MK_FP( pptr_segment, pptr_offset);	/* Ptr. to user line	*/
						/*   style.		*/
current_pattern_no = pattern;		/* Save current style.		*/
if( current_pattern_no == 0xff )	/* User defined line style	*/
    {
    set_pattern( (unsigned char *)&current_pattern, pptr);
    				/* Make pattern accessible to filling 	*/
				/*   routine.				*/
    for( i = 0; i < 8; i++)	/* Save User pattern for later use.	*/
         {
	 user_pattern[i] = *(int far *)(pptr + i*sizeof( int));
	 }
    }
else
    {
    set_pattern( (unsigned char *)current_pattern, 
                 (unsigned char far *)&def_patterns[pattern]);
    				/* Make pattern accessible to filling 	*/
				/*   routine.				*/
    }
}

void update_pattern( void)	/* Update pattern to take care of new	*/
{				/*   colours.				*/

if( current_pattern_no == 0xff )	/* User defined line style	*/
    {
    set_pattern( (unsigned char *)current_pattern, 
                 (unsigned char far *)user_pattern);
    				/* Make pattern accessible to filling 	*/
				/*   routine.				*/
    }
else
    {
    set_pattern( (unsigned char *)current_pattern, 
                 (unsigned char far *)&def_patterns[current_pattern_no]);
    				/* Make pattern accessible to filling 	*/
				/*   routine.				*/
    }
}

/******************************* LINESTYLE ******************************/
/*	Set the current line style.  This includes drawing pattern and	*/
/*  width.								*/
/************************************************************************/

void linestyle( char style, int pattern, int width)
{

switch( style)			/* Set appropriate line pattern.	*/
    {
    case SOLID_LINE:
         current_line_style = 0xffff;
	 break;

    case DOTTED_LINE:
         current_line_style = 0xCCCC;
	 break;

    case CENTRE_LINE:
         current_line_style = 0xFC78;
	 break;

    case DASHED_LINE:
         current_line_style = 0xF8F8;
	 break;

    case USER_LINE:
         current_line_style = pattern;
	 break;

    default:
         break;
    }
current_line_width = width;		/* Save the width.		*/
}

/******************************* TEXTSTYLE ******************************/
/*	Set the text path and size.  Returns x & y size as a long.	*/
/************************************************************************/

long textstyle( char number, char path, int xsize, int ysize)
{

char_path = path;			/* Record path.			*/
char_size = xsize >> 3;			/* Convert text size to a	*/
					/*  multiple of 8.		*/
if( char_size == 0)			/* Must be at least 1.		*/
    char_size = 1;
xsize = ysize = char_size << 3;		/* Compute actual size.		*/
return( (((long)xsize) << 16) | ysize);	/* Return actual size.		*/
}

/******************************** TEXT **********************************/
/*	Draw a text string.						*/
/************************************************************************/

void text( int length, unsigned int offset, unsigned int segment)
{
char far * cptr = MK_FP( segment, offset);
int i;

for( i = 0; i < length; ++i ) 		/* For all characters ...	*/
    {
    char_draw( *cptr++);		/* Draw it.			*/
    }
}

/****************************** FLOODFILL *******************************/
/*	Unimplemented. Null function.  Is this really of importance to	*/
/*  anyone?								*/
/************************************************************************/

void floodfill( int x, int y, unsigned char boundary)
{
}

/******************************* BITMAPUTIL *****************************/
/*	Get the address of the bit map utility table.			*/
/************************************************************************/

void *bitmaputil( void )
{

return(&Utility_Table);
}

/*									*/
/*	The following define the bit map utility functions.		*/
/*									*/

void enter_graphics( void )		/* Enter graphics mode function.*/
{					/* Null function.		*/
}

void leave_graphics( void )		/* Leave graphics mode function */
{					/* Null function.		*/
}

int bits_per_pixel( void )		/* Enter graphics mode function */
{

return( 8);				/* Always 8 bits/pixel.		*/
}

void putpix( int x, int y, char colour)	/* Write a pixel function	*/
{

colour &= 0xff;
CALC_ADDR( x, y);			/* Calculate address.		*/
POINT( colour);				/* Draw pixel.			*/
}

char getpix( int x, int y)		/* Read a pixel function	*/
{

CALC_ADDR( x, y);			/* Calculate address.		*/
return( RD_POINT());			/* Read pixel colour.		*/
}

void set_page( char page)		/* Set the active drawing page	*/
{					/* Null function.  Only 1 page.	*/

}

void set_visual( char page)		/* Set the active display page	*/
{					/* Null function.  Only 1 page.	*/

}

void set_write_mode( int mode)		/* Set the current write mode	*/
{

current_write_mode = mode;		/* Save write mode.		*/
}


/****************************** RESTOREBITMAP ***************************/
/*	Copy a bitmap to video memory.					*/
/************************************************************************/

void restorebitmap( char mode, unsigned int segment, unsigned int offset,
		int x1, int y1, int x2, int y2)
{
const unsigned char far *buffer;
int xsize, x3, y3, bump;

x1 = x2;		/* Bug in doc's. */
y1 = y2;

buffer = (const unsigned char far *)MK_FP( segment, offset);
					/* Bitmap address.		*/
xsize = *(int far *)buffer + 1;		/* Columns.			*/
buffer += sizeof( int);
y2 = y1 + *(int far *)buffer + 1;	/* Rows.			*/
bump = xsize + 1;			/* Size of row in memory.  This	*/
					/* should be agreed on with 	*/
					/* 'savebitmap'.		*/
buffer += sizeof(int);			/* Increment to storage.	*/
for( ;y1 < y2; y1++)			/* For each line...		*/
    {
    x3 = x1;
    y3 = y1;
    CALC_ADDR( x3, y3);			/* Find the beginning.		*/
    if( current_address + xsize < current_address)
         {
	 int size2;

	 size2 = -(int)current_address;
         copy_image( buffer, size2, current_address, size2, mode);
    					/* Copy the line using the	*/
					/*  appropriate mode.		*/
         CALC_ADDR( x3 + size2, y3);	/* Find the beginning.		*/
         copy_image( buffer + size2, xsize - size2, current_address, 
	             xsize - size2, mode);
    					/* Copy the line using the	*/
					/*  appropriate mode.		*/
	 }
    else
         {
         copy_image( buffer, xsize, current_address, xsize, mode);
    					/* Copy the line using the	*/
					/*  appropriate mode.		*/
	 }
    buffer += bump;			/* Increment bitmap to the next	*/
    }					/*  line.			*/
}

/****************************** SAVEBITMAP ******************************/
/*	Copy an area of video memory to a bitmap.			*/
/************************************************************************/

void savebitmap( unsigned int buff_segment, unsigned int buff_offset,
		int x1, int y1, int x2, int y2)
{
unsigned char far *buffer;
int xsize, x3, y3, bump;

x1 = x2;		/* Bug in doc's. */
y1 = y2;
buffer = (unsigned char far *)MK_FP( buff_segment, buff_offset);
					/* Bitmap address.		*/
xsize = *(int far *)buffer + 1;		/* Columns.			*/
buffer += sizeof( int);
y2 = y1 + *(int far *)buffer + 1;	/* Rows.			*/
bump = xsize +1;			/* Size of row in memory.  This	*/
					/* should be agreed on with 	*/
					/* 'restorebitmap'.		*/
buffer += sizeof(int);			/* Increment to storage.	*/
for( ;y1 < y2; y1++)			/* For each line...		*/
    {
    x3 = x1;
    y3 = y1;
    CALC_ADDR( x3, y3);			/* Find the beginning.		*/
    if( current_address + xsize < current_address)
         {
	 int size2;

	 size2 = -(int)current_address;
         copy_image( current_address, size2, buffer, size2, COPY);
    					/* Copy the line using the	*/
					/*  copy mode.			*/
         CALC_ADDR( x3 + size2, y3);	/* Find the beginning.		*/
         copy_image( current_address, xsize - size2, buffer + size2, 
	             xsize - size2, COPY);
    					/* Copy the line using the	*/
					/*  copy mode.			*/
	 }
    else
         {
         copy_image( current_address, xsize, buffer, xsize, COPY);
    					/* Copy the line using the	*/
					/*  copy mode.			*/
	 }
    buffer += bump;			/* Increment bitmap to the next	*/
    }					/*  line.			*/
}

/****************************** SETCLIP *********************************/
/*	Set the clipping area.	Library?				*/
/************************************************************************/

void setclip( int x1, int y1, int x2, int y2)
{

MINX = x1;				/* Save the clipping limits.	*/
MAXY = y1;
MAXX = x2;
MAXY = y2;
}

/***************************** GET_PIXEL ********************************/
/*	Read a pixel colour from the screen.				*/
/************************************************************************/

char get_pixel( int x, int y)
{

CALC_ADDR( x, y);			/* Calculate the address.	*/
return( RD_POINT());			/* Read the pixel.		*/
}

/***************************** SET_PIXEL ********************************/
/*	Set a pixel to a specific colour.				*/
/************************************************************************/

void set_pixel( int x, int y, char colour)
{

colour &= 0xff;
CALC_ADDR( x, y);			/* Calculate address.		*/
POINT( colour);				/* Draw the pixel.		*/
}

/****************************** TEXTSIZ *********************************/
/*	Return the pixel size of a string.				*/
/************************************************************************/

long textsiz( int length, unsigned int offset, unsigned int segment)
{
if( char_path == NORMAL_PATH)		/* Horizontal.			*/
    {
    return( (((long)(length*char_size*8))<<16) | (8*char_size));
    }
else					/* Vertical.			*/
    {
    return( (((long)(char_size*8))<<16) | (length*8*char_size));
    }
}

/****************************** COLOR_QUERY *****************************/
/*	Get colour palette & size.					*/
/************************************************************************/

long color_query( char command_type)
{
int i;
long ret_val;


switch( command_type )			/* Act on the input command.	*/
    {
    case 0:				/* Color palette size query.	*/
         ret_val = ((long)(255)) << 16;
         ret_val |= 256;
         break;

    case 1:				/* Default palette settings.	*/
         ret_val = (unsigned long)&Default_Palette;
         break;

    default:				/* Unknown command.		*/
         break;
    }
return( ret_val);
}

/********************* CHAR_DRAW ****************************************/
/*	Draw a character.						*/
/************************************************************************/
void char_draw( unsigned char c)
{
static unsigned char char_bit_mask[8] = { 128, 64, 32, 16, 8, 4, 2, 1};
const CHAR_TABLE_ENTRY far *current;
int i, j, k, l, tx, ty;
unsigned char row_cur;

if( c > 127) return;		/* Don't do upper 128.			*/
current = char_def + c;		/* Get character definition from ROM.	*/
if( char_path == NORMAL_PATH)	/* Draw horizontal.			*/
    {
    for( i = 0; i < 8; i++)			/*  For each row...	*/
         {
         row_cur = (current->row)[i];		/* Def. for this line.	*/
         for( j = 0; j < char_size; j++)	/* Size multiplier.	*/
              {
	      for( k = 0; k < 8; k++) 		/*  For each column...	*/
	           {
	           if( row_cur & char_bit_mask[k])	/* Column def.	*/
	                {
	                for( l = 0; l < char_size; l++)	/* Draw 'size'	*/
	                     {				/*  points.	*/
			     tx = CP_X;
			     ty = CP_Y;
			     DRAW_POINT( tx, ty);
		             CP_X++;
		             }
		        }
	           else			/* Advance 'size' points.	*/
	                {
		        CP_X += char_size;
	                }
	           }
	      CP_X -= char_size*8;		/* Back to begining col.*/
	      CP_Y++;				/* Next line.		*/
	      }
         }
    CP_X += char_size*8;		/* Next character.		*/
    CP_Y -= char_size*8;		/* Charcter top.		*/
    }
else				/* Draw vertical.			*/
    {
    for( i = 0; i < 8; i++)			/*  For each row...	*/
         {
         row_cur = (current->row)[i];		/* Def. for this line.	*/
         for( j = 0; j < char_size; j++)	/* Size multiplier.	*/
              {
	      for( k = 0; k < 8; k++) 		/*  For each column...	*/
	           {
	           if( row_cur & char_bit_mask[k])	/* Column def.	*/
	                {
	                for( l = 0; l < char_size; l++)
	                     {
			     tx = CP_X;
			     ty = CP_Y;
			     DRAW_POINT( tx, ty);
		             CP_Y--;
		             }
		        }
	           else			/* Advance 'size' points.	*/
	                {
		        CP_Y -= char_size;
	                }
	           }
	      CP_Y += char_size*8;		/* Back to begining col.*/
	      CP_X++;		  		/* Next line.		*/
	      }
         }
    CP_Y -= char_size*8;	  	/* Next character.		*/
    CP_X -= char_size*8;	  	/* Charcter top.		*/
    }
}

/************************* COPY_IMAGE ***********************************/
/*	Copy an image line from one area of memory to another.  The 	*/
/*  source image is repeated as necessary to fill the destination.  	*/
/*  This is sufficient to form the core of a basic set of two parameter	*/
/*  BiTBlT routines except for the fact that no precautions are taken 	*/
/*  against overlap.  Could be sped up considerably by recoding in	*/
/*  assembly code.							*/
/************************************************************************/

void copy_image( 
	unsigned char const far *from, 
	int from_xsize, 
        unsigned char far *to, 
	int to_xsize,
	int mode
	)
{

switch( mode)				/* Copy mode.			*/
    {
    case COPY:				/* Copy verbatim.		*/
         copy_mem( from, from_xsize, to, to_xsize);
	 break;

    case XOR:			       	/* Copy xor.			*/
         xor_mem( from, from_xsize, to, to_xsize);
	 break;

    case OR:			       	/* Copy or.			*/
         or_mem( from, from_xsize, to, to_xsize);
	 break;

    case AND:			       	/* Copy and.			*/
         and_mem( from, from_xsize, to, to_xsize);
	 break;

    case NEGATE:		       	/* Copy negate.			*/
         neg_mem( from, from_xsize, to, to_xsize);
	 break;

    default:
         break;
    }
}

/*************************** SET_PATTERN ********************************/
/*	Change a bit pattern into a colour image pattern that can be	*/
/*  easily copied.  The bit pattern is an 8x8 pattern.			*/
/************************************************************************/

void set_pattern( unsigned char *current_pattern, 
		  unsigned char const far *pattern)
{
static unsigned char pat_mask[8] = { 128, 64, 32, 16, 8, 4, 2, 1};
int i, j;

for( i = 0; i < 8; i++)		/* For each line.			*/
    {
    for( j = 0; j < 8; j++)	/* For each column.			*/
         {
	 if( (*pattern) & pat_mask[j])	/* Set to fill colour.		*/
	      {
	      *current_pattern = fill_colour;
	      }
	 else			/* Or the background colour.		*/
	      {
	      *current_pattern = background_colour;
	      }
	 current_pattern++;
	 }
    pattern++;			/* Next line.				*/
    }
}


