/****************************************************************************

    PROGRAM: Pattern.c

    PURPOSE: Bring up an MDI and pick child windows of various patterns.

    FUNCTIONS:

		WinMain() - calls initialization function, processes message loop
		InitApplication() - initializes window data and registers window
		InitInstance() - saves instance handle and creates main window
		InitNewChild() - creates new child window through MDI client
		MainWndProc() - processes messages
		OurMDIChildProc() - processes messages for child windows
		About() - processes messages for the about dialog box
		PatternSelect() - processes messages for the pattern select d.b.

    COMMENTS:

        Windows can have several copies of your application running at the
        same time.  The variable hInst keeps track of which instance this
        application is so that processing will be to the correct window.

****************************************************************************/

#include "windows.h"		/* required for all Windows applications	 */
#include "pattern.h"		/* specific to this program				     */

HANDLE hInst;				/* current instance						     */
HWND   hWndMain;	        /* Main window handle.		                 */
HWND   hWndClient;	        /* Client window handle.		             */
HANDLE hAccel;				/* handle to accelerator table.				 */
CLIENTCREATESTRUCT clcrMain;/* MDI client menu handle.					 */
int    nCurrentPattern;		/* The current pattern to use as background. */
int	   nPattern;			/* Temporary version of nCurrentPattern.	 */

char * rgszPatternTitle[]	/* Titles for the child window types.		 */
	= { "Diagonal Window", "Horizontal Window", "Vertical Window"
		, "Square Window", "Diamond Window" };
char * rgszPatternClass[]	/* Classes for the child window types.		 */
	= { "DiagsClass", "HorizClass", "VertClass"
		, "SquaresClass", "DiamondsClass" };

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point 
        for the program.  This function calls the application initialization 
        routine, if no other instance of the program is running, and always 
        calls the instance initialization routine.  It then executes a message 
        retrieval and dispatch loop that is the top-level control structure 
        for the remainder of execution.  The loop is terminated when a WM_QUIT 
        message is received, at which time this function exits the application 
        instance by returning the value passed by PostQuitMessage(). 

        If this function must abort before entering the message loop, it 
        returns the conventional value NULL.  

**************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;				     /* current instance		         */
HANDLE hPrevInstance;			     /* previous instance		         */
LPSTR lpCmdLine;				     /* command line				     */
int nCmdShow;					     /* show-window type (open/icon)     */
{
    MSG msg;					     /* message						     */

    if (!hPrevInstance)				 /* Other instances of app running?  */
		if (!InitApplication(hInstance)) /* Initialize shared things     */
		    return (FALSE);			 /* Exits if unable to initialize    */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received.*/
    while (GetMessage(&msg,	   /* message structure					     */
		    NULL,			   /* handle of window receiving the message */
		    NULL,			   /* lowest message to examine			     */
		    NULL))			   /* highest message to examine		     */
	 	if ( !TranslateMDISysAccel (hWndClient, &msg)
		     && !TranslateAccelerator (hWndMain, hAccel, &msg) )
 	 	{
			TranslateMessage(&msg);	   /* Translates virtual key codes	 */
			DispatchMessage(&msg);	   /* Dispatches message to window	 */
    	}
    return (msg.wParam);	   /* Returns the value from PostQuitMessage */
}


/**************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other 
        instances of the application are running.  This function performs 
        initialization tasks that can be done once for any number of running 
        instances.  

        In this case, we initialize a window class by filling out a data 
        structure of type WNDCLASS and calling the Windows RegisterClass() 
        function.  Since all instances of this application use the same window 
        class, we only need to do this when the first instance is initialized.  


****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;		 		       /* current instance			      */
{
    WNDCLASS	wc;

    /* Fill in window class structure with parameters that describe the   */
    /* main window and child windows.                                     */


	/* Style for main window */

	wc.style = NULL;                /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;   /* Function to retrieve messages  	  */
                                    /* for windows of this class.         */
    wc.cbClsExtra = 0;              /* No per-class extra data.           */
    wc.cbWndExtra = 0;              /* No per-window extra data.          */
    wc.hInstance = hInstance;       /* Application that owns the class.   */
    wc.hIcon = LoadIcon(hInstance, "DiagsIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground
		= CreatePatternBrush(LoadBitmap(hInstance, "DiagsBitmap")); 
    wc.lpszMenuName =  "GenericMenu";   /* .RC name of menu resource.	  */
    wc.lpszClassName = "MainClass"; 	/* Name used in CreateWindow.	  */

	if (!RegisterClass(&wc))
		return(FALSE);


	/* Style for horizontal bar window */

    wc.style = NULL;                /* Class style(s).                    */
    wc.lpfnWndProc = OurMDIChildProc; /* Function to retrieve messages	  */
                                    /* for windows of this class.         */
    wc.cbClsExtra = 0;              /* No per-class extra data.           */
    wc.cbWndExtra = 0;              /* No per-window extra data.          */
    wc.hInstance = hInstance;       /* Application that owns the class.   */
    wc.hIcon = LoadIcon(hInstance, "HorizIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground
		= CreatePatternBrush(LoadBitmap(hInstance, "HorizBitmap")); 
    wc.lpszMenuName =  NULL;        /* .RC name of menu resource.		  */
    wc.lpszClassName = "HorizClass";    /* Name used in CreateWindow.	  */

	if (!RegisterClass(&wc))
		return(FALSE);


	/* Style for vertical bar window */

    wc.hIcon = LoadIcon(hInstance, "VertIcon");
    wc.hbrBackground
		= CreatePatternBrush(LoadBitmap(hInstance, "VertBitmap")); 
    wc.lpszClassName = "VertClass";    /* Name used in CreateWindow.	  */

	if (!RegisterClass(&wc))
		return(FALSE);


	/* Style for diagonalal bar window */

    wc.hIcon = LoadIcon(hInstance, "DiagsIcon");
    wc.hbrBackground
		= CreatePatternBrush(LoadBitmap(hInstance, "DiagsBitmap")); 
    wc.lpszClassName = "DiagsClass";    /* Name used in CreateWindow.	  */

	if (!RegisterClass(&wc))
		return(FALSE);


	/* Style for squares window */

    wc.hIcon = LoadIcon(hInstance, "SquaresIcon");
    wc.hbrBackground
		= CreatePatternBrush(LoadBitmap(hInstance, "SquaresBitmap")); 
    wc.lpszClassName = "SquaresClass";    /* Name used in CreateWindow.	  */

	if (!RegisterClass(&wc))
		return(FALSE);


	/* Style for diamonds window */

    wc.hIcon = LoadIcon(hInstance, "DiamondsIcon");
    wc.hbrBackground
		= CreatePatternBrush(LoadBitmap(hInstance, "DiamondsBitmap")); 
    wc.lpszClassName = "DiamondsClass";    /* Name used in CreateWindow.  */

	if (!RegisterClass(&wc))
		return(FALSE);


	return (TRUE);

}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of 
        this application.  This function performs initialization tasks that 
        cannot be shared by multiple instances.  

        In this case, we save the instance handle in a static variable and 
        create and display the main program window.  
        
****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;      /* Current instance identifier.       */
    int             nCmdShow;       /* Param for first ShowWindow() call. */
{
    HWND            hWnd;           /* Main window handle.                */

    /* Save the instance handle in static variable, which will be used in */
    /* many subsequence calls from this application to Windows.           */

    hInst = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "MainClass",	            /* See RegisterClass() call.          */
        "This is a Hack",		    /* Text for window title bar.         */
        WS_OVERLAPPEDWINDOW         /* Window style.                      */
			| WS_CLIPCHILDREN,
        CW_USEDEFAULT,              /* Default horizontal position.       */
        CW_USEDEFAULT,              /* Default vertical position.         */
        CW_USEDEFAULT,              /* Default width.                     */
        CW_USEDEFAULT,              /* Default height.                    */
        NULL,                       /* Overlapped windows have no parent. */
        NULL,                       /* Use the window class menu.         */
        hInstance,                  /* This instance owns this window.    */
        NULL                        /* Pointer not needed.                */
    );

    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);
	hWndMain = hWnd;

	/* Now create the MDI client window ... */

	clcrMain.hWindowMenu = GetSubMenu(GetMenu(hWndMain), IDM_WINDOW_POPUP);
	clcrMain.idFirstChild = IDM_WINDOW_BASE;

	hWnd = CreateWindow(
        "MDICLIENT",	            /* This is an MDI client window		  */
        NULL,					    /* Text for window title bar.         */
		WS_CHILD | WS_CLIPCHILDREN	/* Window style.                      */
			| WS_VSCROLL | WS_HSCROLL,
        0, 0, 0, 0,		            /* Just fill the screen.			  */
        hWndMain,                   /* Parent window.					  */
        NULL,		                /* Use the window class menu.         */
        hInstance,                  /* This instance owns this window.    */
        (LPSTR)&clcrMain            /* Pointer to MDI information		  */
    );

    if (!hWnd)
        return (FALSE);
	hWndClient = hWnd;

    /* Make the window visible; update its client area; and return success */

    ShowWindow(hWndMain, nCmdShow);  /* Show the window                    */
    UpdateWindow(hWndMain);          /* Sends WM_PAINT message             */
	hAccel = LoadAccelerators (hInst, "GenericMenu");

	ShowWindow(hWndClient, SW_SHOW); /* Show the Client window ... 		   */

	return (TRUE);

}

/****************************************************************************

    FUNCTION:  InitNewChild

    PURPOSE:  Creates a child window with a background determined
				by nCurrentPattern.

    COMMENTS:

		This routine is called when the user requests a new window.
		The MDI client window will do the actual creation.

****************************************************************************/

BOOL InitNewChild()
{
    HWND            hWnd;           /* Main window handle.                */
	MDICREATESTRUCT	mdicr;			/* Creation information for child	  */

	/* set up the MDI create structure, MDI client will create window */
	
	mdicr.szTitle = rgszPatternTitle[nCurrentPattern];
	mdicr.szClass = rgszPatternClass[nCurrentPattern];
	mdicr.hOwner = hInst;
	mdicr.x = CW_USEDEFAULT;
	mdicr.y = CW_USEDEFAULT;
	mdicr.cx = CW_USEDEFAULT;
	mdicr.cy = CW_USEDEFAULT;
	mdicr.style = WS_CHILD | WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS;

	/* send a message to the MDI client window to create the window */

	hWnd = (WORD)SendMessage(hWndClient, WM_MDICREATE, NULL
			, (DWORD)(MDICREATESTRUCT FAR *)&mdicr);

	ShowWindow(hWnd, SW_SHOW);		  /* Show the window         */
    UpdateWindow(hWnd);   	     	  /* Sends WM_PAINT message  */

	return (TRUE);               /* Returns the value from PostQuitMessage */

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

    MESSAGES:

		WM_COMMAND    - application menu
		WM_DESTROY    - destroy window

    COMMENTS:

		Note that most of the window control functions are handled by
		simply sending messages to the MDI client window.

****************************************************************************/

long FAR PASCAL MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;					   		  /* window handle				     */
unsigned message;			   		  /* type of message				 */
WORD wParam;				   		  /* additional information		     */
LONG lParam;				   		  /* additional information		     */
{
    FARPROC lpProcDialog;			  /* pointer to the Dialog function  */
	int		fNew;

    switch (message)
	{
		case WM_COMMAND:	   /* message: command from application menu */
			switch(wParam)
			{
				case IDM_ABOUT:
					lpProcDialog = MakeProcInstance(About, hInst);

					DialogBox(hInst,		 /* current instance	     */
					    "AboutBox",			 /* resource to use		     */
					    hWnd,				 /* parent handle		     */
					    lpProcDialog);		 /* dialog instance address  */

					FreeProcInstance(lpProcDialog);
					break;

				case IDM_NEW:
					lpProcDialog = MakeProcInstance(PatternSelect, hInst);

					fNew = DialogBox
						(hInst,				 /* current instance	     */
					    "PatternSelectBox",	 /* resource to use		     */
					    hWnd,				 /* parent handle		     */
					    lpProcDialog);		 /* dialog instance address  */

					FreeProcInstance(lpProcDialog);

					if (fNew)
						InitNewChild();

					break;

				case IDM_EXIT:
		   			PostQuitMessage(0);
		    		break;

				case IDM_CASCADE:
					SendMessage(hWndClient, WM_MDICASCADE, NULL, NULL);
					break;

				case IDM_TILE:
					SendMessage(hWndClient, WM_MDITILE, NULL, NULL);
					break;

				case IDM_ARRANGE:
					SendMessage(hWndClient, WM_MDIICONARRANGE, NULL, NULL);
					break;

				case IDM_NEXT:
					SendMessage(hWndClient, WM_MDINEXT, NULL, NULL);
					break;

				default:		    		 /* Lets Windows process it	 */
					return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));
		    }
			break;

		case WM_DESTROY:			  /* message: window being destroyed */
		    PostQuitMessage(0);
		    break;

		default:					  /* Passes it on if unproccessed    */
		    return (DefFrameProc(hWnd, hWndClient, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: OurMDIChildProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for our child windows

    MESSAGES:

    COMMENTS:

		This function could be eliminated by just using DefMDIChildProc
		in the class definition.  It is kept here as a place holder in
		case non-default child message handling is required.

		See notes in SDK documentation about what messages have to be
		passed to the default handler even if the application handles
		them.

****************************************************************************/

long FAR PASCAL OurMDIChildProc(hWnd, message, wParam, lParam)
HWND hWnd;					   		  /* window handle				     */
unsigned message;			   		  /* type of message				 */
WORD wParam;				   		  /* additional information		     */
LONG lParam;				   		  /* additional information		     */
{
    switch (message)
	{
		default:					  /* Passes it on if unproccessed    */
		    return (DefMDIChildProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for the abort dialog box

    MESSAGES:

		WM_INITDIALOG - initialize dialog box
		WM_COMMAND    - Input received

    COMMENTS:

		No initialization is needed for this particular dialog box, but TRUE
		must be returned to Windows.

		Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL About(hDlg, message, wParam, lParam)
HWND hDlg;                            /* window handle of the dialog box */
unsigned message;                     /* type of message                 */
WORD wParam;                          /* message-specific information    */
LONG lParam;
{
    switch (message)
	{
		case WM_INITDIALOG:			  /* message: initialize dialog box  */
		    return (TRUE);

		case WM_COMMAND:				  /* message: received a command */
		    if (wParam == IDOK)           /* "OK" box selected?		     */
			{
				EndDialog(hDlg, TRUE);	  /* Exits the dialog box	     */
				return (TRUE);
		    }
            else if (wParam == IDCANCEL)  /* System menu close, cancel?  */
			{
				EndDialog(hDlg, FALSE);	  /* Exits the dialog box	     */
				return (TRUE);
		    }
		    break;
    }
    return (FALSE);						  /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: PatternSelect(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for the pattern selection dialog box

    MESSAGES:

		WM_INITDIALOG - initialize dialog box
		WM_COMMAND    - Input received

    COMMENTS:

		No initialization is needed for this particular dialog box, but TRUE
		must be returned to Windows.

		Wait for user to click on "Ok" or "Cancel" buttons, then close the
		dialog box.  Handle checking the radio buttons for visual feedback
		to the user.

****************************************************************************/

BOOL FAR PASCAL PatternSelect(hDlg, message, wParam, lParam)
HWND hDlg;                            /* window handle of the dialog box */
unsigned message;                     /* type of message                 */
WORD wParam;                          /* message-specific information    */
LONG lParam;
{
    switch (message)
	{
		case WM_INITDIALOG:			  /* message: initialize dialog box  */
			nPattern = nCurrentPattern;
			CheckRadioButton(hDlg, IDD_DIAG, IDD_DIAMOND
							 , nPattern + IDD_DIAG);
		    return (TRUE);

		case WM_COMMAND:				  /* message: received a command */
			switch(wParam)
			{
				case IDOK:
					nCurrentPattern = nPattern;
					EndDialog(hDlg, TRUE);
					return (TRUE);

				case IDCANCEL:
					EndDialog(hDlg, FALSE);
					return (TRUE);

				case IDD_DIAG:
				case IDD_HORIZ:
				case IDD_VERT:
				case IDD_SQUARE:
				case IDD_DIAMOND:
					nPattern = wParam - IDD_DIAG;
					CheckRadioButton(hDlg, IDD_DIAG, IDD_DIAMOND, wParam);
					break;
			}
			break;
    }
    return (FALSE);						  /* Didn't process a message    */
}
