/*//////////////////////////////////////////////////////////////////////////
///            ___                                                       ///
///          /_____\                                                     ///
///         |       |                 Copyright (c) 1991                 ///
///         |   R   |                                                    ///
///     ----|_______|----                     by                         ///
///   /------/ | | \------\                                              ///
///  |       | | | |       |      --  Object Resource Group  --          ///
///  |   O   | | | |   G   |                                             ///
///  |       |/   \|       |          4323 Brown Suite 249               ///
///   -------       -------            Dallas,  TX  75219                ///
///   Object Resource Group                                              ///
///                                      (214) 528-2745                  ///
///                                                                      ///
///                                    All Rights Reserved.              ///
///                                                                      ///
//////////////////////////////////////////////////////////////////////////*/

//
//  A small example program using the ORG Btrieve classes
//
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "btdset.hpp"
#include "btskey.hpp"
#include "btdef.hpp"
#include "bttran.hpp"
#include "examples.hpp"


///////////////////////////////////////////////////////////////////////
//////////////////////// Internal Prototypes //////////////////////////
///////////////////////////////////////////////////////////////////////

int CreatePersonDB();
int CreateColorDB();
int GetAndSetColors(PersonDB &personDB, ColorDB &colorDB);
int SetColor(Person &person, ColorDB &colorDB);


///////////////////////////////////////////////////////////////////////
/////////////////////// Class Implementations /////////////////////////
///////////////////////////////////////////////////////////////////////

// Normally class implementations are seperate from main file but
// for convenience of exapmles I have kept them together

// Implementation of PersonDB

// Dump - dumps in person order every one in file
int PersonDB::Dump()
   {
   BT_Key *nameKey = GetKey(0);
   Person person(this);
   char buffer[80];
   int status=nameKey->GetFirst(&person);
   int count=0;

   cout << "\n\n                   -----Dump of PERSON.DB-----\n";
   cout << "\n------------- Name ----------------     --- Color Preference ---\n";
   while (!status)
      {
      count++;
      sprintf(buffer, "%-35.35s     %-25.25s\n", person.Name(),
			person.ColorPreference());
      cout << buffer;
      status=nameKey->GetNext(&person);
      }
   cout << "\nTotal Number of People: " << count << "\n\n";

   if (status == BSTAT_EOF) status=0;
   return status;
   }


// ColorDump - dumps in color order every one in file
int PersonDB::ColorDump()
   {
   BT_SuppKeyDef keyDef;

   int status = keyDef.AddFinalKeySegment(37, 26, BKEY_EXTENDED+BKEY_DUPLICATE,
					 BTYPE_ZSTRING);

   BT_SuppKey *colorKey = GetSuppKey(keyDef);

   Person person(this);
   char buffer[80];
   status=colorKey->GetFirst(&person);
   int count=0;

   cout << "\n\n                -----Color Dump of PERSON.DB-----\n";
   cout << "\n------------- Name ----------------     --- Color Preference ---\n";
   while (!status)
      {
      count++;
      sprintf(buffer, "%-35.35s     %-25.25s\n", person.Name(),
			person.ColorPreference());
      cout << buffer;
      status=colorKey->GetNext(&person);
      }
   cout << "\nTotal Number of People: " << count << "\n\n";

   if (!status || status == BSTAT_EOF)
     status=colorKey->DropKey();
   else
      colorKey->DropKey();	// just drop it anyway but return previous
				// status.
   return status;
   }


// Implementation of ColorDB

// Dump - dumps in person order every one in file
int ColorDB::Dump()
   {
   BT_Key *shadeKey = GetKey(0);
   Color color(this);
   char buffer[80];
   int status=shadeKey->GetFirst(&color);
   int count=0;

   cout << "\n\n             -----Dump of COLOR.DB-----\n";
   cout << "\n----- Color Shade -----    --- Preference Count ---\n";
   while (!status)
      {
      count++;
      sprintf(buffer, "%-25.25s             %d\n", color.Shade(),
			color.PreferenceCount());
      cout << buffer;
      status=shadeKey->GetNext(&color);
      }
   cout << "\nTotal Number of Colors: " << count << "\n\n";

   if (status == BSTAT_EOF) status=0;
   return status;
   }


int ColorDB::DumpFromHighestCount()
   {
   // silly exampe to demonstrate GetPosition and Restore Position
   BT_Key *shadeKey = GetKey(0);
   BT_Key *countKey = GetKey(1);
   Color color(this);
   char buffer[80];
   int count=0;
   int status=countKey->GetLast();

   if (!status)
      {
      ULONG position = GetPosition();
      status = shadeKey->RestorePosition(position, &color);
      // in the above case as only one position was gotton this is also
      // correct:
      //     status = shadeKey->RestorePosition();
      // it will restore to the position of the last GetPosition call

      cout << "\n\n -----Dump of COLOR.DB After Highest Count-----\n";
      cout << "\n----- Color Shade -----    --- Preference Count ---\n";
      while (!status)
	 {
	 count++;
	 sprintf(buffer, "%-25.25s             %d\n", color.Shade(),
			color.PreferenceCount());
	 cout << buffer;
	 status=shadeKey->GetNext(&color);
	 }
      cout << "\nTotal Number of Colors After Highest Count: " << count << "\n\n";
      }

   if (status == BSTAT_EOF) status=0;
   return status;
   }


// Implementation of Person

// Person(name, personDB) - constructor to connect a file to person and gets
//			    the name
Person::Person(char *_name, PersonDB *_personDB) : personDB(_personDB)
   {
   SetName(_name);
   GetRec();
   }

// GetRec() - gets a record associated with objects name
int Person::GetRec()
   {
   BT_Key *nameKey = personDB->GetKey(0);
   int status = nameKey->GetEqual(Name(), &data);
   return status;
   }

// Add() - writes a record to the file
int Person::Add()
   {
   BT_Key *nameKey = personDB->GetKey(0);
   int status = nameKey->Insert(&data);
   return status;
   }


// Implementation of Color

// Color(_name, personDB) - constructor to connect a file to color and get
//			    the shade
Color::Color(char *_name, ColorDB *_colorDB) : colorDB(_colorDB)
   {
   SetShade(_name);
   GetRec();
   }

// GetRec() - gets a record associated with objects name
int Color::GetRec()
   {
   BT_Key *shadeKey = colorDB->GetKey(0);
   int status = shadeKey->GetEqual(Shade(), &data);
   return status;
   }

// Add() - writes a record to the file
int Color::Add()
   {
   BT_Key *shadeKey = colorDB->GetKey(0);
   int status = shadeKey->Insert(&data);
   return status;
   }

// Update() - writes a record to the file
int Color::Update()
   {
   BT_Key *shadeKey = colorDB->GetKey(0);
   int status = shadeKey->Update(&data);
   return status;
   }


///////////////////////////////////////////////////////////////////////
/////////////////////////////// MAIN ///////////////////////////////////
////////////////////////////////////////////////////////////////////////


/////////// Btrieve shell is already running, right ? //////////

main()
    {
    int status;
    // create the file
    CreatePersonDB();
    CreateColorDB();

    // open the files
    PersonDB personDB("person.db");
    ColorDB colorDB("color.db");

    if (!personDB.Status() && !colorDB.Status()) // opened successfully
       {
       // poll the user for names and color preferences
       status = GetAndSetColors(personDB, colorDB);
       }

    // print out the results
    if (!status)
       status = personDB.Dump();

    if (!status)
       status = personDB.ColorDump();

    if (!status)
       status = colorDB.Dump();

    if (!status)
       status = colorDB.DumpFromHighestCount();

    ///// The file is closed when the destructor of person is called /////
    if (status)
       cout << "\nProgram terminated with error: " << status << "\n";
    else
       cout << "Program terminated successfully\n";
    }

// Create - Creates the person file
int CreatePersonDB()
   {
   int status=0;

   BT_Def fileDef(sizeof(PersonData), 0);

   ///// Define the Keys /////

   status = fileDef.AddFinalKeySegment(1, 36, BKEY_EXTENDED+
					      BKEY_MODIFIABLE+
					      BKEY_ALT_SEQUENCE,
					      BTYPE_ZSTRING);
   if (!status)
      {
      ///// set alternate collating sequence to the standard UPPER.ALT /////
      fileDef.SetUpperAlt();

      ///// This actually creates the dataset /////
      status = fileDef.CreateBtrieve("person.db");
      }

   return(status);
   }

// Create - Creates the color file
int CreateColorDB()
   {
   int status=0;

   BT_Def fileDef(sizeof(ColorData), 0);

   ///// Define the Keys /////

   status = fileDef.AddFinalKeySegment(1, 26, BKEY_EXTENDED+
					      BKEY_MODIFIABLE,
					      BTYPE_ZSTRING);
   if (!status)
      status = fileDef.AddFinalKeySegment(27, 2, BKEY_BINARY+
					      BKEY_MODIFIABLE+
					      BKEY_DUPLICATE,
					      BTYPE_INTEGER);
   if (!status)
      {
      ///// This actually creates the dataset /////
      status = fileDef.CreateBtrieve("color.db");
      }

   return(status);
   }

int GetAndSetColors(PersonDB &personDB, ColorDB &colorDB)
   {
   int status=0;

   // loop until no name is given
   char name[36];
   do
      {
      cout << "\nPlease enter person's name: ";
      gets(name);

      if (strlen(name))
	 {
	 // attempt to find in file
	 Person person(name, &personDB);// constructor gets name if possible
					// from personDB.

	 status = person.Status();	// Because constructors can't return
					// a status we use status of file
					// operation.  There are other ways
					// but this is a simple method.
	 if (status == BSTAT_NOTFOUND)
	    {
	    // this name is not in the file so get color and add it
	    status = SetColor(person, colorDB);
	    }
	 else if (!status)
	    {
	    cout << name << "'s color preference is " <<
			   person.ColorPreference() << "\n\n";
	    }
	 }
      else break;		// No Name - Exit Condition
      } while (!status);

   return status;
   }

int SetColor(Person &person, ColorDB &colorDB)
   {
   char colorPref[26];
   int status=0;

   // enter information
   cout << "Please enter " << person.Name() << "'s Color Preference: ";
   gets(colorPref);

   // if color was entered continue
   if (strlen(colorPref))
      {
      // capitalize on the color
      strupr(colorPref);

      // this is the goal:
      //    start a transaction
      //    if color record exists in color file,
      //       get it and increment its prefCount by one
      //    else
      //       add the color record to color file and set count to one
      //    insert the person record
      //    if successful
      //       end the transaction
      //    else
      //       abort transaction

      BT_Transaction transaction;
      if (!transaction.Status())		// creation was successful
	 {
	 Color color(colorPref, &colorDB);	// gets the color record
	 if (!color.Status())		// record was found
	    {
	    color.IncPrefCount();
	    status = color.Update();
	    }
	 else if (color.Status() == BSTAT_NOTFOUND) // record not found
	    {
	    // note that the constructor already set the shade
	    color.SetPrefCount(1);
	    status = color.Add();
	    }

	 if (!status)
	    {
	    person.SetColorPreference(colorPref);
	    status = person.Add();
	    }

	 // Commit if everything succeeded
	 if (!status)
	    status = transaction.Commit();
	 else
	    transaction.RollBack();
	 }
      }
   return status;
   }
