/////////////////////////////////////////////////////////////
// vector.h: Fixed-length shared vector template.
// Copyright(c) 1993 Azarona Software. All rights reserved.
/////////////////////////////////////////////////////////////
#ifndef H_VECTOR
#define H_VECTOR
#include "range.h"
#include "vecptr.h"

#define INLINE

enum SliceType { SHARED, COPIED };

// Vector representation class. Private to the vector class.

template<class TYPE> class Vector; // Forward template declaration

template<class TYPE>
class VecRep {
// WARNING: VecRep objects are meant to be allocated only 
// dynamically. The only exception is null_rep, which is
// a special null version of a VecRep object. Only one
// of these should be created.
#ifdef NO_BORLAND_BUG
private:
  friend class Vector<TYPE>;
#else
public:
#endif
  unsigned alloclen;
  unsigned refcnt;
  TYPE data[1];
  VecRep(unsigned d);
  void *operator new(size_t n, unsigned d);
  void operator delete(void *p);
public: // So we can set up null_rep
  static VecRep<TYPE> null_rep;
  VecRep();
  ~VecRep();
};

// You need to call this for each type of vector, to 
// set up the special null VecRep object. Do it only
// once though.

#define INITNULLVEC(TYPE) VecRep<TYPE> VecRep<TYPE>::null_rep;


// The vector class

template<class TYPE> class Matrix; // Forward template declaration

template<class TYPE>
class Vector {
protected:
  friend class Matrix<TYPE>;
  VecRep<TYPE> *rep; // Pointer to shared vector data
  TYPE *start;       // Pointer to logical start of data
  unsigned len;      // Number of logical elements
  unsigned stride;   // Stride (offset to next logical element)
  int Alloc(unsigned n);
  void Bind(const Vector<TYPE> &v);
  void Unbind();
  void NewBinding(const Vector<TYPE> &v);
public:
  Vector(unsigned n = 0, const TYPE *s = 0);
  Vector(const Vector<TYPE> &v);
  Vector(const Vector<TYPE> &v, SliceType styp,
         unsigned n=0, unsigned str=1, unsigned ofs=0);
  ~Vector();
#ifndef NO_RANGE_CHECK
  unsigned CheckIndx(unsigned i) const;
#endif
  // We'll provide both forms of subscripting
  TYPE &operator[](unsigned i);
  const TYPE &operator[](unsigned i) const;
  void CopyN(const TYPE *src, unsigned n);
  void Copy(const Vector<TYPE> &v);
  void Share(const Vector<TYPE> &v);
  void SetElements(const TYPE &x);
  Vector<TYPE> &operator=(const Vector<TYPE> &v);
  Vector<TYPE> &operator=(const TYPE &x);
  int IsNull() const;
  int IsUnique() const;
  Vector<TYPE> Clone() const;
  int EnsureUnique();
  unsigned Length() const;
  unsigned Stride() const;
  // Low-level hooks
  VecPtr<TYPE> PtrToAll();
  VecPtr<const TYPE> PtrToAll() const;
};

template<class TYPE>
INLINE VecRep<TYPE>::VecRep()
// Constructor to form a null VecRep object.
// This object should only be allocated statically.
// NOTE: TYPE's default constructor called implicitly.
{
  alloclen = 1;
  refcnt = 1;
}

template<class TYPE>
INLINE Vector<TYPE>::~Vector()
// Destructor unbinds this vector from the data
// it was sharing.
{ 
  Unbind(); 
}


template<class TYPE>
INLINE TYPE &Vector<TYPE>::operator[](unsigned i) 
// For non-const vectors.
{ 
  return start[CHECK(i)*stride]; 
}


template<class TYPE>
INLINE const TYPE &Vector<TYPE>::operator[](unsigned i) const 
// For const vectors.
{ 
  return start[CHECK(i)*stride]; 
}

template<class TYPE>
INLINE Vector<TYPE> &Vector<TYPE>::operator=(const Vector<TYPE> &v)
// Share semantics used for assignment.
{ 
  if (this != &v) Share(v); // Note trap for assignment to self
  return *this; 
}

template<class TYPE>
INLINE Vector<TYPE> &Vector<TYPE>::operator=(const TYPE &x)
// Sets each element of the vector to the value x.
{
  SetElements(x);
  return *this;
}

template<class TYPE>
INLINE int Vector<TYPE>::IsNull() const
// Returns true if the vector references null_rep.
{
  return rep == &VecRep<TYPE>::null_rep;
}

template<class TYPE>
INLINE int Vector<TYPE>::IsUnique() const
// Returns true if vector has only reference to shared data,
// or the shared data is null_rep.
{
  return rep->refcnt == 1 || IsNull();
}


template<class TYPE>
INLINE unsigned Vector<TYPE>::Length() const
// Return the logical number of elements of the vector.
{ 
  return len; 
}

template<class TYPE>
INLINE unsigned Vector<TYPE>::Stride() const 
// Return the vector's stride (ie. offset between
// each logical element.)
{
  return stride; 
}

template<class TYPE>
INLINE VecPtr<TYPE> Vector<TYPE>::PtrToAll()
// Returns a vector pointer to the beginning of
// this vector's elements.
{ 
  return VecPtr<TYPE>(start, stride); 
}

template<class TYPE>
INLINE VecPtr<const TYPE> Vector<TYPE>::PtrToAll() const
// Returns a vector pointer to the beginning of
// this vector's elements.
{ 
  return VecPtr<const TYPE>(start, stride); 
}

#undef INLINE

// Whether or not we should include the non-line methods for
// our class templates here is implementation dependent.

#include "vector.mth"

#endif
