/////////////////////////////////////////////////////////////
// tstsmat.cpp: A test program for the SimpleMatrix class.
// Copyright(c) 1993 Azarona Software. All rights reserved.
/////////////////////////////////////////////////////////////
#include <iostream.h>
#include <iomanip.h>
#include "simpmat.h"

template<class TYPE>
void PrtMat(SimpleMatrix<TYPE> &m)
// Prints out all elements of a matrix. 
// Tests subscripting operation as well.
{
  for (int i = 0; i<m.NRows(); i++) {
      for (int j = 0; j<m.NCols(); j++) {
          cout << setw(3) << m[i][j] << ' ';
      }
      cout << '\n';
  }
}

template<class TYPE>
void SetVec(VecPtr<TYPE> &v, unsigned n, const TYPE &x)
// Sets all elements of a vector to x. Assumes
// TYPE as '=' defined. Uses fastest form of vector
// element access.
{
  unsigned i;

  for (i = 0; i<n; i++) {
      *v = x;
      v++;
  }
}

template<class TYPE>
void MatMult(SimpleMatrix<TYPE> &c, const SimpleMatrix<TYPE> &a, const SimpleMatrix<TYPE> &b)
// Does a matrix multiply.
{
  unsigned i, j, k;

  if (a.NCols() == b.NRows() &&   // A & B must conform
      c.NCols() == b.NCols() &&   // C must be right size too
      c.NRows() == a.NRows()) {
    for (i = 0; i < a.NRows(); i++) {
        for (j = 0; j < b.NCols(); j++) {
            TYPE sum = 0;
            for (k = 0; k < b.NRows(); k++) sum += a[i][k] * b[k][j];
            c[i][j] = sum;
        }
    }
  }
  else {
    cout << "Matrices don't conform\n";
  }
}

// BC++ seems to need this prototype

void MatMult(SimpleMatrix<int> &c, const SimpleMatrix<int> &a, const SimpleMatrix<int> &b);

template<class TYPE>
SimpleMatrix<TYPE> operator*(const SimpleMatrix<TYPE> &a, const SimpleMatrix<TYPE> &b)
// Multiplies two matrices together and returns a resulting matrix.
// If matrices don't conform, a message is printed and the resulting
// matrix is set to all zeros.
{
  SimpleMatrix<TYPE> r(a.NRows(), b.NCols()); // Create resulting matrix
  MatMult(r, a, b);
  return r; // Copy constructor called. Bad news.
}


template<class TYPE>
SimpleMatrix<TYPE> Transpose(const SimpleMatrix<TYPE> &m)
// Returns the transpose of a matrix by interchanging
// rows and columns.
{
  SimpleMatrix<TYPE> t(m.NCols(), m.NRows());

  for (unsigned i = 0; i<m.NRows(); i++)
      for (unsigned j = 0; j<m.NCols(); j++) t[j][i] = m[i][j];

  return t; // Copy constructor called.
}


template<class TYPE>
void TRYIT(const SimpleMatrix<TYPE> &m)
// Try updating a constant matrix.
{
  unsigned i;
  VecPtr<const TYPE> v = m.ColPtr(0);
  
  for (i = 0; i<m.NRows(); i++) {
      // *v = 99; Can't do this
      v++;
  }
}


main()
{
  SimpleMatrix<int> m(3, 3);

  cout << "Setting all elements to 4's\n";
  SetVec(m.PtrToAll(), m.NElems(), 4);
  cout << "m:\n"; PrtMat(m); cout << '\n';

  cout << "Setting row 1 to all 2's\n";
  SetVec(m.RowPtr(1), m.NCols(), 2);
  cout << "m:\n"; PrtMat(m); cout << '\n';

  cout << "Setting column 1 to all 3's\n";
  SetVec(m.ColPtr(3), m.NRows(), 3);
  cout << "m:\n"; PrtMat(m); cout << '\n';

  cout << "Setting diagonal to 1's\n";
  SetVec(m.DiagPtr(), m.NRows(), 1);
  cout << "m:\n"; PrtMat(m); cout << '\n';

  // Do a matrix multiply with a 3x2 matrix

  SimpleMatrix<int> m2(3, 2), c(3,2); // Result is 3x2 as well

  unsigned k = 0;
  for(unsigned i = 0; i<m2.NRows(); i++)
     for (unsigned j = 0; j<m2.NCols(); j++) m2[i][j] = k++;

  cout << "m2:\n"; PrtMat(m2); cout << '\n';

  cout << "Multiplying m by m2\n";
  c = m * m2;
  cout << "c = m x m2:\n"; PrtMat(c); cout << '\n';

  // Transpose c
  cout << "Transpose(c):\n"; PrtMat(Transpose(c)); cout << '\n';

  TRYIT(m); // Try updating a constant matrix

  return 0;
}
