/////////////////////////////////////////////////////////////////
// simpmat.h: Class definition for simple dynamic matrices.
// Copyright(c) 1993 Azarona Software. All rights reserved.
/////////////////////////////////////////////////////////////////
#ifndef H_SIMPLEMAT
#define H_SIMPLEMAT
#include "vecptr.h"

#define INLINE

// Note: You can define the macro constant VECTORPTRRETURNS
// to change the class to return vector pointers for rows
// instead of ordinary pointers. By switching back and forth
// you can see for yourself what kind of overhead vector
// pointers do or don't have over ordinary pointers.

template<class TYPE>
class SimpleMatrix {
private:
  TYPE *data;
  unsigned nrows, ncols;
public:
  SimpleMatrix(unsigned nr, unsigned nc);
  SimpleMatrix(const SimpleMatrix<TYPE> &m);
  ~SimpleMatrix();
  void Copy(const SimpleMatrix<TYPE> &m);
  SimpleMatrix<TYPE> &operator=(const SimpleMatrix<TYPE> &m);
#ifndef VECTORPTRRETURNS
  TYPE *operator[](unsigned r);
#else
  VecPtr<TYPE> operator[](unsigned r);
#endif
  VecPtr<TYPE> RowPtr(unsigned r=0);
  VecPtr<TYPE> ColPtr(unsigned c=0);
  VecPtr<TYPE> DiagPtr();
  VecPtr<TYPE> PtrToAll();
#ifndef VECTORPTRRETURNS
  const TYPE *operator[](unsigned r) const;
#else
  VecPtr<const TYPE> operator[](unsigned r) const;
#endif
  VecPtr<const TYPE> RowPtr(unsigned r=0) const;
  VecPtr<const TYPE> ColPtr(unsigned c=0) const;
  VecPtr<const TYPE> DiagPtr() const;
  VecPtr<const TYPE> PtrToAll() const;
  unsigned NRows() const;
  unsigned NCols() const;
  unsigned NElems() const;
};

template<class TYPE>
INLINE SimpleMatrix<TYPE>::~SimpleMatrix()
{
  delete[] data;
}

template<class TYPE>
INLINE SimpleMatrix<TYPE> &
SimpleMatrix<TYPE>::operator=(const SimpleMatrix<TYPE> &m)
{
  if (this != &m) Copy(m); // Trap assignment to self
  return *this;
}

// Selectors for non-const matrices

#ifndef VECTORPTRRETURNS
template<class TYPE>
INLINE TYPE *SimpleMatrix<TYPE>::operator[](unsigned r)
// Returns an ordinary pointer (stride 1) to row r
{
  return data + r*ncols;
}
#else
template<class TYPE>
INLINE VecPtr<TYPE> SimpleMatrix<TYPE>::operator[](unsigned r)
// Returns an vector pointer (stride 1) to row r
{
  return VecPtr<TYPE>(data + r*ncols);
}
#endif

template<class TYPE>
INLINE VecPtr<TYPE> SimpleMatrix<TYPE>::RowPtr(unsigned r)
// Return a row vector pointer
{
  return VecPtr<TYPE>(data + r*ncols, 1);
}

template<class TYPE>
INLINE VecPtr<TYPE> SimpleMatrix<TYPE>::ColPtr(unsigned c)
// Return a column vector pointer
{
  return VecPtr<TYPE>(data+c, nrows);
}

template<class TYPE>
INLINE VecPtr<TYPE> SimpleMatrix<TYPE>::DiagPtr()
// Return a diagonal vector pointer. In case matrix
// isn't square, the smallest dimension is used to
// determine the diagonal length.
{
  unsigned dstride = ((nrows > ncols) ? ncols : nrows) + 1;
  return VecPtr<TYPE>(data, dstride);
}


template<class TYPE>
INLINE VecPtr<TYPE> SimpleMatrix<TYPE>::PtrToAll()
// Return pointer to all elements of the matrix
{
  return VecPtr<TYPE>(data, 1);
}


// Selectors for const matrices

#ifndef VECTORPTRRETURNS
template<class TYPE>
INLINE const TYPE *SimpleMatrix<TYPE>::operator[](unsigned r) const
// Returns an ordinary pointer (stride 1) to row r
{
  return data + r*ncols;
}
#else
template<class TYPE>
INLINE VecPtr<const TYPE> SimpleMatrix<TYPE>::operator[](unsigned r) const
// Returns an vector pointer (stride 1) to row r
{
  return VecPtr<const TYPE>(data + r*ncols);
}
#endif

template<class TYPE>
INLINE VecPtr<const TYPE> SimpleMatrix<TYPE>::RowPtr(unsigned r) const
// Return a row vector pointer
{
  return VecPtr<const TYPE>(data + r*ncols, 1);
}

template<class TYPE>
INLINE VecPtr<const TYPE> SimpleMatrix<TYPE>::ColPtr(unsigned c) const
// Return a column vector pointer
{
  return VecPtr<const TYPE>(data+c, nrows);
}

template<class TYPE>
INLINE VecPtr<const TYPE> SimpleMatrix<TYPE>::DiagPtr() const
// Return a diagonal vector pointer. In case matrix
// isn't square, the smallest dimension is used to
// determine the diagonal length.
{
  unsigned dstride = ((nrows > ncols) ? ncols : nrows) + 1;
  return VecPtr<const TYPE>(data, dstride);
}


template<class TYPE>
INLINE VecPtr<const TYPE> SimpleMatrix<TYPE>::PtrToAll() const
// Return pointer to all elements of the matrix
{
  return VecPtr<const TYPE>(data, 1);
}

template<class TYPE>
INLINE unsigned SimpleMatrix<TYPE>::NRows() const
{
  return nrows;
}

template<class TYPE>
INLINE unsigned SimpleMatrix<TYPE>::NCols() const
{
  return ncols;
}

template<class TYPE>
INLINE unsigned SimpleMatrix<TYPE>::NElems() const
{
  return nrows * ncols;
}

#undef INLINE

// Whether or not we should include the non-line methods for
// our class templates here is implementation dependent.

#include "simpmat.mth"

#endif
