///////////////////////////////////////////////////////////////
// tstvec.cpp: Vector test program using integer vector math.
// Copyright(c) 1993 Azarona Software. All rights reserved.
///////////////////////////////////////////////////////////////
#include <iostream.h>
#include "vector.h"

//
// Some sample vector math routines
//

template<class TYPE>
void operator+=(Vector<TYPE> &v, const TYPE &x)
// Adds x to every element of v.
// ASSUMES TYPE has '+=' defined.
{
  VecPtr<TYPE> elemptr(v.PtrToAll());
  for (unsigned i = 0; i<v.Length(); i++) {
      *elemptr += x;
      elemptr++;
  }
}

template<class TYPE>
void Add(Vector<TYPE> &r, const Vector<TYPE> &a, const Vector<TYPE> &b)
// Adds vectors a and b together, storing result in r.
// The number of elements added is determined by the
// shortest length of all three vectors.
// ASSUMES that TYPE has '+' defined.
{
  unsigned rlen = r.Length();
  unsigned slen = a.Length();
  if (rlen > slen) rlen = slen;
  slen = b.Length();
  if (rlen > slen) rlen = slen;
  VecPtr<TYPE> rptr(r.PtrToAll());
  VecPtr<const TYPE> aptr(a.PtrToAll());
  VecPtr<const TYPE> bptr(b.PtrToAll());
  for (unsigned i = 0; i<rlen; i++) {
      *rptr = *aptr + *bptr;
      rptr++;
      aptr++;
      bptr++;
  }
}

template<class TYPE>
Vector<TYPE> operator+(const Vector<TYPE> &a, const Vector<TYPE> &b)
// Adds vectors a and b together, returning the result (which
// has its data allocated on the heap.) The number of elements
// created is determined by the shortest length a and b.
{
  unsigned rlen = a.Length();
  unsigned slen = b.Length();
  if (rlen > slen) rlen = slen;
  Vector<TYPE> result(rlen);
  Add(result, a, b);
  return result;
}


template<class TYPE>
void PrtVec(const Vector<TYPE> &v)
// Print out the elements of a vector
{
  VecPtr<const TYPE> cursor(v.PtrToAll());
  unsigned i, len = v.Length();
  for (i = 0; i<len; i++) {
      cout << *cursor << ' ';
      cursor++;
  }
  cout << '\n';
}

// BC++ needs the next statements, which generate functions
// for doing operations on vectors

void Add(Vector<int> &r, const Vector<int> &a, const Vector<int> &b);
void operator+=(Vector<int> &v, const int &x);
void PrtVec(const Vector<int> &v);

// You need one of these statements for each type of vector:

INITNULLVEC(int);

// A low-level C array we'll copy

const int arr_size = 7;
int arr[arr_size] = {0, 1, 2, 3, 4, 5, 6};

void test()
{
  cout << "Creating a vector\n";

  Vector<int> myvec(arr_size, arr);
  cout << "myvec: "; PrtVec(myvec);

  cout << "Creating a shared slice of every other element\n";

  Vector<int> everyother(myvec, SHARED, 4, 2);
  cout << "everyother: "; PrtVec(everyother);

  cout << "Creating a copied slice of every other element\n";

  Vector<int> cpslice(myvec, COPIED, 4, 2);
  cout << "cpslice: "; PrtVec(cpslice);

  cout << "Adding 10 to every other element\n";

  everyother += 10;
  cout << "myvec:    "; PrtVec(myvec);
  cout << "everyother:  "; PrtVec(everyother);
  cout << "cpslice: "; PrtVec(cpslice);

  cout << "Adding all odd elements to even elements\n";

  Vector<int> odd(myvec, SHARED, 3, 2, 1);
  Vector<int> even(myvec, SHARED, 4, 2, 0);

  cout << "Result:   "; PrtVec(odd + even);
  cout << "myvec:    "; PrtVec(myvec);

  cout << "Testing consistency checks for stride, length, and offset\n";

  Vector<int> tstslice(everyother, SHARED, 4, 2, 1);
  cout << "everyother:  "; PrtVec(everyother);
  cout << "tstslice: "; PrtVec(tstslice);
}

main()
{
  test();
  return 0;
}

