//////////////////////////////////////////////////////////////
// tstsmat2.cpp: Timing test program for matrix multiplies
// of SimpleMatrix objects.
// Copyright(c) 1993 Azarona Software. All rights reserved.
//////////////////////////////////////////////////////////////

#include <iostream.h>
#include <iomanip.h>
#include <stdlib.h>
#include <conio.h>
#include "simpmat.h"

typedef double Number;

const int d1 = 30;  // We'll do 30x30 matrices
const int d2 = 30;

SimpleMatrix<Number> a(d1, d2);
SimpleMatrix<Number> b(d2, d1);
SimpleMatrix<Number> c(d1, d1);

void Test1(SimpleMatrix<Number> &c, 
  const SimpleMatrix<Number> &a, const SimpleMatrix<Number> &b)
// Uses double subscripting
{
  unsigned i, j, k;

  for (i = 0; i < a.NRows(); i++) {
      for (j = 0; j < b.NCols(); j++) {
          Number sum = 0;
          for (k = 0; k < b.NRows(); k++) sum += a[i][k] * b[k][j];
          c[i][j] = sum;
      }
  }
}

void Test2(SimpleMatrix<Number> &c, 
  const SimpleMatrix<Number> &a, const SimpleMatrix<Number> &b)
// Factors some of the vector construction out of the loops
{
  unsigned i, j, k;

  for (i = 0; i < a.NRows(); i++) {
      VecPtr<const Number> ar = a.RowPtr(i);
      VecPtr<Number> cr = c.RowPtr(i);
      for (j = 0; j < b.NCols(); j++) {
          Number sum = 0;
          VecPtr<const Number> bc = b.ColPtr(j);
          for (k = 0; k < b.NRows(); k++) {
               sum += ar[k] * bc[k];
          }
          cr[j] = sum;
      }
  }
}

void Test3(SimpleMatrix<Number> &c, 
  const SimpleMatrix<Number> &a, const SimpleMatrix<Number> &b)
// Fastest form using vector pointers
{
  unsigned i, j, k;

  VecPtr<const Number> ar = a.RowPtr();
  VecPtr<const Number> ac = a.ColPtr();
  VecPtr<const Number> br = b.RowPtr();
  VecPtr<const Number> bc = b.ColPtr();
  VecPtr<Number> cr = c.RowPtr();
  VecPtr<Number> cc = c.ColPtr();

  Number const *bstart = br;

  for (i = 0; i < a.NRows(); i++) {
      cr = cc; // Point to start of row i. Note: stride not copied
      br = bstart;
      for (j = 0; j < b.NCols(); j++) {
          Number sum = 0;
          ar = ac; // Point to start of row i. Note: stride not copied
          bc = br; // Point to start of col j. Note: stride not copied
          for (k = 0; k < b.NRows(); k++) {
               sum += *ar * *bc;
               ar++; // Next col
               bc++; // Next row
          }
          br++; // Next col
          *cr = sum;
          cr++; // Next col
      }
      ac++; // Next row
      cc++; // Next row
  }
}

void Test4(SimpleMatrix<Number> &c, 
   const SimpleMatrix<Number> &a, const SimpleMatrix<Number> &b)
// Possibly, fastest of all, by assuming row vectors have stride
// of 1, and thus using ordinary pointers instead
{
  unsigned i, j, k;

  const Number *ar        = a.RowPtr();
  VecPtr<const Number> ac = a.ColPtr();
  const Number *br        = b.RowPtr();
  VecPtr<const Number> bc = b.ColPtr();
  Number *cr              = c.RowPtr();
  VecPtr<Number> cc       = c.ColPtr();

  const Number *bstart = br;

  for (i = 0; i < a.NRows(); i++) {
      cr = cc; // Point to start of row i
      br = bstart;
      for (j = 0; j < b.NCols(); j++) {
          Number sum = 0;
          ar = ac; // Point to start of row i.
          bc = br; // Point to start of col j.
          for (k = 0; k < b.NRows(); k++) {
               sum += *ar * *bc;
               ar++; // Next col
               bc++; // Next row
          }
          br++; // Next col
          *cr = sum;
          cr++; // Next col
      }
      ac++; // Next row
      cc++; // Next row
  }
}

void testn(int n, unsigned unsigned num_iter)
{
  unsigned i;

  cout << "Press return to start test " << n;
  getch();
  cout << '\n';
  switch(n) {
    case 1: for(i = 0; i<num_iter; i++) Test1(c, a, b);
    break;
    case 2: for(i = 0; i<num_iter; i++) Test2(c, a, b);
    break;
    case 3: for(i = 0; i<num_iter; i++) Test3(c, a, b);
    break;
    case 4: for(i = 0; i<num_iter; i++) Test4(c, a, b);
    break;
  }
  cout << "Finished\n";
}

main(int argc, char *argv[])
{
  unsigned n = 1;

  if (argc <= 1) {
     cout << "Usage: tstmat2 [num_iter]\n";
     return 1;
  }

  n = (unsigned)atoi(argv[1]);

  unsigned i, j, k;


  for(i = 0, k = 1; i<d1; i++)
     for(j = 0; j<d2; j++) a[i][j] = k++;

  for(i = 0, k = 2; i<d2; i++)
     for(j = 0; j<d1; j++) b[i][j] = k++;

  testn(1, n);
  testn(2, n);
  testn(3, n);
  testn(4, n);
  return 0;
}

