////////////////////////////////////////////////////////////////
// tstmat3.cpp: Matrix test program using sample math routines.
// Copyright(c) 1993 Azarona Software. All rights reserved.
////////////////////////////////////////////////////////////////
#include <iostream.h>
#include <iomanip.h>
#include "matrix.h"

// We'll set up the type of number we'd like to store 
// in the matrices

typedef double Number;

// Choose multiply routine you would like to use for test
// Test4 = Uses double subscripting approach
// Test5 = Uses 2-d subscripting approach
// Test6 = Uses lowest level and fastest 

#define MatMult Test6


// Matrix multiply routines. Include these directly into
// your test programs. (Not a real library-style source file.)

template<class TYPE>
void Test4(Matrix<TYPE> &c, const Matrix<TYPE> &a, const Matrix<TYPE> &b)
// Uses dynamically allocated Matrix objects with double-subscripting.
// Really S--L--O--W.
{
  unsigned i, j, k;
  for (i = 0; i < a.NRows(); i++) {
      for (j = 0; j < b.NCols(); j++) {
          TYPE sum = 0;
          for (k = 0; k < b.NRows(); k++) sum += a[i][k] * b[k][j];
          c[i][j] = sum;
      }
  }
}

template<class TYPE>
void Test5(Matrix<TYPE> &c, const Matrix<TYPE> &a, const Matrix<TYPE> &b)
// Uses dynamically allocated Matrix objects with 2-d subscripting.
// Better than Test4, but still slow.
{
  unsigned i, j, k;

  for (i = 0; i < a.NRows(); i++) {
      for (j = 0; j < b.NCols(); j++) {
          TYPE sum = 0;
          for (k = 0; k < b.NRows(); k++) sum += a(i,k) * b(k,j);
          c(i,j) = sum;
      }
  }
}


template<class TYPE>
void Test6(Matrix<TYPE> &c, const Matrix<TYPE> &a, const Matrix<TYPE> &b)
// Fastest routine for Matrix objects, using vector pointers.
// Should be pretty decent.
{
  unsigned i, j, k;

  VecPtr<const TYPE> ar = a.RowPtr();
  VecPtr<const TYPE> ac = a.ColPtr();
  VecPtr<const TYPE> br = b.RowPtr();
  VecPtr<const TYPE> bc = b.ColPtr();
  VecPtr<TYPE> cr = c.RowPtr();
  VecPtr<TYPE> cc = c.ColPtr();
  
  const TYPE *bstart = b.PtrToAll();

  for (i = 0; i < a.NRows(); i++) {
      cr = cc; // Point to start of row i
      br = bstart;
      for (j = 0; j < b.NCols(); j++) {
          TYPE sum = 0;
          ar = ac; // Point to start of row i
          bc = br; // Point to start of col j
          for (k = 0; k < b.NRows(); k++) {
               sum += *ar * *bc;
               ar++; // Next col
               bc++; // Next row
          }
          br++; // Next col
          *cr = sum;
          cr++; // Next col
      }
      ac++; // Next row
      cc++; // Next row
  }
}

template<class TYPE>
Matrix<TYPE> operator*(const Matrix<TYPE> &a, const Matrix<TYPE> &b)
// Multiplies two matrices together and returns a resulting matrix.
// If matrices don't conform, a message is printed and the resulting
// matrix is set to all zeros.
{
  Matrix<TYPE> r(a.NRows(), b.NCols()); // Create resulting matrix
  MatMult(r, a, b);
  return r;
}

// BC++ seems to need this
void MatMult(Matrix<Number> &r,
             const Matrix<Number> &a, const Matrix<Number> &b);

template<class TYPE>
void PrtMat(const Matrix<TYPE> &m)
// We'll test 2D subscripting here while we're at it.
{
  for (int i = 0; i<m.NRows(); i++) {
      for (int j = 0; j<m.NCols(); j++) {
          cout << setw(3) << m(i,j) << ' ';
      }
      cout << '\n';
  }
}

typedef Vector<Number> vec;
typedef Matrix<Number> mat;

// You need one of these statements for each type of vector:

INITNULLVEC(Number);

// Create matrices out of raw data

Number arra[3*5] = {
   1,  2,  3,  4,  5,
   6,  7,  8,  9, 10,
  11, 12, 13, 14, 15
};

mat a(3, 5, arra);

Number arrb[5*2] = {
  1, 2,
  3, 4,
  5, 6,
  7, 8,
  9, 10
};

mat b(5, 2, arrb);


void test()
{

  cout << "Multiplying two whole matrices:\n\n";
  cout << "a: \n";
  PrtMat(a); cout << '\n';

  cout << "b: \n";
  PrtMat(b); cout << '\n';

  mat r = a * b;

  cout << "r: \n";
  PrtMat(r); cout << '\n';

  cout << "Multiplying two sub-matrices:\n\n";

  mat suba(a, SHARED, 0, 1, 2, 3);
  mat subb(b, SHARED, 2, 0, 3, 2);
  cout << "sub-a: \n";
  PrtMat(suba); cout << '\n';

  cout << "sub-b: \n";
  PrtMat(subb); cout << '\n';

  cout << "sub-r: \n";
  PrtMat(suba * subb); cout << '\n';
}

main()
{
  test();
  return 0;
}
