/////////////////////////////////////////////////////////////
// tstmat.cpp: Matrix test program using integers.
// Copyright(c) 1993 Azarona Software. All rights reserved.
/////////////////////////////////////////////////////////////
#include <iostream.h>
#include <iomanip.h>
#include "matrix.h"

template<class TYPE>
void PrtMat(const Matrix<TYPE> &m)
// We'll test 2D subscripting here while we're at it.
{
  for (int i = 0; i<m.NRows(); i++) {
      for (int j = 0; j<m.NCols(); j++) {
          cout << setw(3) << m(i,j) << ' ';
      }
      cout << '\n';
  }
}

template<class TYPE>
void PrtVec(const Vector<TYPE> &v)
{
  VecPtr<const TYPE> elemptr(v.PtrToAll());
  unsigned i, len = v.Length();
  for (i = 0; i<len; i++) {
      cout << *elemptr << ' ';
      elemptr++;
  }
  cout << '\n';
}

// BC++ needs the next statements

void PrtMat(const Matrix<int> &m);
void PrtVec(const Vector<int> &v);

// You need one of these statements for each type of vector:

INITNULLVEC(int);

#ifdef ONE_WAY
// A low-level C array we'll copy and transform into matrix
const int arr_size = 56;
int arr[arr_size];
#endif
     
void test()
{
  int i;

#ifdef ONE_WAY
  // Set up low level array to be transformed to a matrix
  for (i = 0; i<arr_size; i++) arr[i] = i;
#endif

  cout << "Creating a matrix\n";

#ifdef ONE_WAY
  Matrix<int> mymat(7, 8, arr); 
#else
  Matrix<int> mymat(7, 8);
  Vector<int> myvec(mymat.All());
  // Testing 1D vector subscripting too
  for(i = 0; i<myvec.Length(); i++) myvec[i] = i;
#endif
  cout << "mymat: \n"; 
  PrtMat(mymat); cout << '\n';

  cout << "Creating copy of the matrix\n";

#ifdef ONE_WAY
  Matrix<int> matii(mymat, COPIED);
#else
  /// ANOTHER WAY: Matrix<int> matii(mymat.Clone());
  Matrix<int> matii; // Create a null matrix
  matii = mymat.Clone();  
#endif
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';


  cout << "Creating a share of the diagonal\n";

  Vector<int> Diag = mymat.Diag();
  PrtVec(Diag);

  cout << "Setting shared diagonal = 1\n";
  Diag = 1;
  cout << "mymat: \n"; 
  PrtMat(mymat); cout << '\n';

  cout << "Creating a copy of the diagonal\n";

  Vector<int> Diag2 = mymat.Diag(COPIED);
  PrtVec(Diag2);

  cout << "Setting copied diagonal = 99\n";
  Diag2 = 99;
  PrtVec(Diag2); cout << '\n';
  cout << "mymat: \n"; 
  PrtMat(mymat); cout << '\n';


  cout << "Creating a share of row 3\n";
  Vector<int> Row = mymat.Row(3);
  PrtVec(Row);

  cout << "Setting row 3 to 17's\n";
  Row = 17;
  cout << "mymat: \n"; 
  PrtMat(mymat); cout << '\n';

  cout << "Creating a share of col 5\n";
  Vector<int> Col = mymat.Col(5);
  PrtVec(Col);

  cout << "Setting col 5 to 42's\n";
  Col = 42;
  cout << "mymat: \n"; 
  PrtMat(mymat);

  cout << "\nUsing the copy of the original matrix\n";
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Transposing this copy\n";
  matii = matii.Transpose();
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Set row 2 to 55's\n";
  matii.Row(2) = 55;
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Set col 3 to 77's\n";
  matii.Col(3) = 77;
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Creating shared submatrix of transpose\n";

  Matrix<int> submat(matii, SHARED, 2, 1, 3, 4);
  cout << "submat: \n"; 
  PrtMat(submat); cout << '\n';

  cout << "Transposing the submatrix\n";
  submat = submat.Transpose();
  cout << "submat: \n"; 
  PrtMat(submat); cout << '\n';

  cout << "\nTesting a[][] form of subscripting\n";

  cout << "submat[3][2] = " << submat[3][2] << "\n\n";

  cout << "Setting diagonal of submatrix to 0\n";
  submat.Diag() = 0;
  cout << "submat: \n"; 
  PrtMat(submat); cout << '\n';

  cout << "Setting second row of submatrix to 99's\n";
  submat.Row(1) = 99;
  cout << "submat: \n"; 
  PrtMat(submat); cout << '\n';

  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Creating sub-sub-matrix\n";

  Matrix<int> subsubmat(submat, SHARED, 1, 1, 3, 2);
  cout << "subsubmat: \n"; 
  PrtMat(subsubmat); cout << '\n';

  cout << "Set rows of subsubmatrix to 91's, 92's and 93's\n";

  subsubmat.Row(0) = 91;
  subsubmat.Row(1) = 92;
  subsubmat.Row(2) = 93;
  cout << "subsubmat: \n"; 
  PrtMat(subsubmat); cout << '\n';
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Set diagonal of subsubmatrix to 77's\n";

  subsubmat.Diag() = 77;
  cout << "subsubmat: \n"; 
  PrtMat(subsubmat); cout << '\n';
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';

  cout << "Testing sub-sub-matrix transpose\n";

  subsubmat = subsubmat.Transpose();
  cout << "subsubmat transposed: \n";
  PrtMat(subsubmat); cout << '\n';

  cout << "Set all elements of sub-sub-matrix to 42's\n";
  subsubmat = 42;
  cout << "subsubmat: \n"; 
  PrtMat(subsubmat); cout << '\n';
  cout << "matii: \n"; 
  PrtMat(matii); cout << '\n';
}

main()
{
  test();
  return 0;
}

