////////////////////////////////////////////////////////////
// matrix.h: Matrix class template.
// Copyright(c) 1993 Azarona Software. All rights reserved.
////////////////////////////////////////////////////////////
#ifndef H_MATRIX
#define H_MATRIX
#include "vector.h"

#define INLINE

template<class TYPE>
class Matrix {
protected:
  Vector<TYPE> data;
  unsigned nrows, ncols, rowstride, colstride;
public:
  Matrix(unsigned nr=0, unsigned nc=0, const TYPE *s = 0);
  Matrix(const Matrix<TYPE> &m);
  Matrix(const Matrix<TYPE> &m, SliceType styp,
         unsigned sr=0, unsigned sc=0, 
         unsigned nr=0, unsigned nc=0);
  void Copy(const Matrix<TYPE> &m);
  void Share(const Matrix<TYPE> &m);
  Matrix<TYPE> Transpose();
  Matrix<TYPE> &operator=(const Matrix<TYPE> &m);
  Matrix<TYPE> &operator=(const TYPE &x);
#ifndef NO_RANGE_CHECK
  unsigned CheckRow(unsigned i) const;
  unsigned CheckCol(unsigned i) const;
#endif
  Vector<TYPE> operator[](unsigned r);
  const Vector<TYPE> operator[](unsigned r) const;
  TYPE &operator()(unsigned r, unsigned c);
  const TYPE &operator()(unsigned r, unsigned c) const;
  Vector<TYPE> Row(unsigned r, SliceType styp=SHARED);
  Vector<TYPE> Col(unsigned c, SliceType styp=SHARED);
  Vector<TYPE> Diag(SliceType styp=SHARED);
  Vector<TYPE> All(SliceType styp=SHARED);
  const Vector<TYPE> Row(unsigned r, SliceType styp=SHARED) const;
  const Vector<TYPE> Col(unsigned c, SliceType styp=SHARED) const;
  const Vector<TYPE> Diag(SliceType styp=SHARED) const;
  const Vector<TYPE> All(SliceType styp=SHARED) const;
  // Mid-level hooks. Use these at your own risk:
  VecPtr<TYPE> RowPtr(unsigned r=0);
  VecPtr<TYPE> ColPtr(unsigned c=0);
  VecPtr<TYPE> DiagPtr();
  VecPtr<TYPE> PtrToAll();
  VecPtr<const TYPE> RowPtr(unsigned r=0) const;
  VecPtr<const TYPE> ColPtr(unsigned c=0) const;
  VecPtr<const TYPE> DiagPtr() const;
  VecPtr<const TYPE> PtrToAll() const;
  int IsNull() const;
  int IsUnique() const;
  Matrix<TYPE> Clone() const;
  int EnsureUnique();
  unsigned NCols() const;
  unsigned NRows() const;
  unsigned RowStride() const;
  unsigned ColStride() const;
  int IsRowMajor() const;
  int IsColMajor() const;
  int IsSquare() const;
};

template<class TYPE>
INLINE Matrix<TYPE> &Matrix<TYPE>::operator=(const Matrix<TYPE> &m)
// Share semantics used for assignment.
{
  if (this != &m) Share(m); // Note trap for assignment to self.
  return *this;
}

template<class TYPE>
INLINE Vector<TYPE> Matrix<TYPE>::operator[](unsigned r)
// Row subscripting operator for non-const matrices. 
// This routine does the same thing as Row(r,SHARED).
{
  return Vector<TYPE>(data, SHARED,
                      ncols, rowstride, CHECKROW(r)*colstride);
}

template<class TYPE>
INLINE const Vector<TYPE> Matrix<TYPE>::operator[](unsigned r) const
// Row subscripting operator for const matrices. 
// This routine does the same thing as Row(r,SHARED).
{
  return Vector<TYPE>(data, SHARED,
                      ncols, rowstride, CHECKROW(r)*colstride);
}

template<class TYPE>
INLINE VecPtr<TYPE> Matrix<TYPE>::RowPtr(unsigned r)
// Returns vector pointer to row r. No reference counting
// done. Use at your own risk.
{
  return VecPtr<TYPE>(data.start+CHECKROW(r)*colstride, rowstride);
}

template<class TYPE>
INLINE VecPtr<const TYPE> Matrix<TYPE>::RowPtr(unsigned r) const
// Returns vector pointer to row r. No reference counting
// done. Use at your own risk.
{
  return VecPtr<const TYPE>(data.start+CHECKROW(r)*colstride, rowstride);
}


template<class TYPE>
INLINE VecPtr<TYPE> Matrix<TYPE>::ColPtr(unsigned c)
// Returns vector pointer to column c. No reference counting
// done. Use at your own risk.
{
  return VecPtr<TYPE>(data.start+CHECKCOL(c)*rowstride, colstride);
}

template<class TYPE>
INLINE VecPtr<const TYPE> Matrix<TYPE>::ColPtr(unsigned c) const
// Returns vector pointer to column c. No reference counting
// done. Use at your own risk.
{
  return VecPtr<const TYPE>(data.start+CHECKCOL(c)*rowstride, colstride);
}

template<class TYPE>
INLINE VecPtr<TYPE> Matrix<TYPE>::DiagPtr()
// Returns vector pointer to right/downward diagonal.
{
  unsigned diastride = (rowstride == 1) ? colstride : rowstride;
  diastride++;
  return VecPtr<TYPE>(data.start, diastride);
}

template<class TYPE>
INLINE VecPtr<const TYPE> Matrix<TYPE>::DiagPtr() const
// Returns vector pointer to right/downward diagonal.
{
  unsigned diastride = (rowstride == 1) ? colstride : rowstride;
  diastride++;
  return VecPtr<const TYPE>(data.start, diastride);
}

template<class TYPE>
INLINE VecPtr<TYPE> Matrix<TYPE>::PtrToAll()
// Returns vector pointer to underlying 1D vector.
// No reference counting done, so use at your own risk.
{
  return VecPtr<TYPE>(data.start, 1);
}

template<class TYPE>
INLINE VecPtr<const TYPE> Matrix<TYPE>::PtrToAll() const
// Returns vector pointer to underlying 1D vector.
// No reference counting done, so use at your own risk.
{
  return VecPtr<const TYPE>(data.start, 1);
}

template<class TYPE>
INLINE int Matrix<TYPE>::IsNull() const
// Returns true if the matrix references null_rep.
{
  return data.IsNull();
}

template<class TYPE>
INLINE int Matrix<TYPE>::IsUnique() const
// Returns true of matrix has only reference to shared data.
{
  return data.IsUnique();
}

template<class TYPE>
INLINE unsigned Matrix<TYPE>::NCols() const
{
  return ncols;
}

template<class TYPE>
INLINE unsigned Matrix<TYPE>::NRows() const
{
  return nrows;
}

template<class TYPE>
INLINE unsigned Matrix<TYPE>::RowStride() const
// Returns the stride that a row vector would have
// in the matrix. For normal row-major matrices,
// the row stride will be 1. For shared submatrices, 
// the rowstride is that of the parent matrix.
{
  return rowstride;
}

template<class TYPE>
INLINE unsigned Matrix<TYPE>::ColStride() const
// Returns the stride that a column vector would have
// in the matrix. For a column-major matrix, the
// column stride will be 1. For shared submatrices.
// the colstride is that of the parent matrix.
{
  return colstride;
}

template<class TYPE>
INLINE int Matrix<TYPE>::IsRowMajor() const
{
  return rowstride == 1;
}

template<class TYPE>
INLINE int Matrix<TYPE>::IsColMajor() const
{
  return colstride == 1;
}

template<class TYPE>
INLINE int Matrix<TYPE>::IsSquare() const
{
  return nrows == ncols;
}

#undef INLINE

// Whether or not we should include the non-line methods for
// our class templates here is implementation dependent.

#include "matrix.mth"

#endif
