//////////////////////////////////////
//
// File:     bitmap.cpp
//
// Author:   Scott Wood, CCP
//
// Purpose:  Convert Windows RGB bitmap to Borland's putimage format
//
// Routines: ReadBitmap (const char *name)
//             Given a bitmap file, convert and return a putimage bitmap
//
//           DrawBitmap (BitMap *bm, int x, int y);
//             Display converted bitmap using putimage
//
//           DestroyBitmap (Bitmap *bm);
//             Delete unused bitmap
//
// Note: Program has only been tested using 16 color bitmaps.  It should
//        work for 256, but Borland's BGI routines will only work with
//        16 colors.
//////////////////////////////////////
#include <stdio.H>
#include <windows.h>
#include <graphics.h>
#include <io.h>
#include <fcntl.h>
#include <mem.h>
#include <bitmap.h>

static huge BYTE RGBMap[256][4];

// tempory storage space for reading bitmap.  defined large enough
// for a bit bitmap
//
// Note:  getimage and putimage have restrictions on the size of a bitmap
//        image.  Can be no larger than 32760.  about 256 x 256
//        this routine does not correct for this problem
//        a solution might be to sparse the image and correct during display.
//        For my purposes this was not necessary.

static huge BYTE data[4096];

BitMap _FAR *ReadBitmap (const char *name)
  {
    int chno = open (name, O_RDONLY | O_BINARY);
    if (chno < 0)
      return (void *)0;

    // bitmap File Header
    BITMAPFILEHEADER fileHdr;
    read (chno, &fileHdr, sizeof(fileHdr));

    // bitmap definition
    BITMAPINFOHEADER infoHdr;
    read (chno, &infoHdr, sizeof(infoHdr));

    // number of colors in bitmap
    int rgbcnt = (int)infoHdr.biClrUsed;
    int pixByte = 1;
    if (!rgbcnt)
      switch (infoHdr.biBitCount)
        {
          case 1: rgbcnt = 2;  pixByte = 8; break;
          case 4: rgbcnt = 16; pixByte = 2; break;
          case 8: rgbcnt = 256; pixByte = 1; break;
          default: rgbcnt = 0; pixByte = 1; break;
        };

    // RGB palette used to define bitmap
    read (chno, RGBMap, rgbcnt * sizeof(RGBQUAD));

    // first byte of bitmap image
    long offset = sizeof(BITMAPFILEHEADER) + infoHdr.biSize + rgbcnt * sizeof(RGBQUAD);

    // number of bytes stored per row.  make sure on even boundary
    int byteCnt = (int)infoHdr.biWidth / 8;
    if (infoHdr.biWidth != (infoHdr.biWidth / 8) * 8)
      byteCnt++;

    // number of pixels in image =  ByteCntPerRow * NumberOfPlanes * rows
    int pixels = byteCnt * 4 * (int)infoHdr.biHeight;

    // the following header is used for information only
    // bmBits is defined large enough to store pixel data
    int size = sizeof(BitMap) + pixels;

    BitMap *bm = (BitMap *)new char[size];
    memset (bm, 0, size);
    bm->bmType   = 0;
    bm->bmWidth  = (int)infoHdr.biWidth;
    bm->bmHeight = (int)infoHdr.biHeight;
    bm->bmWidthBytes = 1;
    bm->bmPlanes = 1;
    bm->bmBitsPixel = infoHdr.biBitCount;
    bm->bmImageSize = (int)infoHdr.biSizeImage;
    if (bm->bmImageSize == 0)
      bm->bmImageSize = bm->bmWidth * bm->bmHeight;

    // number of bytes per row in bitmap
    int div = bm->bmImageSize / bm->bmHeight;

    // header for image
    char *bmBits = bm->bmBits;
    *(int *)bmBits = (int)infoHdr.biWidth - 1;
    bmBits += 2;
    *(int *)bmBits = (int)infoHdr.biHeight - 1;
    bmBits += 2;

    // do for each row
    for (int row=0; row < infoHdr.biHeight; row++)
      {
        // read rows worth of data, start at bottom
        lseek (chno, offset + ((infoHdr.biHeight - 1) - row) * div, 0);
        read (chno, data, (int)infoHdr.biWidth / pixByte + 1);

        // calculate offset into new bitmap.  Starts on word boundary
        int rowOffset = row * byteCnt * 4;

        // for each column upto width size
        for (int column=0; column < div; column++)
          {
            // four planes defined, 0-intensity, 1-red, 2-green, 3-blue
            int colBit = column * pixByte;

            // for each word, get each element, 256=1, 16=2, 1=8
            for (int pixno=0; pixno < pixByte; pixno++)
              {
                int colIdx = colBit + pixno;

                // calculate index into RGBMap. Bitmaps are index into this table
                unsigned idx = (data[column] >> ((pixByte - 1 - pixno) * bm->bmBitsPixel)) & (rgbcnt - 1);

                // calculate OR bit for color definition
                BYTE bit = 0x80 >> (colIdx % 8);

                // calculate byte within row
                int colOffset = rowOffset + colIdx / 8;

                // if color is > 8 set intensity
                if (idx >= 8)
                  bmBits[colOffset] |= bit;

                // do for Red, Green, Blue unless DarkGray
                if (idx != 8)
                  for (int plane=1; plane < 4; plane++)
                    {
                      // if respective color defined in map, set color in image
                      if (RGBMap[idx][3-plane])
                        bmBits[colOffset + byteCnt * plane] |= bit;
                    };
              };
          };
      };

    close (chno);
    
    return bm;
  };

void DrawBitmap (BitMap _FAR *bm, int x, int y)
  {
    putimage (x, y, bm->bmBits, COPY_PUT);
  };

void DestroyBitmap (BitMap _FAR *bm)
  {
    if (!bm)
      return;

    delete bm;
  };

