/*
    clip.c

    line-clipping routines using 
    Cohen-Sutherland clipping algorithm
*/

#include "defs.h"

#define BIT1    1
#define BIT2    2
#define BIT3    4
#define BIT4    8

/* result of line clip */
int rx1, ry1, rx2, ry2;

/*
    clip codes for line endpoints

                |           |
        1001    |   1000    |   1010
                |           |
    -------------------------------------
                |           |
        0001    |   0000    |   0010
                |           |
    -------------------------------------
                |           |
        0101    |   0100    |   0110
                |           |
*/

int clip_code(x,y, window)
int     x, y;
rect_t  *window;
{
int code = 0;

    if(x < window->left)
        code += BIT1;
    else if(x > window->right)
        code += BIT2;

    if(y < window->bottom)
        code += BIT3;
    else if(y > window->top)
        code += BIT4;

    return(code);
}

/*  trivial rejection check */

BOOLEAN reject_check(code1, code2)
int code1, code2;
{
    if(code1 & code2)
        return(TRUE);
    else
        return(FALSE);
}

/* trivial acceptance check */

BOOLEAN accept_check(code1, code2)
int code1, code2;
{
    if( ! (code1 | code2) )
        return(TRUE);
    else
        return(FALSE);
}

/* clip routine */

BOOLEAN clipper(x1, y1, x2, y2, window)
int x1, y1, x2, y2;
rect_t  *window;
{
int     code1, code2 ,done, accept, temp, swaps;
double  n, d;

    swaps = 0;
    done = accept = FALSE;

    while(done == FALSE)
        {
        code1 = clip_code(x1, y1, window);
        code2 = clip_code(x2, y2, window);

        if(reject_check(code1, code2))
            done = TRUE;
        else
            {
            accept = accept_check(code1, code2);
            if(accept)
                done = TRUE;
            else
                /* subdivide line because one
                   point at most is inside window */
                {
                if(code1 == 0)
                    {   /* make point 1 outside */
                    swaps++;
                    temp = x1;      x1 = x2;        x2 = temp;
                    temp = y1;      y1 = y2;        y2 = temp;
                    temp = code1;   code1 = code2;  code2 = temp;
                    }

                if(code1 & BIT4)    /* divide at top of window */
                    {
                    n = (double) ((x2 - x1) * (window->top - y1));
                    d = (double) (y2 - y1);

                    x1 = x1 + (int) (n / d);
                    y1 = window->top;
                    }
                else if(code1 & BIT3)   /* divide at bottom of window */
                    {
                    n = (double) ((x2 - x1) * (window->bottom - y1));
                    d = (double) (y2 - y1);

                    x1 = x1 + (int) (n / d);
                    y1 = window->bottom;
                    }

                else if(code1 & BIT2)    /* divide at right of window */
                    {
                    n = (double) ((y2 - y1) * (window->right - x1));
                    d = (double) (x2 - x1);

                    y1 = y1 + (int) (n / d);
                    x1 = window->right;
                    }
                else if(code1 & BIT1)   /* divide at left of window */
                    {
                    n = (double) ((y2 - y1) * (window->left - x1));
                    d = (double) (x2 - x1);

                    y1 = y1 + (int) (n / d);
                    x1 = window->left;
                    }
                }
            }   /* end of no reject */
        }   /* end of while */

    if(accept)
        {
        if(swaps & 1) /* swap points back to original order */
            {
            rx1 = x2;   rx2 = x1;
            ry1 = y2;   ry2 = y1;
            }
        else
            {
            rx1 = x1;   rx2 = x2;
            ry1 = y1;   ry2 = y2;
            }
        return(TRUE);
        }
    else
        return(FALSE);
}
