//============================================================================
// Class: 	BCD
// Version: 1.0
// Files: 	bcd.cpp bcd.hpp util.hpp x_range.hpp xception.hpp
// ------------------------------------------------------------------------
// Description: A C++ class that implements Binary Coded Decimal (BCD).
// 				 Representing floating point values in a BCD representation
// 				 alleviates the inaccuracies inherent in a binary 
//					 representation of real numbers.
//============================================================================
// DISCLAIMER
// 								BCD C++ Class
// 						Copyright 1994 Troy Hakala
// 
// This class is shareware.  If you use this class, a $20 (or whatever you 
// think is fair) contribution would be greatly appreciated.  You are free to 
// redistribute this class in any format as long as the this disclaimer
// remains intact and the source code is not modified in any way.
//============================================================================
//
//Please send contributions to the following address:
//
//					Troy Hakala
//					46702 Sandalwood Square
//					Sterling VA 20164
//
//Please send any/all suggestions/comments/criticisms to one of the 
//following email addresses:
//	
//					Internet: troy@digex.net
//					CompuServe: 70254,3072
//
//============================================================================

#include "bcd.hpp"
#include "util.hpp"

ostream& operator<<(ostream& os, const BCD& bcd)
{
#if defined(DEBUG)
	for (unsigned int xx = 0; xx < sizeof(bcd._val); ++xx)
		cout << hex << setw(2) << setfill('0') << (int)bcd._val[xx];
	cout << dec << '(';
#endif

	unsigned int byte_with_decimal = sizeof(bcd._val)-((bcd._dec+1)/2);

	// skip all leading zeroes
	for (unsigned int i(0); i < ((bcd.mantissa_space()-1)/2) && (bcd._val[i] == 0);)
		++i;

	if (bcd._sign == BCD::NEGATIVE)
		os << '-';

	for (; i < sizeof(bcd._val); ++i)
	{
		if (i == byte_with_decimal)
		{
			if (isodd(bcd._dec))
				os << setw(1) << setfill(' ') << (int)(bcd._val[i] >> 4) << '.' << (int)(bcd._val[i] & 0x0f);
			else
				os << hex << (int)bcd._val[i] << '.';
			os << setw(2) << setfill('0');
		}
		else
			os << hex << (int)bcd._val[i] << setw(2) << setfill('0');
	}
	os << setfill(' ') << setw(1) << dec;
#if defined(DEBUG)
	cout << ')';
#endif
	return(os);
}

BCD::BCD(int n, unsigned int decs)
	: _dec(decs), _sign(n >= 0 ? POSITIVE : NEGATIVE)
{
	memset(_val, 0, sizeof(_val));
	for (int i = sizeof(_val); i && n; --i)
	{
		_val[i-1] = int2bcd(n % 100);
		n /= 100;
	}
}

BCD::BCD(const char* strval, unsigned int decs)
	: _dec(decs), _sign(*strval == '-' ? NEGATIVE : POSITIVE)
{
	memset(_val, 0, sizeof(_val));

	const char* decpt = strchr(strval, '.');
	if (!decpt)
	{
		if (!*strval)
			decpt = strval;	// point to last digit (of empty string)
		else
			decpt = &strval[strlen(strval)];	// point to last digit
	}

	unsigned int mantissa_len = decpt - strval;
	unsigned int i = sizeof(_val) - ((_dec + mantissa_len + 1) / 2);
	// i is the index into the _val array of the byte that contains 
	// the decimal point.

	int hi_byte;
	if (isodd(_dec))
	{
		if (isodd(mantissa_len))
			hi_byte = 1;
		else
			hi_byte = 0;
	}
	else
	{
		if (isodd(mantissa_len))
			hi_byte = 0;
		else
			hi_byte = 1;
	}

	int decimal_found = 0;
					
	for (const char* p = strval; *p && (isdigit(*p) || *p=='.'); ++p)
	{
		if (i >= sizeof(_val))
			break;
		if (*p == '.')
		{
			decimal_found = 1;
			continue;
		}
		if (hi_byte)
			_val[i] |= (*p & 0x0f) << 4;
		else
			_val[i++] |= *p & 0x0f;
		hi_byte = !hi_byte;
	}
	if (*p)
	{
		// this means that the string is longer than we will can fit!
		// what to do?  If we're in the integer part of the value,
		// throw an exception (OVERFLOW).  If we're in the fractional
		// part, round the last digit.
		if (!decimal_found)
#if !defined(__WATCOMC__)			
			return;
#else
			throw(RangeException(RangeException::OVERFLOW, __FILE__,__LINE__));
#endif

		if (*p >= '5')
		{
			// round it up!
			int carry = 1;
			for (unsigned int round_index = sizeof(_val); carry && round_index; --round_index)
			{
				unsigned int sum = bcd2int(_val[round_index-1]) + carry;
				carry = sum > 99;
				sum %= 100;
				_val[round_index-1] = int2bcd(sum);
			}
		}
	}

}

BCD& BCD::operator=(const BCD& b)
{
	memcpy(_val, b._val, sizeof(_val));
	_dec = b._dec;
	_sign = b._sign;
	return *this;
}

char BCD::int2bcd(unsigned int n)
{
	if (n > 99)
#if !defined(__WATCOMC__)			
		return(0);
#else
		throw(RangeException(RangeException::OVERFLOW, __FILE__, __LINE__));
#endif

	char c = (n / 10) << 4;
	c |= n % 10;
	return(c);
}

unsigned int BCD::bcd2int(char c)
{
	return((((c & 0xf0) >> 4) * 10) + (c & 0x0f));
}

unsigned int BCD::operand(const BCD& b, int pos)
{
	unsigned int byte_with_decimal = sizeof(b._val) - ((b._dec+1) / 2);
	unsigned int i = byte_with_decimal;
	int split;

#if defined(DEBUG)
	cout << "requesting pos:" << pos;
#endif

	int upper_bound = (sizeof(b._val) * 2) - b._dec - 1;
	int lower_bound = -b._dec;

	if (pos > upper_bound || lower_bound > pos)
#if !defined(__WATCOMC__)
		return 0;
#else
		throw(RangeException(RangeException::OVERFLOW, __FILE__, __LINE__));
#endif

	// we're trying to map 'pos' to the actual byte in the _val array.

	// byte_with_decimal is either the byte before the decimal point
	// or contains the decimal point.
	if (isodd(b._dec))
	{
		// the decimal point is between the two digits 
		// in _val[byte_with_decimal]
		split = 1;
	}
	else
	{
		// the decimal point is after the two digits 
		// in _val[byte_with_decimal]
		split = 0;
	}

	unsigned int value = 0;
	if (split)
	{
		if (pos < 0)
		{
		 	i -= pos / 2;
			if (isodd(pos))
				value = bcd2int(b._val[i] & 0x0f);
			else
				value = bcd2int(b._val[i] >> 4);
		}
		else
		{
		 	i -= (pos+1) / 2;
			if (isodd(pos))
				value = bcd2int(b._val[i] & 0x0f);
			else
				value = bcd2int(b._val[i] >> 4);
		}
	}
	else
	{
		i -= ((pos-1) / 2);
		if (pos < 0)
		{
			if (isodd(pos))
				value = bcd2int(b._val[i] >> 4);
			else
				value = bcd2int(b._val[i] & 0x0f);
		}
		else
		{
			if (isodd(pos))
				value = bcd2int(b._val[i] >> 4);
			else
				value = bcd2int(b._val[i] & 0x0f);
		}
	}

#if defined(DEBUG)
	cout << " -> " << value << endl;
#endif

	return(value);
}

unsigned int BCD::integer_value_length() const
{
	for (unsigned int i = mantissa_space(); i; )
	{
		if (operand(*this, --i))
			break;
	}
	return(i+1);
}

unsigned int BCD::fractional_value_length() const
{
	for (int i = -1 * fractional_space(); i; ++i)
	{
		if (operand(*this, i))
			break;
	}
	return(abs(i));
}

BCD& BCD::operator+=(const BCD& b)
{
	*this = *this + b;
	return(*this);
}

BCD& BCD::operator-=(const BCD& b)
{
	*this = *this - b;
	return(*this);
}

BCD  BCD::operator+(const BCD& b) const
{
	unsigned int m = maximum(_dec, b._dec);
	BCD sum(0, m);
	int i = -m;
	int carry = 0;
	int operand_sum = 0;
	for (unsigned int n = sizeof(sum._val); n;)
	{
		int x = (operand(*this, i+1) * 10) + operand(*this, i);
		int y = (operand(b, i+1) * 10) + operand(b, i);

#if defined(DEBUG)
		cout << "adding " << x << " and " << y << " w/ carry:" << carry;
#endif
		operand_sum = x + y + carry;
#if defined(DEBUG)
		cout << " = " << operand_sum << endl;
#endif

		carry = operand_sum >= 100;
		operand_sum %= 100;

		--n;
		sum._val[n] = int2bcd(operand_sum);
#if defined(DEBUG)
		cout << "Storing:" << hex << setw(2) << setfill('0') << (int)sum._val[n] << dec << endl;
#endif
		i += 2;
	}
	return(sum);
}

BCD  BCD::operator-(const BCD& b) const
{
	unsigned int m = maximum(_dec, b._dec);
	BCD diff(0, m);

	if (*this < b)
	{
		BCD b_copy = b;
		diff = b_copy - *this;
		diff._sign = NEGATIVE;
		return(diff);
	}

	int i = -m;
	int borrow = 0;
	int operand_diff = 0;
	for (unsigned int n = sizeof(diff._val); n;)
	{
		int x = (operand(*this, i+1)*10) + operand(*this, i);
		int y = (operand(b, i+1) * 10) + operand(b, i);
#if defined(DEBUG)
		cout << "x=" << x << ", y=" << y << endl;
#endif
		if (borrow)
		{
#if defined(DEBUG)
			cout << "accounting for previous borrow" << endl;
#endif			
			x -= 1;
			borrow = 0;
		}
		if (x < y)
		{
#if defined(DEBUG)			
			cout << "borrowing" << endl;
#endif
			borrow = 1;
			x += 100;
		}
#if defined(DEBUG)
		cout << "subtracting: " << x << " - " << y << endl;
#endif
		operand_diff = x - y;
#if defined(DEBUG)
		cout << "storing: " << operand_diff << endl;
#endif
		diff._val[--n] = int2bcd(operand_diff);
		i += 2;
	}
//	if (borrow)
//		diff._sign = NEGATIVE;
//	else
//		diff._sign = POSITIVE;
	return(diff);
}

int BCD::operator<(const BCD& b) const 
{
	if (_sign != b._sign)
		return(_sign == NEGATIVE);
	if (integer_value_length() != b.integer_value_length())
	{
		if (_sign == POSITIVE)
			return(integer_value_length() < b.integer_value_length());
		else
			return(integer_value_length() > b.integer_value_length());
	}
	for (unsigned int i = integer_value_length(); i; )
	{
		--i;
		unsigned int x = operand(*this, i);
		unsigned int y = operand(b, i);
		if (x != y)
			return(x < y);
	}
	// at this point, everything except for the fractional part is equal.
	// so, check the fractional part
	for (i = 1; i <= minimum(fractional_value_length(), b.fractional_value_length()); i++)
	{
		unsigned int x = operand(*this, -i);
		unsigned int y = operand(b, -i);
		if (x != y)
		{
			if (_sign == POSITIVE)
				return(x < y);
			else
				return(x > y);
		}
	}
	// now, the entire numbers match except for trailing fractional
	// digits at one of the numbers.  All we have to do now is determine
	// which BCD number has that trailing digit and that number is
	// either less than or greater than the other, depending on the
	// positivity or negativity of their (equal) signs.
	if (fractional_value_length() < b.fractional_value_length())
		return(_sign == POSITIVE);
	return(_sign == NEGATIVE);
}

int BCD::operator!=(const BCD& b) const 
{
	int max_frac_len = maximum(fractional_value_length(), b.fractional_value_length());
	int max_int_len  = maximum(integer_value_length(), b.integer_value_length());
	for (int i = -1 * max_frac_len; i < max_int_len; ++i)
	{
		if (operand(*this, i) != operand(b, i))
			return(1);
	}
	return(0);
}



#if defined(TEST)

int main(int argc, char* argv[])
{
	if (argc != 3)
	{
		cerr << "Usage: " << argv[0] << " <operand 1> <operand 2>" << endl;
		return(1);
	}

#if defined(__WATCOMC__)
	try
	{
#endif
		BCD x(argv[1],5);
		BCD y(argv[2],5);
	
//		BCD sum = x + y;
		BCD diff = x - y;

		cout << "   :----+----|----+----|----+----|----+----|----+----|----+----|----+----|" << endl;
		cout << "  x:" << x << endl;
		cout << "  y:" << y << endl;
//		cout << "Sum:" << sum << endl;
		cout << "Dif:" << diff << endl;
		cout << "x < y : " << (x <  y) << endl;
		cout << "x > y : " << (x >  y) << endl;
		cout << "x <= y: " << (x <= y) << endl;
		cout << "x >= y: " << (x >= y) << endl;
		cout << "x == y: " << (x == y) << endl;
		cout << "x != y: " << (x != y) << endl;
#if defined(__WATCOMC__)
	}
	catch(Exception x)
	{
		cout << x << endl;
	}
	catch(...)
	{
		cout << "Uncaught Exception!" << endl;
	}
#endif

	return(0);
}

#endif
