#ifndef     __async_h
#define     __async_h

#define     NUMBEROFPORTS   4       // Number of COM ports available
                                    // If this number is changed, and you plan
                                    // on opening a com port greater than 4
                                    // then you must create an ISR for it.
                                    // See UART_ISR1..4 and you shouldn't have
                                    // any problems.

#define     COM1            0       // COM port 1
#define     COM2            1       // COM port 2
#define     COM3            2       // COM port 3
#define     COM4            3       // COM port 4

#define     UART_NONE       0       // No UART attached
#define     UART_8250       1       // UART is an 8250
#define     UART_16460      2       // UART is a 16460
#define     UART_16550      3       // UART is a 16550
#define     UART_16550A     4       // UART is a 16550A

#define     HSHAKE_NONE     0       // Handshaking:  None
#define     HSHAKE_XONXOFF  1       // Handshaking:  XON/XOFF

#define     TXBUFSIZE       2048    // Transmit Buffer Size
#define     RXBUFSIZE       2048    // Receive Buffer Size

#define     BASE            0       // Base (No real reason to have this)
#define     IER             1       // Interrupt Enable Register
#define     IIR             2       // Interrupt Identification Register
#define     FCR             2       // 16550 FIFO Control Register
#define     LCR             3       // Line Control Register
#define     MCR             4       // Modem Control Register
#define     LSR             5       // Line Status Register
#define     MSR             6       // Modem Status Register
#define     SCRATCH         7       // Scratch Pad (Is this ever used?)

#define     MCR_DTR         0x01    // DTR
#define     MCR_RTS         0x02    // RTS
#define     MCR_OUT1        0x04    // OUT1
#define     MCR_OUT2        0x08    // OUT2
#define     MCR_LOOPBACK    0x10    // Loopback

#define     LCR_PNONE       0       // None
#define     LCR_PODD        8       // Odd
#define     LCR_PEVEN       24      // Even
#define     LCR_PMARK       40      // Mark
#define     LCR_PSPACE      56      // Space

struct  async_portS {
    int     port_open;

    void    interrupt (*oldISR)();

    char    *txbuf;
    int     txhead, txtail;
    int     txbuflength;

    char    *rxbuf;
    int     rxhead, rxtail;
    int     rxbuflength;

    int     overrun_errors;
    int     parity_errors;
    int     framing_errors;

    int     uart_type;

    int     handshaking;                    // Not yet implemented
};

// setuart.c        -- UART setup routines
void    set_UART_port(int comport, int value);
void    set_UART_int(int comport, int value);
void    set_UART_onmask(int comport, int value);
void    set_UART_offmask(int comport, int value);
int     set_parity(int comport, int parity);
int     set_stopbits(int comport, int stopbits);
int     set_wordlength(int comport, int wordlength);
int     set_baudrate(int comport, long baudrate);
void    set_handshaking(int comport, int status);

// detect.c         -- Various detecting routines: needs work
int     async_detect_uart(int comport);
int     async_detect_irq(int comport);

// isr.c            -- ISR setup routines, and the actual ISR's themselves
void    init_ISR(int comport);
void    deinit_ISR(int comport);

// fifo.c           -- FIFO routines: needs work
int     init_fifo(int comport, int enable);

// async.c          -- Async setup routines
int     open_port(int comport, long baudrate, int parity, int stopbits, int wordlength);
int     close_port(int comport, int rts, int dtr);

// line.c           -- Line control/monitoring routines: needs work
int     async_LCR(int comport);
int     async_MCR(int comport);
int     async_LSR(int comport);
int     async_MSR(int comport);
int     async_DTR_status(int comport);
int     async_RTS_status(int comport);
void    async_set_DTR(int comport, int status);
void    async_set_RTS(int comport, int status);

// asyio.c          -- IO/Buffer routines: needs work
int     async_putch_timeout(int comport, char c, long timeout);
void    async_putch(int comport, char c);
void    async_puts(int comport, char *s);
int     async_putblock_timeout(int comport, const char *block, int size, long timeout);
void    async_putblock(int comport, const char *block, int size);
int     async_ready(int comport);
char    async_getch(int comport);
int     async_getblock(int comport, char *block, int size);
char    async_peek(int comport);
void    async_flushrx(int comport);
void    async_flushtx(int comport);

#endif
