* Author.: Joseph D. Booth
* Company: Precision Engineered Parts, Inc
*          610 Lakeside Drive
*          Southhampton, PA 18966
*
* W Phone: 215-953-1202    (8:30-5:30 E.S.T)
*
* H Phone: 215-245-6774
*
* Purpose: These functions may be used to compute UPS shipping charges for a
*          selected weight.  There are two functions:
*
*           UPSCOST() - which returns the shipping cost for a given weight
*
*                       and
*
*           UPSZONE() - which returns the UPS zone code
*
*
* Syntax.: UPSCOST( <expN1> , <expC1> [, <expN2> ] )
* Where..:
*          <expN1>     the weight of the item to be shipped
*
*          <expC1>     the string   UPS   - for standard UPS charges
*                             or    BLUE  - for second day charges
*                             or    RED   - for overnight charges
*
*          <expN2>     the ups zone digit 2 thru 8, as an INTEGER
*
*
* Returns:           Real number representing shipping cost
*           0.0      An error occurred, or UPS cannot ship this item
*
*
* Syntax.: UPSZONE( <expC1> , <expC2> )
* Where..:
*          <expC1>     the zip code where the item should be shipped to
*
*          <expC2>     the zip code where the item should be shipped from
*
*
* Returns:           Single digit, representing the UPS zone
*             0      An error occurred, or the zone could not be determined
*
*
*
* Notes..:  These functions require that two databases be set up.  The first
*           database is UPSCHART, which contains the UPS charges table.  The
*           second is UPSXXX, which represents the UPS zone chart, and XXX
*           is the first three numbers of the zip code which the program is
*           running from.
*
*           Structure for database:    UPSCHART
*           Number of data records:          70
*
*           Field  Field Name  Type       Width    Dec
*               1  RED         Numeric        6      2
*               2  BLUE        Numeric        6      2
*               3  COST02      Numeric        6      2
*               4  COST03      Numeric        6      2
*               5  COST04      Numeric        6      2
*               6  COST05      Numeric        6      2
*               7  COST06      Numeric        6      2
*               8  COST07      Numeric        6      2
*               9  COST08      Numeric        6      2
*           ** Total **                      55
*
*
*           Structure for database:    UPSXXX
*           Number of data records:       105
*
*           Field  Field Name  Type       Width    Dec
*               1  ZIPHI       Numeric        3
*               2  UPS_ZONE    Numeric        1
*           ** Total **                       5
*
*
*           My zip code is 18966, so I have included UPS189.DBF and UPSCHART
*           with the program.
*
*
* If you find these functions useful, a $10 contribution is requested.  This
* fee entitles you to support about the functions, and creating the UPS ZONE
* charts.  I have several other zone charts for different areas of the country
* or you may create your own.
*
*****************************************************************************
FUNCTION upscost
*
* Return the UPS shipping cost
*
PARAMETER u_weight,u_ups,u_zone
PRIVATE   u_oldarea,u_retcost,u_tmp

u_weight   = INT(u_weight+.5)             && Express weight as an INTEGER and
u_retcost  = 0.0                          && round it up to next weight
u_oldarea  = STR(select(),2)              && Save the current work area

IF u_weight > 0 .AND. u_weight <= 70      && Are we with UPS weight allowances
   IF FILE("upschart.dbf")
      SELECT 0
      USE upschart                        && Use the UPSCHART database
      GOTO u_weight                       && Go to the selected wieght
      IF .NOT. eof()
         DO CASE
            CASE u_ups = "RED"            && Overnight RED label charges
                 u_retcost = red
            CASE u_ups = "BLUE"           && Second day charges
                 u_retcost = blue
            OTHERWISE
                 IF u_zone > 1 .AND. u_zone < 9
                    u_tmp     = "COST0"+STR(u_zone,1)
                    u_retcost = &u_tmp
                 ENDIF
         ENDCASE
      ENDIF
   ENDIF
ENDIF

SELECT &u_oldarea
RETURN u_retcost

* ---------------------------------------------------------------------------

FUNCTION upszone
*
* Return the UPS zone code, based on zip code
*
PARAMETER u_zipto,u_zipfrom
PRIVATE u_three,u_return,u_oldalias,u_file,u_ntx

u_three    = VAL(SUBSTR(u_zipto,1,3))
u_return   = 0
u_oldalias = STR( select(),2 )
u_file     = "UPS"+SUBSTR(u_zipfrom,1,3)+".DBF"
u_ntx      = "UPS"+SUBSTR(u_zipfrom,1,3)+".NTX"

IF FILE(u_file)

   SELECT 0
   USE &u_file
   IF .NOT. FILE(u_ntx)
      INDEX ON ziphi TO &u_ntx
   ENDIF
   SET INDEX TO &u_ntx

   IF fndrange(u_three)
      u_return = ups_zone
   ENDIF
   USE

ENDIF

SELECT &u_oldalias
RETURN u_return

* ---------------------------------------------------------------------------


FUNCTION fndrange
*
* Used internally by the UPSZONE function to locate the proper zip range
*
PARAMETER u_zipcod
PRIVATE u_find

u_find = .F.

DO WHILE .NOT. u_find
   SEEK u_zipcod
   IF .NOT. eof()
      u_find = .T.
   ELSE
      u_zipcod = u_zipcod +1
      IF u_zipcod > 999
         EXIT
      ENDIF
   ENDIF
ENDDO

RETURN u_find

* ---------------------------------------------------------------------------




