/****************************************************
* UserFilt - Allow the user to create a filter
*
* This function allows the end-user to create their own
* filter condition.  If the filter is valid, it is set
* on in the previously SELECTed data area.  No record
* movement is performed by this function.
*
* Compile with /n/w
*
******************************************************/

#include "inkey.ch"

FUNCTION UserFilt ()

   // This function sets or clear a user defined filter
   // expression.  It assumes that lines 23 & 24 can be
   // used for function communications (input or output).

   LOCAL cFilter
   LOCAL bFilter
   LOCAL bLastError
   LOCAL oErr

   @ 23, 00 SAY 'Please enter Filter to use ' + ;
               '(Example: COMPANY = "NANTUCKET")'

   // Install an error handler for testing below
   bLastError := ERRORBLOCK ({|oE| BREAK (oE)})

   DO WHILE .T.
      // Get existing filter
      cFilter := PADR (DBFILTER (), 255)

      // Ask user for a new filter expression
      @ 24, 00 GET cFilter PICTURE '@KS80'

      SET CURSOR ON
      READ
      SET CURSOR OFF

      IF LASTKEY () == K_ESC
         EXIT
      ENDIF

      IF EMPTY (cFilter)
         cFilter := ""
         DBCLEARFILTER ()
         EXIT
      ENDIF

      BEGIN SEQUENCE
         // First trim the character version of the filter
         // expression Then use macro expansion with a
         // "{||" prefix, and a "}" suffix to convert 
         // the character expression to a code block

         cFilter := ALLTRIM (cFilter)
         bFilter := &("{||" + cFilter + "}")

         // Now let's check to see if the code block
         // EVALs to a logical
         IF VALTYPE (EVAL (bFilter)) != "L"
            // If not a logical, break without passing
            // an error object
            BREAK
         ENDIF

         // Got a good filter, set it
         DBSETFILTER (bFilter, cFilter)

      RECOVER USING oErr

         // If we got an error object, then the macro expansion
         // or the EVAL had a problem. If no error object was
         // passed, then the filter expression does not return
         // a logical value.

         IF VALTYPE (oErr) == "O"
            ALERT ("Error - Invalid filter expression; " + ;
                  oErr:description)
         ELSE
            ALERT ("Filter expression does not evaluate " + ;
                  "to a True or False")
         ENDIF
         LOOP
      END SEQUENCE

      EXIT
   ENDDO

   // Restore the previous error handler
   ERRORBLOCK (bLastError)

   // Clear the communication area
   SCROLL (23, 00, 24, 79, 0)

   // Return the filter selection
   // (It will be upto the calling module to re-position)
   RETURN (cFilter)

// EOF UserFilt.PRG
