/****************************************************
* Messaging System Functions
*
* These functions provide a rudimentary messaging
* system for use with screen and validation functions.
*
* Compile with /n/w
*
******************************************************/

// Define parameters for each array element

#define aMsgLn   aMessage [1]
#define aMsgStd  aMessage [2]
#define aMsgErr  aMessage [3]
#define aMsgCtr  aMessage [4]
#define aMsgTxt  aMessage [5]

// Define the size of the message storage array

#define aMsgSize 5

STATIC aMessage := {24, "W/N", "R/N", .T., " "}
STATIC nStdPos  := 0

/**********************************************************/
//
// MsgSetup
//
// Initialize the message environment for the calling program.
// Returns the previous enviroment as the funcation return 
// (to be used by MSG_REST to restore the old environment).
// 
// Parameters: 
//   1) Message display line                    Numeric
//   2) Message standard color                  Character 
//   3) Message error color                     Character 
//   4) Message centered flag                   Logical 
//   5) Standard message text                   Character 

FUNCTION MsgSetup (nLn, cStd, cErr, lCtr, cTxt)

   LOCAL aOldMsg := ACOPY (aMessage)       // Save old env

   // Setup the message environment, default if not specified

   aMsgLn  := IIF (VALTYPE (nLn)  == "N", nLn,  24)
   aMsgStd := IIF (VALTYPE (cStd) == "C", cStd, "W/N")
   aMsgErr := IIF (VALTYPE (cErr) == "C", cErr, "R/N")
   aMsgCtr := IIF (VALTYPE (lCtr) == "L", lCtr, .T.)
   aMsgTxt := IIF (VALTYPE (cTxt) == "C", cTxt, " ")

   RETURN (aOldMsg)

/**********************************************************/
//
// MsgRest
//
// Restores a message environment using the function return 
// from the MsgInit function. 

FUNCTION MsgRest (aMsg)

   LOCAL i

   // Parse the message environment array and reset the
   // message components accordingly.

   IF VALTYPE (aMsg) == "A"
      i := 1
      AEVAL (aMsg, {|aM| aMessage [i++] := aM})
   ENDIF

   // Center message?
   IF aMsgCtr
      nStdPos := 
   RETURN (NIL)

/**********************************************************/
//
// MsgText
//
// Reset the standard message text, returning the previous 
// standard text. 

FUNCTION MsgText (cTxt)

   LOCAL cOldText := aMsgTxt

   IF VALTYPE (cTxt) == "C"
      aMsgTxt := cTxt
   ENDIF

   RETURN cOldText

/**********************************************************/
//
// MsgSay (cErr, lWait)
//
// Output a message, based on parameters passed.
//
// Parameters:
//   1) Error message          Character       (Optional)
//   2) Wait flag              Logical         (Optional)

FUNCTION MsgSay (cErr, lWait)

   LOCAL cColor, cText, lError

   cErr  := IIF (VALTYPE (cErr)  == "C", cErr,  "")
   lWait := IIF (VALTYPE (lWait) == "L", lWait, .F.)

   // If there is no error message use the standard message 
   // text, otherwise use the error text supplied.
   // Set the display color and error flag accordingly. 

   IF EMPTY (p1)
      lError := .T.
      cText  := aMsgTxt
      cColor := aMsgStd
   ELSE
      lError := .T.
      cText  := cErr
      cColor := aMsgErr
   ENDIF

   // Trim the message text. If a wait state is to be 
   // invoked, append the text "Press any key". 

   cText := ALLTRIM (cText)
   IF lWait
      cText += " Press any key."
   ENDIF

   // If we are centering the message, pad the text out to 
   // the screen width. Otherwise, clear the line.

   IF aMsgCtr
      cText := PADC (cText, (MAXCOL () + 1))
   ELSE
      SCROLL (aMsgLn, 00, aMsgLn, MAXCOL (), 0)
   ENDIF

   // Output the message text. If outputting the error text, 
   // also ring the bell. 

   IF lError
      DEVPOS (aMsgLn, 00)
      ?? CHR (7)
   ENDIF

   @ aMsgLn, 00 SAY cText COLOR cColor

   // Invoke a wait state? If yes, call MsgInkey (0).

   IF lWait
      MsgInkey (0)
   ENDIF

   // If we output the standard text, return a true (.T.)
   // If we output the error text, return a false (.F.)

   RETURN !lError

/*************************************************************/
//
// MsgInkey (<seconds>)
//
// Returns integer numeric value of key pressed.
//
// <seconds> parameter is same as CA-Clipper INKEY()
//
// Emulates CA-Clipper INKEY() except this function
// will run any SETKEY defined.
//

FUNCTION MsgInkey (nSec)

   LOCAL nKey, bKey

   DO WHILE .T.
      IF VALTYPE (nSec) != "N"
         nKey := INKEY ()
      ELSE
         nKey := INKEY (nSec)
      ENDIF

      IF (bKey := SETKEY (nKey)) != NIL
         // If invoking a SETKEY, pass the calling
         // procedure name and line number
         EVAL (bKey, PROCNAME (1), PROCLINE (1), READVAR ())
      ELSE
         // No SETKEY defined, exit the loop
         EXIT
      ENDIF
   ENDDO

   RETURN nKey

// EOF MESSAGE.PRG
