/***
*  Errorsys.prg
*  Standard Clipper 5.0 error handler
*  Copyright (c) 1990 Nantucket Corp.  All rights reserved.
*
*  Compile:  /m/n/w
*/

/****
*  ErrSys02.prg
*  Enhanced error processor based on standard ERRORSYS.PRG
*  Includes Software Trouble Report (STR) generator
*  (Changes are shown in Bold Face)
*/

#include "inkey.ch"
#include "error.ch"
#include "directry.ch"

// used below
#define NTRIM(n)     (LTrim (Str (n)))

#command ? <list,...>   =>  ?? Chr(13) + Chr(10) ; ?? <list>
#command ?? <list,...>  =>  OutErr(<list>)
#command ??? <list,...> =>  QQOUT (<list>); QQOUT (Chr(13) + Chr(10))

// File-wide statics needed by the Software Trouble Report

STATIC cSerialNo := "TestSer#"
STATIC cEndUser  := "In-house"
STATIC cVersion  := "V 1.0"
STATIC nHandle

/***
*  ErrorSys()
*
*  Note:  automatically executes at startup
*/

proc ErrorSys()
   ErrorBlock( {|e| DefError(e)} )

   // New code to handle the Software Trouble Report option

   IF FILE ("STR_FILE.TXT")
      nHandle := FOPEN ("STR_FILE.TXT")
   ELSE
      nHandle := FCREATE ("STR_FILE.TXT")
   ENDIF

   // End of new code

   return

/***
*  StrInit (cSerial, cUser, cVers)
*/
FUNCTION StrInit (cSerial, cUser, cVers)

// Initialization function for the Software Trouble Report
// Provides a mechanism for setting the serial number,
// end-user name, and program version number for the report.
// Should be called as soon as these parameters can be safely
// and accurately established.

IF VALTYPE (cSerial) == "C"
   cSerialNo := cSerial
ENDIF

IF VALTYPE (cUser) == "C"
   cEndUser := cUser
ENDIF

IF VALTYPE (cVers) == "C"
   cVersion := cVers
ENDIF

RETURN NIL

/***
*  StrClose ()
*/
FUNCTION StrClose ()

// Shutdown function for the Software Trouble Report
// Provides a mechanism for deleting the file if no
// errors have been recorded. Should be the last function
// called before the main line function RETURNs or QUITs.

LOCAL aFile

// Close STR_FILE.TXT
FCLOSE (nHandle)

// Get the file information
aFile := DIRECTORY ("STR_FILE.TXT")

// If the file exists
IF LEN (aFile) > 0

   // and the size is zero (no errors recorded
   IF aFile [1] [F_SIZE] == 0

      // Let's get rid of it
      FERASE ("STR_FILE.TXT")
   ENDIF
ENDIF

RETURN NIL

/***
*  DefError()
*/
static func DefError(e)
local i, cMessage, aOptions, nChoice

// by default, division by zero yields zero
if ( e:genCode == EG_ZERODIV )
   return (0)
end

// for network open error, set NETERR() and subsystem default
if ( e:genCode == EG_OPEN .and. e:osCode == 32 .and. e:canDefault )
   NetErr (.t.)
   return (.f.)
end

// for lock error during APPEND BLANK, set NETERR() and subsystem default
if ( e:genCode == EG_APPENDLOCK .and. e:canDefault )
   NetErr (.t.)
   return (.f.)
end

// build error message
cMessage := ErrorMessage(e)

// build options array
// aOptions := {"Break", "Quit"}

// Start with a new option - Trouble Report
aOptions := {"Trouble Report", "Quit"}

if (e:canRetry)
   AAdd (aOptions, "Retry")
end

if (e:canDefault)
   AAdd (aOptions, "Default")
end

// put up alert box
nChoice := 0
while (nChoice == 0 )
   if (Empty (e:osCode) )
      nChoice := Alert (cMessage, aOptions)
   else
      nChoice := Alert( cMessage + ;
                  ";(DOS Error " + NTRIM(e:osCode) + ")", ;
                  aOptions )
   end

   if ( nChoice == NIL )
      exit
   end
end

if ( !Empty(nChoice) )
   // do as instructed
   if ( aOptions [nChoice] == "Break" )
      Break (e)
   elseif ( aOptions[nChoice] == "Retry" )
      return (.t.)
   elseif ( aOptions[nChoice] == "Default" )
      return (.f.)
   elseif ( aOptions[nChoice] == "Trouble Report" )

      // Software Trouble Report desired
      StrReport (e, ProcName (2), ProcLine (2), ReadVar ())

      // Skip call traceback to console - abort now
      // Enhancement note - add the traceback to the STR

      Errorlevel (1)

      // When debugging the error system, add a call to the
      // CA-Clipper debugger before exiting. 
      // Let's check some things out.

      #ifdef ERRDEBUG
         ALTD ()
      #endif

      QUIT

      RETURN .F.
   end
end

// display message and traceback
if ( !Empty(e:osCode) )
   cMessage += " (DOS Error " + NTRIM (e:osCode) + ") "
end

? cMessage
i := 2
while (!Empty (ProcName (i)))
   ? "Called from", Trim (ProcName (i)) + "(" + NTRIM (ProcLine (i)) + ")  "

   i++
end

// give up
ErrorLevel (1)

// When debugging the error system, add a call to the
// CA-Clipper debugger before exiting. 
// Let's check some things out.

#ifdef ERRDEBUG
   ALTD ()
#endif

// Add a call here to properly close/delete the STR file.
StrClose ()

QUIT

return (.f.)

/***
*  ErrorMessage()
*/
static func ErrorMessage(e)
local cMessage

// start error message
cMessage := if( e:severity > ES_WARNING, "Error ", "Warning " )

// add subsystem name if available
if ( ValType(e:subsystem) == "C" )
   cMessage += e:subsystem()
else
   cMessage += "???"
end

// add subsystem's error code if available
if ( ValType(e:subCode) == "N" )
   cMessage += ("/" + NTRIM(e:subCode))
else
   cMessage += "/???"
end

// add error description if available
if ( ValType(e:description) == "C" )
   cMessage += ("  " + e:description)
end

// add either filename or operation
if ( !Empty(e:filename) )
   cMessage += (": " + e:filename)
elseif ( !Empty(e:operation) )
   cMessage += (": " + e:operation)
end

return (cMessage)

/**********************************************************************/

FUNCTION StrReport (oErr, cProc, nLine, cVar)

// State save variables - just in case we want to call
// this function elsewhere and return.

LOCAL cSaveScreen := SAVESCREEN (00, 00, 24, 79)
LOCAL cSaveColor  := SETCOLOR ()
LOCAL lConsole    := SET (_SET_CONSOLE, .T.)
LOCAL cDevice     := SET (_SET_DEVICE,  "SCREEN")
LOCAL lPrinter    := SET (_SET_PRINTER, .F.)

SETCOLOR ("W/N")
CLEAR SCREEN
SET CURSOR ON
@ 09, 19, 15, 60 BOX "ͻȺ "
SETCOLOR ("W+/N")
@ 10, 21 SAY PADC ("Outputting STR to file STR_FILE.TXT", 38)
@ 12, 21 SAY PADC ("* * W O R K I N G * *", 38) COLOR ("W+*/N")
@ 14, 21 SAY PADC ("Forward this file to Tech Support.", 38)
SETCOLOR ("W/N")
@ 16, 00
SET CONSOLE OFF
IF VALTYPE (oErr) == "O"             // If we are processing an error
   FCLOSE (nHandle)                  // Insure an open file for STR
ENDIF
SET ALTERNATE TO ("STR_FILE.TXT") ADDITIVE
SET ALTERNATE ON
StrPrint (oErr, cProc, nLine, cVar, cSaveScreen)
CLOSE ALTERNATE
SET CONSOLE ON
@ 12, 21 SAY PADC ("* * F I N I S H E D * *", 38) COLOR ("W+*/N")
SETCOLOR ("W/N")
@ 16, 00

SET (_SET_CONSOLE, lConsole)
SET (_SET_DEVICE,  cDevice)
SET (_SET_PRINTER, lPrinter)

SETCOLOR (cSaveColor)
SAVESCREEN (00, 00, 24, 79, cSaveScreen)

RETURN (NIL)

/**********************************************************************/

STATIC FUNCTION StrPrint (oErr, cProc, nLine, cVar, cSaveScreen)

LOCAL cCRLF     := CHR (13)  + CHR (10)
LOCAL cSoftCRLF := CHR (141) + CHR (10)
LOCAL i
LOCAL j

??? PADC ("Software Trouble Report", 80)
??? PADC (StrDate (), 80)
??? REPLICATE ("-", 20) + PADC (cVersion, 40) + REPLICATE ("-", 20)
??? PADR ("Tracking Number", 20) + REPLICATE (" ", 40) + ;
    PADL ("Completion Date", 20)
QQOUT (cCRLF)

??? PADC ("[ Runtime Environment ]", 80, "*")
QQOUT (cCRLF)

??? PADC ("Serial # : " + cSerialNo + "         Licensee : " + cEndUser, 80)
??? "  The current program module is:  " + cProc
??? "     The current line number is:  ", TRANSFORM (nLine, "@R 99,999")
??? "         The active variable is:  " + cVar
QQOUT (cCRLF)

??? "   Program Memory is as follows: 0) ",  ;
      TRANSFORM (MEMORY (0), "@R 999,999"),  ;
      "    1) ",                             ;
      TRANSFORM (MEMORY (1), "@R 999,999"),  ;
      "    2) ",                             ;
      TRANSFORM (MEMORY (2), "@R 999,999")
QQOUT (cCRLF)

??? PADC ("[ Current Screen ]", 80, "*")
QQOUT (cCRLF)

IF !EMPTY (cSaveScreen)
    FOR i := 0 TO 3999 step 160
        FOR j := 1 TO 160 STEP 2
            QQOUT (SUBSTR (cSaveScreen, (i + j), 1))
        NEXT
        QQOUT (cCRLF)
    NEXT
ENDIF
QQOUT (cCRLF)

??? PADC ("[ Error System Flags ]", 80, "*")
QQOUT (cCRLF)

IF VALTYPE (oErr) != "O"
   ??? PADC ("* * * No Error Block to Report From * * *", 80)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
   QQOUT (cCRLF)
ELSE
   QQOUT ("Can Default:    ")
   QQOUT (IIF (oErr:canDefault, "Y", "N"))
   QQOUT ("     Description: ")
   QQOUT (SUBSTR (oErr:description, 1, 45))
   QQOUT (cCRLF)

   QQOUT ("Can Retry:      ")
   QQOUT (IIF (oErr:canRetry, "Y", "N"))
   QQOUT ("     Filename:    ")
   QQOUT (SUBSTR (oErr:filename, 1, 45))
   QQOUT (cCRLF)

   QQOUT ("Can Substitute: ")
   QQOUT (IIF (oErr:canSubstitute, "Y", "N"))
   QQOUT ("     Operation:   ")
   QQOUT (SUBSTR (oErr:operation, 1, 45))
   QQOUT (cCRLF)

   QQOUT ("Gen Code:      ")
   QQOUT (TRANSFORM (oErr:genCode, "99999"))
   QQOUT ("  Sub System:  ")
   QQOUT (SUBSTR (oErr:subSystem, 1, 45))
   QQOUT (cCRLF)

   QQOUT ("OS  Code:      ")
   QQOUT (TRANSFORM (oErr:osCode, "99999"))
   QQOUT ("  Sub Code:    ")
   QQOUT (TRANSFORM (oErr:subCode, "99999"))
   QQOUT ("        Tries:  ")
   QQOUT (TRANSFORM (oErr:tries, "99999"))
   QQOUT (cCRLF)

   QQOUT (cCRLF)

   IF VALTYPE (oErr:args) == "A"
      i := LEN (oErr:args)
   ELSE
      i := 0
   ENDIF
   QQOUT ("Arguments: ")
   QQOUT (ALLTRIM (STR (i)))
   QQOUT (cCRLF)

   IF i == 0
      QQOUT (cCRLF)
      QQOUT (cCRLF)
      QQOUT (cCRLF)
   ELSE
      FOR j := 1 TO i
         QQOUT ("Arg #")
         QQOUT (SUBSTR ("123", j, 1))
         QQOUT ("(")
         QQOUT (VALTYPE (oErr:args [j]))
         QQOUT ("):")
         DO CASE
            CASE VALTYPE (oErr:args [j]) == "B"
               QQOUT (" {Code Block}")
            CASE VALTYPE (oErr:args [j]) == "A"
               QQOUT (" {Array}")
            CASE VALTYPE (oErr:args [j]) == "O"
               QQOUT (" {Object}")
            CASE VALTYPE (oErr:args [j]) == "U"
               QQOUT (" {NIL}")
            OTHERWISE
               QQOUT (oErr:args [j])
         ENDCASE
         QQOUT (cCRLF)
      NEXT
      DO WHILE i < 3
         QQOUT (cCRLF)
         i ++
      ENDDO
   ENDIF
ENDIF

// Output a Form Feed
QQOUT (CHR (12))

RETURN NIL

STATIC FUNCTION StrDate (dDate)

dDate := IIF (VALTYPE (dDate) == "D", dDate, DATE ())

RETURN ALLTRIM (CMONTH (dDate)) + " " + ;
         TRANSFORM (DAY (dDate), "99") + ", " + ;
         TRANSFORM (YEAR (DATE ()), "9999")

// EOF ERRSYS02.PRG

