********************************************************************************
* Program Id: tagdbf.prg
*    Version: 1.00
*       Date: 5-4-89
*     Author: John F. Kaster
********************************************************************************
*
*    Purpose: allows tagging of records in a data file.
*
*      Notes: in order to keep track of these tags, a character memory variable
*             of the same length as the number of records in the file is used.
*             Obviously, this limits the size of the datafile that can success-
*             fully be used with this to 65535 records, since clipper's 
*             character variables are limited to that length.
*
*             The only complexity in this character field is that the tags go
*             exclusively by record number, not relative position.  This was 
*             done because changing the index could result in the tags being
*             associated with different records.  Therefore, in order to
*             process the file according to the tags, go to the record number of
*             the character positions that are "" (character 251).
*
*             The only keys that are used by tagrecs() are [Alt-T], which tags
*             all records; [Alt-U], which untags all records; [Space Bar],
*             which toggles individual records; [Esc], which aborts selection;
*             and [Enter], which preserves tags upon return.
*
*             The data file is NEVER EDITED, and this example is set up to
*             support multi-user access to the file.  I only spent an hour on
*             this routine, so it's not the fanciest, but hopefully it's 
*             modular enough for you to modify it to your heart's content.
*
*             Of course, there are many things that can be done with this,
*             including passing window dimensions, and support for files larger
*             than 64K by using bits instead of characters.
*
********************************************************************************
*
*  Databases: whatever is active
*
********************************************************************************
* Date     Name Description
*
* 05/03/89 jfk  Original program
* 05/26/89 jfk  allows for 64K * 8 records by using RVH's BIT.C function
********************************************************************************
parameter dbfalias, tagproc
private pc

pc = pcount()
if pc = 0
  ?? 'tagdbf <filename>'
  quit
endif

if pc < 2
  tagproc = '""'
endif

set exclusive off
use &dbfalias

taglist = &tagproc

* do while lastkey() # 27
  taglist = tagrecs( m->taglist )
* enddo

clea
proctags( taglist, 'printrec()' )

return

*****************
* printrec()
*
* example function called from proctags()
*****************
function printrec
private fn
fn = field(1)
? &fn
return .t.


*****************
* proctags( <taglist>, <user-defined function> [, <datafile> ] )
*
* Uses <taglist> to determine which records to perform <udf> on.
* If <datafile> is not passed, then the current datafile is used.
* This could be a problem if many records are being added to the
* system at a time.
*
* <taglist> - the bit string toggles for tags on the records in the data file
* <udf>     - the macro-able expression to call whenever a tagged record is
*             found
* <aliname> - 
*****************
function proctags
parameter taglist, udf, aliname
private garbage, pc, cursel, currec, alirec
pc = pcount()
if m->pc < 2
  return .f.
elseif m->pc < 3
  aliname = alias()
endif

* Checking if aliname is a valid alias
if select( m->aliname ) = 0
  return .f.
endif

* Checking to see if there are any records in the file to process!
if &aliname->( lastrec() ) = 0
  return .f.
endif

* Saving current datafile and position
cursel  = select()
currec  = recno()
select select( m->aliname )
alirec  = recno()

* Make sure we don't try to do tags that aren't there any more
l = min( slen( m->taglist ) * 8, lastrec() )
for i = 1 to m->l
  if bit( m->taglist, m->i ) && tag is on
    goto m->i
    garbage = &udf
  endif
next i

* Restoring locations in files that may have been moved
if recno() # m->alirec
  goto m->alirec
endif
if m->cursel # select( m->aliname )
  select select( m->cursel )
  if recno() # m->currec
    goto m->currec
  endif
endif
return .t.

****************
*
* Syntax:   TagRecs( <tag list> [, <field list> [, <headings> ] ] )
*
* Notes:    uses the existing <tag list> if passed, fills with blanks if not.
*           If [Esc] is used to exit, a blank list is returned.  If [Enter]
*           is used, the updated list is returned.
*         
* Returns:  updated tag list.
*
****************
function tagrecs

parameters taglist, field_list, headings
private pc, i, taglen, reclen
pc = pcount()

* declare tag list
if m->pc < 1
  taglen  = 0
  taglist = ''
else
  * slen() is a 'C' function that returns the REAL length of the string, 
  * regardless of embedded chr(0)s
  taglen = slen( m->taglist )
endif

* Since we're using bits, the length of the string is 8 times smaller than
* the number of records.  Add 1 just to make sure of something not exactly
* divisible by 8
reclen = int( lastrec() / 8 + 1 )

if m->taglen < m->reclen
  * adjusting size of taglist, because more records were evidently added
  taglist = m->taglist + repl( chr(0), m->reclen - m->taglen )
endif
taglen = slen( m->taglist )

* Declare field_list
if m->pc < 2
  l = fcount() + 1
  private field_list[ m->l ]
  for i = 2 to m->l
    field_list[ m->i ] = field( m->i - 1 )
  next && i
endif

* Declare headings
if m->pc < 3
  l = fcount() + 1
  private headings[ m->l ]
  for i = 2 to m->l
    headings  [ m->i ] = field_list[ m->i ]
  next && i
endif

* set up first field to be the tags
field_list[ 1 ] = "if( bit( m->taglist, recno() ), '', ' ' )"
headings  [ 1 ] = ""

* a little help
@ 24,0 SAY '[Space] to tag/untag  [Esc] to abort selections'

* call dbedit()
dbedit( 1,0,23,79, field_list, 'tagem', "", headings )

* check to see how user exited the selection routine
if lastkey() = 27
  * return a blank tag list since [Esc] was pressed
  return repl( chr(0), m->taglen )
endif

* return the tag list
return m->taglist


**************
*
* Syntax: tagem()
*
*  Notes: tagem() is called from dbedit() by being specified as the UDF
*         in the dbedit() parameter list in tagrecs().
*
*         The memory variable taglen is public from tagfile.
*
**************
function tagem
parameter status, fld_ptr
private retval, tagpos
retval = 1
keypress = lastkey()
do case
* keyboard exceptions
case status = 4
  if chr( m->keypress ) = ' '
    * toggling the bit value
    tagpos = recno()
    bit( m->taglist, m->tagpos, ! bit( m->taglist, m->tagpos ) )
  elseif m->keypress = 276 && Alt-T
    taglist = repl( chr(255), m->taglen )
    retval = 2
  elseif m->keypress = 278 && Alt-U
    taglist = repl( chr(0), m->taglen )
    retval = 2
  elseif m->keypress = 27 .or. m->keypress = 13
    retval = 0
  endif

endcase
return m->retval

