***********************************************************
* Author  : Paul D. Sheriff
* Date    : Sep. 24, 1992
* Function: Str_2Array() - Convert Delimited String to Array
*
* Description:
* Convert a delimited string into an array of many elements.
* This routine will trim the elements before putting them
* into the array.  If any are empty, they will not be entered.
* The Type of the element may be passed.  Thus the string may
* be converted into numeric, date or logical types.
*
* Syntax: <A> Str_2Array(ExpC, [ExpC], [ExpC])
*
* Parameters:
* <C>  cStr        => String to Convert.
* <B>  bBlock      => Code Block To Format Data
*                     Default => NIL
* <C>  cDelimit    => Delimiter.
*                     Default => ","
*
* Tables:
* None.
*
* Example:
* aItems := Str_2Array(cStr)
* aItems := Str_2Array("Paul, Steve, Dennis")
* aItems := Str_2Array("0001, 0002, 0003", ;
*                     { | cVal | Transform(cVal, "99-99") })
* aItems := Str_2Array("01/01/92, 02/01/92, 03/01/92", ;
*                     { | dVal | Ctod(dVal) })
*
* Notice: Released into Public Domain by Author
***********************************************************
FUNCTION Str_2Array( cStr, cDelimit, bBlock )

// Declare Local Variables
LOCAL aItems := {}, cTemp

// Check Parameters
cDelimit := If(ValType(cDelimit) != 'C', ",", cDelimit)

IF !Empty(cStr)
   cStr := ' ' + cStr + cDelimit + ' '
ENDIF
DO WHILE !Empty(cStr)
   cTemp := AllTrim(Substr(cStr, 1, At(cDelimit, cStr) - 1))
   cStr  := Substr(cStr, At(cDelimit, cStr) + Len(cDelimit))

   IF !Empty(cTemp)
      IF ValType(bBlock) == 'B'
         Aadd(aItems, Eval(bBlock, cTemp))
      ELSE
         Aadd(aItems, cTemp)
      ENDIF
   ENDIF
ENDDO

Return(aItems)

************************************************************
* Author  : Paul D. Sheriff
* Date    : Sep. 24, 1992
* Function: Arr_2Str() - Convert Array To Delimited String
*
* Description:
* Convert an array of values to a delimited string.
*
* Syntax: <C> Arr_2Str(ExpA, [ExpC])
*
* Parameters:
* <A>  aValues     => Array of Values To Convert
* <C>  cDelimit    => Delimiter.
*                     Default => ","
*
* Example:
* aValues[1] := "Paul"
* aValues[2] := "Dennis"
* aValues[3] := "Steve"
* cStr := Arr_2Str(aValues)
* ? cStr        // Prints "Paul,Dennis,Steve"
*
* Notice: Released Into Public Domain by Author
************************************************************
FUNCTION Arr_2Str( aValues, cDelimit )

// Declare Local Variables
LOCAL cRetVal := "", nLen := Len(aValues), nLoop

// Check Parameters
cDelimit := If(ValType(cDelimit)  != 'C', ",", cDelimit)

FOR nLoop := 1 TO nLen
   cRetVal += Fld_Convert('C', aValues[nLoop])
   IF nLoop < nLen
      cRetVal += cDelimit
   ENDIF
NEXT

Return(cRetVal)

************************************************************
* Author  : Paul D. Sheriff
* Date    : Sep. 24, 1992
* Function: Fld_Convert() - Convert any data type
*
* Description:
* Convert any data type to another type.
*
* Syntax: <C> Fld_Convert(ExpA, [ExpC])
*
* Parameters:
* <C> cType    => The type to convert to
* <x> xVal     => The data to convert
*
* Notice: Released Into Public Domain by Author
************************************************************
FUNCTION Fld_Convert( cType, xVal )

LOCAL cXType := ValType(xVal)

IF cXType != cType
   DO CASE
      CASE cXType == "N" .AND. cType == 'C'
         xVal := Str(xVal)       // Numeric to Character

      CASE cXType == "D" .AND. cType == 'C'
         xVal := Dtoc(xVal)      // Date To Character

      CASE cXType == "L" .AND. cType == 'C'
        xVal := If(xVal, "T", "F")  // Logical To Char

      CASE cXType == "C" .AND. cType == 'L'
         xVal := If(xVal $ "TtYy", .T., .F.)  // Char To Logical

      CASE cXType == "C" .AND. cType == 'D'
         xVal := Ctod(xVal)     // Character To Date

      CASE cXType == "C" .AND. cType == 'N'
         xVal := Val(xVal)      // Character To Numeric

   ENDCASE
ENDIF

Return(xVal)
