/* =========================================================================

   Dvmouse.c       A Clipper and C mouse interface library
	 Original Written By Steve Forbes Clipper User Group Melborne
										Compuserve 100033,2073
	 Change/Re Written By David Palmer and Karl Lysinger
										Sacramento CA-Clipper User group
										Compuserve 73123,3354 Dave Palmer
										CA-Clipper User Group President
	 Compile          BCC -c -G -f- -N- -ml -B dvmouse.c


   ========================================================================= */

#pragma inline

#include "c:\clipper5\include\extend.h"
#include <dos.h>
#include "dvmouse.h"

/* This variable prevents mouse cursor being hidden more than once at a time */

static Boolean mouse_cursor_visible = FALSE;

/* This array contains binary masks used in some of the mouse functions. The
   array is indexed by the #defines LBUTTON, RBUTTON, MBUTTON and ALLBUTTON
   which appear in the #include file "mouselib.h"                            */

static int mask_table[ 4 ] = { LEFT, RIGHT, MIDDLE, ALL };

/* -------------------------------------------------------------------------

   Function             _MReset()

   Description:         Resets the mouse and verifies its existence.

   Parameters:
    (output)            status     Status of the mouse
    (output)            buttons    Number of mouse buttons

   Calls:               mouse_reset()

   Returned:            (function returns nothing)
Ref MouseReset
   -------------------------------------------------------------------------
*/

CLIPPER _MReset()
{

    int status, buttons;

    mouse_reset( &status, &buttons );

    _storni( status, 1 );
    _storni( buttons, 2 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function             _IsMDriver()

   Description:         Verifies the existence of a mouse.

   Parameters:          none

   Calls:               mouse_storage()
                        _xgrab()
                        mouse_save()
                        mouse_hide()
                        mouse_reset()
                        mouse_restore()
                        mouse_show()
                        _xfree()

   Returns:             Logical (True if exists, false otherwise)
Ref MouseExists
   -------------------------------------------------------------------------
*/

CLIPPER _IsMDriver()
{

    int bytes;
    char *buffer;
    int status, buttons, reshow_mouse;

    mouse_storage( &bytes );

    buffer = _xgrab( bytes );

    mouse_save( buffer );

    if ( mouse_cursor_visible )
    {
        mouse_hide();
        reshow_mouse = TRUE;
    }
    else
    {
        reshow_mouse = FALSE;
    }

    mouse_reset( &status, &buttons );

    mouse_restore( buffer );

    if ( reshow_mouse )
        mouse_show();

    _xfree( buffer );

    _retl( status );

    return;

}

/* -------------------------------------------------------------------------

   Function             _Mouseon()

   Description:         Make the mouse cursor visible

   Parameters:          (none)

   Calls:               mouse_show()

   Returns:             (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _Mouseon()
{

    mouse_show();
    mouse_cursor_visible = TRUE;

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function             _Mouseoff()

   Description:         Make the mouse cursor invisible

   Parameters:          (none)

   Calls:               mouse_hide()

   Returns:             (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseOff()
{

    if( mouse_cursor_visible )
    {
        mouse_hide();
        mouse_cursor_visible = FALSE;
    }

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MousePut()

   Description:         Sets the mouse cursor to the indicated position

   Parameters:
   (output)             horizontal      Horizontal position
   (output)             vertical        Vertical position

   Calls:               mouse_setpos()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MousePut()
{

    int row, col;

    row = _parni( 1 );
    col = _parni( 2 );

    mouse_setpos( row, col );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _ButtonPress()

   Description:         Gets button press information

   Parameters:
   (input)              button          Left or right button
   (output)             status          Status of the button
   (output)             presses         Number of button presses
   (output)             vert_pos        Vertical position at last press
   (output)             horz_pos        Horizontal position at last press

   Calls:               mouse_press()

   Returned:            (function returns nothing)
	 Reference Buttonpress/mousepress
   -------------------------------------------------------------------------
*/

CLIPPER _BtnPress()
{

    int button, status, presses, row, column;

    button = _parni( 1 );

    mouse_press( button, &status, &presses, &row, &column );

    _storni( status, 2 );
    _storni( presses, 3 );
    _storni( row, 4 );
    _storni( column, 5 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _ButtonRel()

   Description:         Gets button release information

   Parameters:
   (input)              button          Left or right button
   (output)             status          Status of the button
   (output)             releases        Number of button releases
   (output)             horz_pos        Horizontal position at last release
   (output)             vert_pos        Vertical position at last release

   Calls:               mouse_release()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _ButtonRel()
{

    int button, status, releases, row, column;

    button = _parni( 1 );

    mouse_release( button, &status, &releases, &row, &column );

    _storni( status, 2 );
    _storni( releases, 3 );
    _storni( row, 4 );
    _storni( column, 5 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _DblThrHld()

   Description:         Sets the mouse double speed threshold

   Parameters:
   (input)              mickeys_per_second  Double speed threshold

   Calls:               mouse_setdouble()

   Returned:            (function returns nothing)
	 Ref MouseDbl()
   -------------------------------------------------------------------------
*/

CLIPPER _DblThrHld()
{

    int mickeys_per_second;

    mickeys_per_second = _parni( 1 );

    mouse_setdouble( mickeys_per_second );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseTCur()

   Description:         Sets the text mode hardware or software cursor

   Parameters:
   (input)              cursor_select   Hardware or software cursor
   (input)              screen_mask     Screen mask (or start scan line)
   (input)              cursor_mask     Cursor mask (or end scan line)

   Calls:               mouse_settcurs()

   Returned:            (function returns nothing)
ref mousetcurs()
   -------------------------------------------------------------------------
*/

CLIPPER _MouseTCur()
{

    int cursor_select, screen_mask, cursor_mask;

    cursor_select = _parni( 1 );
    screen_mask   = _parni( 2 );
    cursor_mask   = _parni( 3 );

    mouse_settcurs( cursor_select, screen_mask, cursor_mask );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MlastPos()

   Description:         Gets the accumulated mouse motion counts (mickeys)
                        since the last call to this function

   Parameters:
   (output)             vert_mickeys    Vertical mickeys
   (output)             horz_mickeys    Horizontal mickeys

   Calls:               mouse_motion()

   Returned:            (function returns nothing)
	Ref MouseMoves()
   -------------------------------------------------------------------------
*/

CLIPPER _MLastPos()
{

    int v_mickeys, h_mickeys;

    mouse_motion( &v_mickeys, &h_mickeys );

    _storni( v_mickeys, 1 );
    _storni( h_mickeys, 2 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MSpeed()

   Description:         Sets the mickey/pixel ratios for mouse motion

   Parameters:
   (output)             vertical        Vertical mickey/pixel ratio
   (output)             horizontal      Horizontal mickey/pixel ratio

   Calls:               mouse_setratios()

   Returned:            (function returns nothing)
ref MouseRatio
   -------------------------------------------------------------------------
*/

CLIPPER _MSpeed()
{

    int vertical, horizontal;

    vertical   = _parni( 1 );
    horizontal = _parni( 2 );

    mouse_setratios( vertical, horizontal );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseSetH()

   Description:         Sets the minimum and maximum horizontal mouse
                        cursor positions

   Parameters:
   (input)              horz_min        Minimum horizontal cursor position
   (input)              horz_max        Maximum horizontal cursor position

   Calls:               mouse_sethorz()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseSetH()
{

    int horz_min, horz_max;

    horz_min = _parni( 1 );
    horz_max = _parni( 2 );

    mouse_sethorz( horz_min, horz_max );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseSetV()

   Description:         Sets the minimum and maximum vertical mouse
                        cursor positions

   Parameters:
   (input)              vert_min        Minimum vertical cursor position
   (input)              vert_max        Maximum vertical cursor position

   Calls:               mouse_setvert()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseSetV()
{

    int vert_min, vert_max;

    vert_min = _parni( 1 );
    vert_max = _parni( 2 );

    mouse_setvert( vert_min, vert_max );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MOffBx()

   Description:         Sets a region for conditionally turning off the
                        mouse cursor

   Parameters:
   (input)              x1              Upper left corner of region
   (input)              y1              Upper left corner of region
   (input)              x2              Lower right corner of region
   (input)              y2              Lower right corner of region

   Calls:               mouse_condoff()

   Returned:            (function returns nothing)
Ref MouseoffBx
   -------------------------------------------------------------------------
*/

CLIPPER _MOffBx()
{

    int top, left, bottom, right;

    top    = _parni( 1 );
    left   = _parni( 2 );
    bottom = _parni( 3 );
    right  = _parni( 4 );

    mouse_condoff( top, left, bottom, right );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseStore()

   Description:         Determines the number of bytes required for saving
                        the current state of the mouse

   Parameters:
   (output)             buffer_size     Bytes for saving mouse state

   Calls:               mouse_storage()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseStore()
{

    int buffer_size;

    mouse_storage( &buffer_size );

    _storni( buffer_size, 1 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseSave()

   Description:         Save the current state of the mouse

   Parameters:
   (input/output)       buffer          Buffer for saving mouse state

   Calls:               mouse_save()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseSave()
{
    char *buffer;

    buffer = _parc( 1 );

    mouse_save( buffer );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseRest()

   Description:         Restores the current state of the mouse

   Parameters:
   (input/output)       buffer          Buffer for restoring mouse state

   Calls:               mouse_restore()

   Returned:            (function returns nothing)
	 ref MouseRest
   -------------------------------------------------------------------------
*/

CLIPPER _MouseRest()
{

    char *buffer;

    buffer = _parc( 1 );

    mouse_restore( buffer );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseSetSn()

   Description:         Sets the mouse sensitivity and double speed
                        threshold

   Parameters:
   (input)              vert            Relative vertical sensitivity
   (input)              horz            Relative horizontal sensitivity
   (input)              threshold       Relative double speed threshold

   Calls:               mouse_setsensitivity()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseSetSn()
{

    int vert, horz, threshold;

    vert      = _parni( 1 );
    horz      = _parni( 2 );
    threshold = _parni( 3 );

    mouse_setsensitivity( vert, horz, threshold );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseGetSn()

   Description:         Gets the mouse sensitivity and double speed
                        threshold

   Parameters:
   (output)             vert            Relative vertical sensitivity
   (output)             horz            Relative horizontal sensitivity
   (output)             threshold       Relative double speed threshold

   Calls:               mouse_getsensitivity()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseGetSn()
{

    int vert, horz, threshold;

    mouse_getsensitivity( &vert, &horz, &threshold );

    _storni( vert, 1 );
    _storni( horz, 2 );
    _storni( threshold, 3 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------
   Function:            _MouseArea()

   Description:         Sets the minimum and maximum horizontal and vertical
                        mouse cursor positions, limiting mouse movement to a
                        specified rectangular area

   Parameters:
   (input)              top             Minimum vertical cursor position
   (input)              left            Minimum horizontal cursor position
   (input)              bottom          Maximum vertical cursor position
   (input)              right           Maximum horizontal cursor position

   Calls:               mouse_setvert()
                        mouse_sethorz()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseArea()
{

    mouse_setvert( _parni( 1 ), _parni( 3 ) );
    mouse_sethorz( _parni( 2 ), _parni( 4 ) );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseSetPg()

   Description:         Sets the video page where mouse cursor appears

   Parameters:
   (input)              crt_page        Video page for mouse cursor

   Calls:               mouse_setpage()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseSetPg()
{

    int video_page;

    video_page = _parni( 1 );

    mouse_setpage( video_page );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseGetPg()

   Description:         Gets the video page in which mouse cursor appears

   Parameters:
   (input)              crt_page        Video page for mouse cursor

   Calls:               mouse_getpage()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseGetPg()
{

    int current_page;

    mouse_getpage( &current_page );

    _storni( current_page, 1 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _IsButton()

   Description:

   Parameters:
   (input)              which_button    Video page for mouse cursor

   Calls:               mouse_button_pressed()

   Returned:            logical

   -------------------------------------------------------------------------
*/

CLIPPER _IsButton()
{

    int which_button, pressed;

    which_button = _parni( 1 );

    pressed = mouse_button_pressed( mask_table[ which_button ] );

    _retl( pressed );

}

/* -------------------------------------------------------------------------

   Function:            _MInBox()

   Description:

   Parameters:
   (input)              button          Left or right button
   (output)             status          Status of the button
   (output)             presses         Number of button presses
   (output)             vert_pos        Vertical position at last press
   (output)             horz_pos        Horizontal position at last press

   Calls:               mouse_status()

   Returned:            (function returns nothing)
Ref MouseinBox
   -------------------------------------------------------------------------
*/

CLIPPER _MInBox()
{

    int top, left, bottom, right, lbutton, rbutton, mbutton, row, column;

    top    = _parni( 1 );
    left   = _parni( 2 );
    bottom = _parni( 3 );
    right  = _parni( 4 );

    mouse_status( &lbutton, &rbutton, &mbutton, &row, &column );

    _storni( lbutton, 5 );
    _storni( rbutton, 6 );
    _storni( mbutton, 7 );

    if( ( row >= top && row <= bottom) && ( column >= left && column <= right ) )
        _retl( 1 );
    else _retl( 0 );

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MState()

   Description:         Gets the current state of the mouse buttons and
                        the mouse cursor position

   Parameters:
   (output)             left_button     State of the left button
   (output)             right_button    State of the right button
   (output)             horz_pos        Horizontal position of the cursor
   (output)             vert_pos        Vertical position of the cursor

   Calls:               mouse_status()

   Returned:            (function returns nothing)
Ref MouseState
   -------------------------------------------------------------------------
*/

CLIPPER _MState()
{

    int lbutton, rbutton, mbutton, row, column;

    mouse_status( &lbutton, &rbutton, &mbutton, &row, &column );

    _storni( lbutton, 1 );
    _storni( rbutton, 2 );
    _storni( mbutton, 3 );
    _storni( row, 4 );
    _storni( column, 5 );

    _ret();

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseSeen()

   Description:         Report whether the mouse cursor is currently visible

   Parameters:          (none)

   Calls:               (none)

   Returned:            logical - the current value of mouse_cursor_visible

   -------------------------------------------------------------------------
*/

CLIPPER _MouseSeen()
{

    _retl( mouse_cursor_visible );

    return;

}

/* -------------------------------------------------------------------------

   Function:            _MouseRow()

   Description:         Reports current row position of the couse cursor

   Parameters:
   (output)             row             Current row position

   Calls:               mouse_row()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseRow()
{

    int row;

    mouse_row( &row );

    _storni( row, 1 );

    _ret();

}

/* -------------------------------------------------------------------------

   Function:            _MouseCol()

   Description:         Reports current column position of the couse cursor

   Parameters:
   (output)             col             Current column position

   Calls:               mouse_col()

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

CLIPPER _MouseCol()
{

    int col;

    mouse_col( &col );

    _storni( col, 1 );

    _ret();

}

/* -------------------------------------------------------------------------

   Function:            _M_Regions()

   Description:         Reports whether the mouse cursor is within a set of
                        regions passed in four arrays.

   Parameters:
   (input)              tops    Array containing the top row co-ordinate
                                for each of the areas to be checked
   (input)              lefts   Array containing the left column co-ordinate
                                for each of the areas to be checked
   (input)              bottoms Array containing the bottom row co-ordinate
                                for each of the areas to be checked
   (input)              rights  Array containing the right column co-ordinate
                                for each of the areas to be checked

   Calls:               mouse_row()
                        mouse_col()

   Returned:            Array index of the area where the mouse is, otherwise
                        returns zero.

   -------------------------------------------------------------------------
*/

CLIPPER _M_Regions()
{

    void mouse_row( int *vert_pos );
    void mouse_col( int *horz_pos );

    int i, len, row, col, retval;
    int t, l, b, r;

    len = ALENGTH( 1 );
    retval = 0;

    mouse_row( &row );
    mouse_col( &col );

    for ( i = 1; i <= len; i++ )
    {

        t = _parni( 1, i );
        l = _parni( 2, i );
        b = _parni( 3, i );
        r = _parni( 4, i );

        if ( ( row >= t && row <= b ) && ( col >= l && col <= r ) )
        {
            retval = i;
        }

    }

    _retni( retval );

    return;

}

/* -------------------------- Mouse Function #0 ----------------------------

                              mouse_reset()

   Description:         Resets the mouse and verifies its existence.

   Parameters:
    (output)            status     Status of the mouse
    (output)            buttons    Number of mouse buttons

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_reset( int *status, int *buttons )
{

    int m1, m2;

    asm {
        xor     ax, ax
        int     33h
        mov     m1, ax
        mov     m2, bx
    }
    *status = m1;
    *buttons = m2;
}

/* -------------------------- Mouse Function #1 ----------------------------

                               mouse_show()

   Description:         Make the mouse cursor visible

   Parameters:          (none)

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_show( void )
{

    asm {
        mov     ax, 1
        int     33h
    }

}

/* -------------------------- Mouse Function #2 ----------------------------

                               mouse_hide()

   Description:         Make the mouse cursor invisible

   Parameters:          (none)

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_hide( void )
{

    asm {
        mov     ax, 2
        int     33h
    }

}

/* -------------------------- Mouse Function #3 ----------------------------

                               mouse_status()

   Description:         Gets the current state of the mouse buttons and
                        the mouse cursor position

   Parameters:
   (output)             left_button     State of the left button
   (output)             right_button    State of the right button
   (output)             middle_button   State of the middle button
   (output)             horz_pos        Horizontal position of the cursor
   (output)             vert_pos        Vertical position of the cursor

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_status( int *left_button, int *right_button,
                          int *middle_button, int *vert_pos, int *horz_pos )
{

    int m2, m3, m4;

    asm {
        mov     ax, 3
        int     33h
        mov     m2, bx
        mov     m3, cx
        mov     m4, dx
    }

    *left_button = m2 & 1;
    *right_button = ( m2 >> 1 ) & 1;
    *middle_button = (m2 >> 2 ) & 1;
    *vert_pos = m4 / 8;
    *horz_pos = m3 / 8;

}

/* -------------------------- Mouse Function #4 ----------------------------

                              mouse_setpos()

   Description:         Sets the mouse cursor to the indicated position

   Parameters:
   (output)             vertical        Vertical position
   (output)             horizontal      Horizontal position

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_setpos( int row, int col )
{

    int vertical, horizontal;

    vertical   = row * 8;
    horizontal = col * 8;

    asm {
        mov     ax, 4
        mov     cx, horizontal
        mov     dx, vertical
        int     33h
    }
}

/* -------------------------- Mouse Function #5 ----------------------------

                              mouse_press()

   Description:         Gets button press information

   Parameters:
   (input)              button          Left or right button
   (output)             status          Status of the button
   (output)             presses         Number of button presses
   (output)             horz_pos        Horizontal position at last press
   (output)             vert_pos        Vertical position at last press

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_press( int button, int *status, int *presses,
                         int *vert_pos , int *horz_pos )
{

    int m1, m2, m3, m4;

    asm {
        mov     ax, 5
        mov     bx, button
        int     33h
        mov     m1, ax
        mov     m2, bx
        mov     m3, cx
        mov     m4, dx
    }

    if ( button == LBUTTON )
        *status = m1 & 1;
    else
    {
        if ( button == RBUTTON )
            *status = ( m1 >> 1 ) & 1;
        else
            *status = ( m1 >> 2 ) & 1;
    }

    *presses = m2;
    *vert_pos = m4 / 8;
    *horz_pos = m3 / 8;

}

/* -------------------------- Mouse Function #6 ----------------------------

                              mouse_release()

   Description:         Gets button release information

   Parameters:
   (input)              button          Left or right button
   (output)             status          Status of the button
   (output)             releases        Number of button releases
   (output)             horz_pos        Horizontal position at last release
   (output)             vert_pos        Vertical position at last release

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_release( int button, int *status, int *releases,
                           int *vert_pos, int *horz_pos )
{

    int m1, m2, m3, m4;

    asm {
        mov     ax, 6
        mov     bx, button
        int     33h
        mov     m1, ax
        mov     m2, bx
        mov     m3, cx
        mov     m4, dx
    }

    if ( button == LBUTTON )
        *status = m1 & 1;
    else
    {
        if ( button == RBUTTON )
            *status = ( m1 >> 1 ) & 1;
        else
            *status = ( m1 >> 2 ) & 1;
    }

    *releases = m2 / 8;
    *vert_pos = m4 / 8;
    *horz_pos = m3 / 8;

}

/* -------------------------- Mouse Function #7 ----------------------------

                              mouse_sethorz()

   Description:         Sets the minimum and maximum horizontal mouse
                        cursor positions

   Parameters:
   (input)              horz_min        Minimum horizontal cursor position
   (input)              horz_max        Maximum horizontal cursor position

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_sethorz( int min, int max )
{

    int min_column, max_column;

    min_column = min * 8;
    max_column = max * 8;

    asm {
        mov     ax, 7
        mov     cx, min_column
        mov     dx, max_column
        int     33h
    }
}

/* -------------------------- Mouse Function #8 ----------------------------

                              mouse_setvert()

   Description:         Sets the minimum and maximum vertical mouse
                        cursor positions

   Parameters:
   (input)              vert_min        Minimum vertical cursor position
   (input)              vert_max        Maximum vertical cursor position

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_setvert( int min, int max )
{

    int min_row, max_row;

    min_row = min * 8;
    max_row = max * 8;

    asm {
        mov     ax, 8
        mov     cx, min_row
        mov     dx, max_row
        int     33h
    }

}

/* -------------------------- Mouse Function #10 ---------------------------

                              mouse_settcurs()

   Description:         Sets the text mode hardware or software cursor

   Parameters:
   (input)              cursor_select   Hardware or software cursor
   (input)              screen_mask     Screen mask (or start scan line)
   (input)              cursor_mask     Cursor mask (or end scan line)

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_settcurs( int cursor_select, int screen_mask, int cursor_mask )
{

    asm {
        mov     ax, 10
        mov     bx, cursor_select
        mov     cx, screen_mask
        mov     dx, cursor_mask
        int     33h
    }
}

/* -------------------------- Mouse Function #11 ---------------------------

                              mouse_motion()

   Description:         Gets the accumulated mouse motion counts (mickeys)
                        since the last call to this function

   Parameters:
   (output)             horz_mickeys    Horizontal mickeys
   (output)             vert_mickeys    Vertical mickeys

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_motion( int *vert_mickeys, int *horz_mickeys )
{

    int m3, m4;

    asm {
        mov     ax, 11
        int     33h
        mov     m3, cx
        mov     m4, dx
    }

    *vert_mickeys = m4 / 8;
    *horz_mickeys = m3 / 8;

}

/* -------------------------- Mouse Function #15 ---------------------------

                              mouse_setratios()

   Description:         Sets the mickey/pixel ratios for mouse motion

   Parameters:
   (output)             horizontal      Horizontal mickey/pixel ratio
   (output)             vertical        Vertical mickey/pixel ratio

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_setratios( int vertical, int horizontal )
{

    asm {
        mov     ax, 15
        mov     cx, horizontal
        mov     dx, vertical
        int     33h
    }

}

/* -------------------------- Mouse Function #16 ---------------------------

                              mouse_condoff()

   Description:         Sets a region for conditionally turning off the
                        mouse cursor

   Parameters:
   (input)              top             Upper left corner of region
   (input)              left            Upper left corner of region
   (input)              bottom          Lower right corner of region
   (input)              right           Lower right corner of region

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_condoff( int top, int left, int bottom, int right )
{

    int m1, m2, m3, m4;

    m1 = top    * 8;
    m2 = left   * 8;
    m3 = bottom * 8;
    m4 = right  * 8;

    asm {
        mov     ax, 16
        mov     cx, m2
        mov     dx, m1
        mov     si, m4
        mov     di, m3
        int     33h
    }

}

/* -------------------------- Mouse Function #19 ---------------------------

                              mouse_setdouble()

   Description:         Sets the mouse double speed threshold

   Parameters:
   (input)              mickeys_per_second  Double speed threshold

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_setdouble( int mickeys_per_second )
{

    asm {
        mov     ax, 19
        mov     dx, mickeys_per_second
        int     33h
    }

}

/* -------------------------- Mouse Function #21 ---------------------------

                              mouse_storage()

   Description:         Determines the number of bytes required for saving
                        the current state of the mouse

   Parameters:
   (output)             buffer_size     Bytes for saving mouse state

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_storage( int *buffer_size )
{

    int m2;

    asm {
        mov     ax, 21
        int     33h
        mov     m2, bx
    }

    *buffer_size = m2;

}

/* -------------------------- Mouse Function #22 ---------------------------

                               mouse_save()

   Description:         Save the current state of the mouse

   Parameters:
   (input/output)       buffer          Buffer for saving mouse state

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_save( char far *buffer )
{

    unsigned buffer_seg = FP_SEG( buffer );
    unsigned buffer_off = FP_OFF( buffer );

    asm {
        mov     ax, 22
        mov     es, buffer_seg
        mov     dx, buffer_off
        int     33h
    }

}

/* -------------------------- Mouse Function #23 ---------------------------

                              mouse_restore()

   Description:         Restores the current state of the mouse

   Parameters:
   (input/output)       buffer          Buffer for restoring mouse state

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_restore( char far *buffer )
{

    unsigned buffer_seg = FP_SEG( buffer );
    unsigned buffer_off = FP_OFF( buffer );

    asm {
        mov     ax, 23
        mov     es, buffer_seg
        mov     dx, buffer_off
        int     33h
    }

}

/* -------------------------- Mouse Function #26 ---------------------------

                              mouse_setsensitivity()

   Description:         Sets the mouse sensitivity and double speed
                        threshold

   Parameters:
   (input)              horz            Relative horizontal sensitivity
   (input)              vert            Relative vertical sensitivity
   (input)              threshold       Relative double speed threshold

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_setsensitivity( int vert, int horz, int threshold )
{

    asm {
        mov     ax, 26
        mov     bx, horz
        mov     cx, vert
        mov     dx, threshold
        int     33h
    }

}

/* -------------------------- Mouse Function #27 ---------------------------

                              mouse_getsensitivity()

   Description:         Gets the mouse sensitivity and double speed
                        threshold

   Parameters:
   (output)             vert            Relative vertical sensitivity
   (output)             horz            Relative horizontal sensitivity
   (output)             threshold       Relative double speed threshold

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_getsensitivity( int *vert, int *horz, int *threshold )
{

    int m2, m3, m4;

    asm {
        mov     ax, 27
        int     33h
        mov     m2, bx
        mov     m3, cx
        mov     m4, dx
    }

    *vert = m3;
    *horz = m2;
    *threshold = m4;

}

/* -------------------------- Mouse Function #29 ---------------------------

                              mouse_setpage()

   Description:         Sets the video page where mouse cursor appears

   Parameters:
   (input)              crt_page        Video page for mouse cursor

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_setpage( int crt_page )
{

    asm {
        mov     ax, 29
        mov     bx, crt_page
        int     33h
    }

}

void mouse_getpage( int *crt_page )
{

    int m2;

    asm {
        mov     ax, 30
        int     33h
        mov     m2, bx
    }

    *crt_page = m2;

}

/* -------------------------------------------------------------------------

                              mouse_button_pressed()

   Description:         Returns whether a button or combination of buttons
                        has been pressed

   Parameters:
   (input)              button_mask     The mask to use to determine whether
                                        a specific button has been pressed

   Returned:            zero if no button pressed, otherwise returns the
                        following:

   -------------------------------------------------------------------------
*/

int mouse_button_pressed( int button_mask )
{

    int which_button;

    asm {
        mov     ax, 3
        int     33h
        mov     which_button, bx
    }

    return ( which_button & button_mask );

}

/* -------------------------------------------------------------------------

                              mouse_row()

   Description:         Returns current row position of the mouse cursor

   Parameters:          (none)

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_row( int *vert_pos )
{

    int row;

    asm {
        mov     ax, 3
        int     33h
        mov     row, dx
    }

    *vert_pos = row / 8;

}

/* -------------------------------------------------------------------------

                              mouse_col()

   Description:         Returns current column position of the mouse cursor

   Parameters:          (none)

   Returned:            (function returns nothing)

   -------------------------------------------------------------------------
*/

void mouse_col( int *horz_pos )
{

    int col;

    asm {
        mov     ax, 3
        int     33h
        mov     col, cx
    }

    *horz_pos = col / 8;

}

/* ====================================================================== */
