* Function: MARKREC
* Author..: Richard Low
* Syntax..: MARKREC(top, left, bottom, right, output, markkey, field, colors)
* Notes...: Function for cursoring through a box-menu selection of records
*           from the currently selected database, and marking the records
*           to work with by pressing a designated key (default = F9)
* Returns.: A character string of selected record numbers, each eight digits
*           long, delimited with a comma ",", or a null string if Escape
*           was pressed.
*
* Assumes.: Expects to be passed the following parameters:
*
*           p1 = exp<N> - top row of the box contents
*           p2 = exp<N> - top left column of box contents
*           p3 = exp<N> - bottom row of box contents
*           p4 = exp<N> - bottom right column of box contents
*           p5 = exp<C> - field list to be displayed in box
*           p6 = exp<N> - ASCII key value of mark/unmark key (default = F9)
*           p7 = exp<C> - character field name to add to mark list
*           p8 = exp<A> - color settings
*
* Example: records = MARKED( 6, 40, 18, 78, "Fnm+' '+Lnm", -4,  )
*
FUNCTION MARKREC
PARAMETERS p_top,p_left,p_bottom,p_right,p_output,p_markkey,p_mkfield,p_colors
PRIVATE f_lkey, f_lastrec, f_marked, f_count, f_markdata, f_marklen,;
        f_position, f_standard, f_highlite, f_seekstr

*-- verify first 5 parameters given are correct type
IF TYPE('p_top')   + TYPE('p_left') + TYPE('p_bottom') +;
   TYPE('p_right') + TYPE('p_output') != 'NNNNC'
   RETURN 0
ENDIF

p_markkey = IF( TYPE('p_markkey') = 'N', p_markkey, -8 )                 && INKEY() value of F9 key
p_mkfield = IF( TYPE('p_mkfield') = 'C', p_mkfield, ' ' )
p_mkfield = IF( EMPTY(p_mkfield), 'STR(RECNO(),8,0)', p_mkfield )        && default mark field is Record number

*-- save length of a marked data item, plus 1 for the trailing comma
f_marklen = LEN(&p_mkfield) + 1                                          


in_color = SETCOLOR()

*-- use <color array> if it is an array AND it has at least 5 elements
IF IF( TYPE('p_colors') = 'A', IF(LEN(p_colors) >= 5, .T., .F.) , .F. )
   f_display  = p_colors[1]
   f_bright   = p_colors[2]
   f_reverse  = p_colors[3]
   f_revblink = p_colors[4]
ELSE
   f_display  = SETCOLOR()
   f_bright   = BRIGHT(in_color)
   f_reverse  = GETPARM(2,in_color)
   f_revblink = BRIGHT(f_reverse)                                        && puts a '+' at end of forground part
   f_revblink = STUFF( f_revblink, AT('+',f_revblink), 1, '*')           && replace '+' with '*' to make it blinking
ENDIF

SETCOLOR(f_display)

IF LEN(&p_output) != p_right - p_left + 1                             && see if width of output is different from width of box
   IF LEN(&p_output) > p_right - p_left + 1                                    && if wider than box
      p_output = 'SUBSTR(' + p_output + ',1,p_right - p_left + 1)'             && shorten it
   ELSE
      padding = SPACE( p_right - p_left + 1 - LEN(&p_output) )                 && otherwise, pad it with spaces
      p_output = p_output + " + padding"                              && pad output with spaces
   ENDIF
ENDIF

f_lastrec = RECNO()
@ p_top,p_left SAY ' '                                                && put normal video blank, otherwise scroll get reverse
SCROLL( p_top, p_left, p_bottom, p_right, 0 )                         && clear inside of box to be filled with records
mrow = p_top                                                          && set up first row for display
DO WHILE mrow <= p_bottom .AND. (.NOT. EOF())                         && fill box with available records
   @ mrow,p_left SAY &p_output                                        && from database in normal video
   mrow = mrow + 1
   SKIP
ENDDO
mrow = p_top                                                          && set back to first row
GOTO f_lastrec

f_seekstr  = ""
f_marked   = ""                                                        && initialize string to store record nums
f_standard = .F.                                                       && easily identify operation of the MarkDisplay procedure
f_highlite = .T.

DO WHILE .T.
   DO MarkDisplay WITH f_highlite
   f_lkey = INKEY(0)
   DO MarkDisplay WITH f_standard
   f_lastrec = RECNO()

   DO CASE
      CASE f_lkey = 5
         *-- Up Arrow
         f_seekstr = ""
         SKIP -1
         IF BOF()
            GOTO f_lastrec
            LOOP
         ENDIF
         mrow = mrow - 1
         IF mrow < p_top
            SCROLL( p_top, p_left, p_bottom, p_right, -1 )
            mrow = p_top
         ENDIF

      CASE f_lkey = 24
         *-- Down Arrow
         f_seekstr = ""
         SKIP
         IF EOF()
            GOTO f_lastrec
            LOOP
         ENDIF
         mrow = mrow + 1
         IF mrow > p_bottom
            SCROLL( p_top, p_left, p_bottom, p_right, 1 )
            mrow = p_bottom
         ENDIF

      CASE f_lkey = 27
         *-- Escape Key
         f_marked = ""
         EXIT

      CASE f_lkey = 13
         *-- Enter Key
         *-- if no records are marked
         IF LEN(f_marked) = 0
            *-- this is the only one selected, so add it
            f_marked = &p_mkfield + ","
         ENDIF
         DO MarkDisplay WITH f_highlite
         EXIT

      CASE f_lkey = p_markkey
         f_seekstr = ""
         f_markdata = &p_mkfield + ","                                && extract data and add trailing comma
         f_position = AT( f_markdata, f_marked )
         IF f_position = 0                                            && not found in string
            f_marked = f_marked + f_markdata                          && mark/add to string
         ELSE
            f_marked = STUFF(f_marked, f_position, f_marklen, "")     && delete from string
         ENDIF

      CASE f_lkey = 18
         *-- Page Up
         f_seekstr = ""
         f_count = 1
         DO WHILE f_count < p_bottom - p_top + 1 .AND. (.NOT. BOF())
            DO MarkDisplay WITH f_standard
            SKIP -1
            IF BOF()
               GO TOP
               EXIT
            ENDIF
            mrow = mrow - 1
            IF mrow < p_top
               SCROLL( p_top, p_left, p_bottom, p_right, -1 )
               mrow = p_top
            ENDIF
            DO MarkDisplay WITH f_highlite
            f_count = f_count + 1
         ENDDO

      CASE f_lkey = 3
         *-- Page Down
         f_seekstr = ""
         f_count = 1
         DO WHILE f_count < p_bottom - p_top + 1 .AND. (.NOT. EOF())
            DO MarkDisplay WITH f_standard
            SKIP
            IF EOF()
               GO BOTTOM
               EXIT
            ENDIF
            mrow = mrow + 1
            IF mrow > p_bottom
               SCROLL( p_top, p_left, p_bottom, p_right, 1 )
               mrow = p_bottom
            ENDIF
            DO MarkDisplay WITH f_highlite
            f_count = f_count + 1
         ENDDO

      CASE f_lkey = 1
         *-- Home Key
         f_seekstr = ""
         GO TOP
         DO MarkRefresh WITH mrow

      CASE f_lkey = 6
         *-- End Key
         f_seekstr = ""
         f_lkey = 0
         DO WHILE f_lkey = 0 .AND. (.NOT. EOF())
            DO MarkDisplay WITH f_standard
            SKIP
            IF EOF()
               GO BOTTOM
               EXIT
            ENDIF
            mrow = mrow + 1
            IF mrow > p_bottom
               SCROLL( p_top, p_left, p_bottom, p_right, 1 )
               mrow = p_bottom
            ENDIF
            DO MarkDisplay WITH f_highlite
            f_lkey = INKEY()
         ENDDO

      CASE f_lkey > 31 .AND. f_lkey < 127                                 && printable character range
         IF EMPTY(INDEXKEY(0))                                            && if no index is controlling
            LOOP                                                          && skip this proc
         ENDIF
         f_seekstr = f_seekstr + UPPER(CHR(f_lkey))
         SEEK f_seekstr                                                   && seek upper case first
         IF EOF()
            SEEK LOWER(f_seekstr)                                         && try finding lower case match
            IF EOF()
               f_seekstr = ''
               GOTO f_lastrec
               ?? CHR(7)
               LOOP
            ENDIF
         ENDIF
         f_lastrec = RECNO()
         DO MarkRefresh WITH mrow

   ENDCASE
ENDDO
SETCOLOR(in_color)
RETURN f_marked


*----------------------------------------------------------------------------
* Procedure: MarkDisplay
* Notes....: Sub-routine to display the <p_output> in the proper color setting.
* Parameter: Logical True|False indicates if the output display is currently
*            selected or not.  Selected output is displayed in one of two
*            colors different from unselected output.
*
*              Un-selected Un-marked - Standard setting   <f_display >
*              Un-selected Marked    - Bright Standard    <f_bright  >
*              Selected    Un-marked - Enhanced setting   <f_reverse >
*              Selected    Marked    - Blinking Enhanced  <f_revblink>
*----------------------------------------------------------------------------
PROCEDURE MarkDisplay
PARAMETER selected
IF selected
   SETCOLOR( IF( &p_mkfield + "," $ f_marked, f_revblink, f_reverse) )
ELSE
   SETCOLOR( IF( &p_mkfield + "," $ f_marked, f_bright,   f_display) )
ENDIF
@ mrow,p_left SAY &p_output
RETURN



*----------------------------------------------------------------------------
* Procedure: MarkRefresh
* Notes....: Sub-procedure to refresh the entire display box from the current
*            record.  After the display is complete, the record pointer is
*            re-positioned to the incoming record pointer location.
* Assumes..: The record pointer is positioned at the first record to be
*            displayed on th first line of the box.
* Parameter: Gets <mrow> as a parameter to ensure it can change its value.
*----------------------------------------------------------------------------
PROCEDURE MarkRefresh
PARAMETER mrow
PRIVATE inrec
inrec = RECNO()
mrow = p_top
SETCOLOR(f_display)
@ p_top,p_left SAY ' '
SCROLL( p_top, p_left, p_bottom, p_right, 0 )
DO WHILE mrow <= p_bottom .AND. (.NOT. EOF())
   DO MarkDisplay WITH f_standard
   mrow = mrow + 1
   SKIP
ENDDO
mrow = p_top
GOTO inrec
RETURN
