* Function: PICKREC
* Author..: Richard Low
* Syntax..: PICKREC( top, left, bottom, right, output, proc, condition, row )
* Notes...: Function for cursoring through a box-menu selection of records
*           from the currently selected database, and selecting a record
*           to work with by pressing the enter key.
* Returns.: The row number of the selected record,  or zero if the Escape
*           Key was pressed to exit.  If either the insert or delete keys
*           are pressed, the routine exits to the calling procedure which
*           can test for Insert or Delete with the LASTKEY() function.
*
* Assumes.: Expects to be passed the following parameters:
*
*           top       = exp<N> - top row of the box contents
*           left      = exp<N> - top left column of box contents
*           bottom    = exp<N> - bottom row of box contents
*           right     = exp<N> - bottom column of box contents
*           output    = exp<C> - character expression for output display
*           proc      = exp<C> - Optional PROCEDURE to call on each keypress
*           condition = exp<C> - Optional condition expression
*           row       = exp<N> - current row number (used to reposition bar)
*                              = 0 - GO TOP and fill the box with records
*                              < 0 - erase box and re-fresh from current record
*
*           If a parameter is to be skipped, pass a 'dummy' parameter
*           such as a null string in place of the actual parameter.
*
* Ex:  foutput = "Lastname + ', ' + Firstname"
*
*      rownum = PICKREC( 6, 40, 18, 78, foutput, 'REDISPLAY', '', rownum )
*

FUNCTION PICKREC
PARAMETERS p_top, p_left, p_bot, p_rite, p_output, p_proc, p_cond, p_row
PRIVATE do_proc, num_cols, padding, mrec, lkey, counter, f_rowcount,;
        in_color, f_bright, f_reverse, f_seekstr

*-- verify first 5 parameters given are correct type
IF TYPE('p_top')  + TYPE('p_left') + TYPE('p_bot') +;
   TYPE('p_rite') + TYPE('p_output') != 'NNNNC'
   RETURN 0
ENDIF

*-- verify procedure name is a character string
p_proc  = IF( TYPE('p_proc') = 'C', p_proc, '' )
do_proc = (.NOT. EMPTY(p_proc))


*-- verify any condition given is  a character string
p_cond  = IF( TYPE('p_cond') = 'C', p_cond, '.T.' )

*-- and that it evaluates to a logical answer
IF TYPE(p_cond) != 'L'
   p_cond = '.T.'
ENDIF


*-- get incoming color setting and build the bright and reverse settings
in_color  = UPPER(SETCOLOR())
f_bright  = BRIGHT(in_color)
f_reverse = GETPARM(2,in_color)

SETCOLOR(in_color)

num_cols = p_rite - p_left + 1                                    && available width in box
IF LEN(&p_output) > num_cols
   p_output = 'SUBSTR(' + p_output + ',1,num_cols)'                       && shorten output
ENDIF
IF LEN(&p_output) < num_cols
   padding = SPACE( num_cols - LEN(&p_output) )
   p_output = p_output + " + padding"                                     && pad output with spaces
ENDIF

IF TYPE('p_row') != 'N'
   p_row = 0
ENDIF

IF p_row <= 0                                                          && first time being called by proc
   IF p_row = 0
      IF p_cond = '.T.'                                                && if no condition provided
         GO TOP                                                        && go to top of database
      ELSE
         *-- if the current record does not meet the supplied condition
         IF .NOT. &p_cond
            *-- position the record pointer to EOF()
            GO BOTTOM
            SKIP
         ENDIF
         *-- otherwise, find first record meeting the condition specified
         DO WHILE (&p_cond) .AND. (.NOT. BOF())
            mrec = RECNO()
            SKIP-1
            IF BOF() .OR. (.NOT. (&p_cond))
               GOTO mrec
               EXIT
            ENDIF
         ENDDO
      ENDIF
   ENDIF
   mrec = RECNO()                                                     && x marks the spot
   @ p_top,p_left SAY ' '                                             && put normal video blank, otherwise scroll get reverse
   SCROLL( p_top, p_left, p_bot, p_rite, 0 )                          && clear inside of box to be filled with records
   p_row = p_top                                                      && set up first row for display
   DO WHILE p_row <= p_bot .AND. (&p_cond) .AND. (.NOT. EOF())                       && fill box with available records
      @ p_row,p_left SAY &p_output                                    && from database in normal video
      p_row = p_row + 1
      SKIP
   ENDDO
   p_row = p_top                                                      && set back to first row
   GOTO mrec                                                          && go back to where we started
ENDIF

f_rowcount = p_bot - p_top + 1
f_seekstr  = ""                                                       && string to initialize for key searches

DO WHILE .T.
   SETCOLOR(f_reverse)
   @ p_row, p_left SAY &p_output
   SETCOLOR(in_color)

   *-- do routine if it exists and they are not stomping on a key
   IF do_proc .AND. NEXTKEY() = 0
      DO &p_proc
   ENDIF
   mrec = RECNO()
   lkey = INKEY(0)

   DO CASE

      CASE lkey = 5
         *-- Up Arrow
         f_seekstr = ''                                             && cancel current search string
         @ p_row, p_left SAY &p_output
         SKIP-1
         IF BOF() .OR. (.NOT. (&p_cond))
            GOTO mrec
            LOOP
         ENDIF
         p_row = p_row - 1
         IF p_row < p_top
            SCROLL( p_top, p_left, p_bot, p_rite, -1 )
            p_row = p_top
         ENDIF

      CASE lkey = 24
         *-- DownArrow
         f_seekstr = ''                                             && cancel current search string
         @ p_row, p_left SAY &p_output
         SKIP
         IF EOF() .OR. (.NOT. (&p_cond))
            GOTO mrec
            LOOP
         ENDIF
         p_row = p_row + 1
         IF p_row > p_bot
            SCROLL( p_top, p_left, p_bot, p_rite, 1 )
            p_row = p_bot
         ENDIF

      CASE lkey = 27
         *-- EscapeKey
         @ p_row, p_left SAY &p_output
         p_row = 0
         EXIT

      CASE lkey = 13
         *-- EnterKey
         SETCOLOR(f_bright)
         @ p_row, p_left SAY &p_output
         SETCOLOR(in_color)
         EXIT

      CASE lkey = 18
         *-- PageUp
         f_seekstr = ''                                             && cancel current search string
         FOR counter = 1 TO f_rowcount
            @ p_row,p_left SAY &p_output
            mrec = RECNO()
            SKIP-1
            IF BOF() .OR. (.NOT. (&p_cond))
               GOTO mrec
               SETCOLOR(f_reverse)
               @ p_row,p_left SAY &p_output
               SETCOLOR(in_color)
               EXIT
            ENDIF
            p_row = p_row - 1
            IF p_row < p_top
               SCROLL( p_top, p_left, p_bot, p_rite, -1 )
               p_row = p_top
            ENDIF
            SETCOLOR(f_reverse)
            @ p_row,p_left SAY &p_output
            SETCOLOR(in_color)
         NEXT counter

      CASE lkey = 3
         *-- PageDown
         f_seekstr = ''                                             && cancel current search string
         FOR counter = 1 TO f_rowcount
            @ p_row,p_left SAY &p_output
            mrec = RECNO()
            SKIP
            IF EOF() .OR. (.NOT. (&p_cond))
               GOTO mrec
               SETCOLOR(f_reverse)
               @ p_row,p_left SAY &p_output
               SETCOLOR(in_color)
               EXIT
            ENDIF
            p_row = p_row + 1
            IF p_row > p_bot
               SCROLL( p_top, p_left, p_bot, p_rite, 1 )
               p_row = p_bot
            ENDIF
            SETCOLOR(f_reverse)
            @ p_row,p_left SAY &p_output
            SETCOLOR(in_color)
         NEXT counter

      CASE lkey = 1
         *-- Home Key
         f_seekstr = ''                                             && cancel current search string
         IF p_cond = '.T.'
            *-- if no condition supplied, go to top of database
            GO TOP
         ELSE
            *-- otherwise, find first record meeting condition
            DO WHILE (&p_cond) .AND. (.NOT. BOF())
               mrec = RECNO()
               SKIP-1
               IF BOF() .OR. (.NOT. (&p_cond))
                  GOTO mrec
                  EXIT
               ENDIF
            ENDDO
         ENDIF
         *-- now clear window and display records
         mrec = RECNO()
         @ p_top,p_left SAY ' '                                         && put normal video blank, otherwise scroll get reverse
         SCROLL( p_top, p_left, p_bot, p_rite, 0 )                  && clear inside of box to be filled with records
         p_row = p_top
         DO WHILE p_row <= p_bot .AND. (&p_cond) .AND. (.NOT. EOF())
            @ p_row,p_left SAY &p_output
            p_row = p_row + 1
            SKIP
         ENDDO
         p_row = p_top
         GOTO mrec

      CASE lkey = 6
         *-- End Key
         f_seekstr = ''                                             && cancel current search string
         lkey = 0
         DO WHILE lkey = 0 .AND. (&p_cond) .AND. (.NOT. EOF())
            @ p_row,p_left SAY &p_output
            mrec = RECNO()
            SKIP
            IF EOF() .OR. (.NOT. (&p_cond))
               GOTO mrec
               SETCOLOR(f_reverse)
               @ p_row,p_left SAY &p_output
               SETCOLOR(in_color)
               EXIT
            ENDIF
            p_row = p_row + 1
            IF p_row > p_bot
               SCROLL( p_top, p_left, p_bot, p_rite, 1 )
               p_row = p_bot
            ENDIF
            SETCOLOR(f_reverse)
            @ p_row,p_left SAY &p_output
            SETCOLOR(in_color)
            lkey = INKEY()
         ENDDO

      CASE lkey = 22
         *-- Insert Key
         SETCOLOR(in_color)
         @ p_row, p_left SAY &p_output
         EXIT

      CASE lkey = 7
         *-- Delete Key
         EXIT

*      CASE lkey = 28
*         *-- F1 = Help Key
*         DO Help WITH PROCNAME(), PROCLINE(), "LKEY"

      CASE lkey > 31 .AND. lkey < 127                                     && printable character range
         IF EMPTY(INDEXKEY(0))                                            && if no index is controlling
            LOOP                                                          && skip this proc
         ENDIF
         mrec = RECNO()                                                   && save record number
         f_seekstr = f_seekstr + UPPER(CHR(lkey))
         SEEK f_seekstr                                                  && seek upper case first
         IF EOF() .OR. (.NOT. (&p_cond))
            SEEK LOWER(f_seekstr)                                                && try finding lower case match
            IF EOF() .OR. (.NOT. (&p_cond))
               f_seekstr = ''
               GOTO mrec
               ?? CHR(7)
               LOOP
            ENDIF
         ENDIF
         mrec = RECNO()
         @ p_top,p_left SAY ' '                                       && put normal video blank, otherwise scroll get reverse
         SCROLL( p_top, p_left, p_bot, p_rite, 0 )                && clear inside of box to be filled with records
         p_row = p_top                                                   && set up first row for display
         DO WHILE p_row <= p_bot .AND. (&p_cond) .AND. (.NOT. EOF())                     && fill box with available records
            @ p_row,p_left SAY &p_output                                   && from database in normal video
            p_row = p_row + 1
            SKIP
         ENDDO
         p_row = p_top                                                   && set back to first row
         GOTO mrec

   ENDCASE
ENDDO
SETCOLOR(in_color)
RETURN (p_row)
