/*  Recycle.prg

    This program uses the Trigger RDD of the RDDKIT to implement recycling
    of deleted records.  To do this, the Delete and Append Trigger actions
    are used.

    In the Delete Trigger function, the deleted record is cleared and is
    then its key value is assigned low-values to enable it to be easily
    found.

    The Append Trigger function first tries to find a deleted record, and
    if that fails, to allow the record to be conventially appended.


    To compile:     CLIPPER RECYCLE /N

    Author: Tim Beck
    Copyright (C) 1992 Beck Software Systems, All Rights Reserved
*/

#include "trigger.ch"
#include "set.ch"

/* Main()
*
*  Opens the database file and browses it
*/
PROCEDURE Main

SetBlink(.F.)
SetColor("W/B,N/W*,,,B/W")

CLS
DevPos(0,0)
DevOut("Recycle Deleted Record Test", "GR+/B")

// Open file using the Trigger RDD
USE RecyTest VIA "TRIGGER" TRIGGER Recycle NEW
INDEX ON Surname TO RecyTest

// Note that Browse() does not do a refreshAll() after delete, so to view
// the changes, we have supplied a slightly modified version of BROWSE.PRG
// otherwise this must be forced (eg: go eof() and go top)

Browse(3, 2, 23, 76)

RETURN

/* Recycle()
*
*  Main Trigger Function
*/
FUNCTION Recycle (nArea,nAction, nField, xValue)    // Note: NOT static function

STATIC lExpired := .F.
LOCAL lReturn

DO CASE
CASE nAction == T_DEMO
    IF !lExpired
        alert("Trigger Demonstration Limit Exceeded.  Demonstration will";
                        + " Continue without Trigger")
        lExpired := .T.
    ENDIF
    lReturn := .T.
CASE nAction == T_ERROR;        lReturn := .F.
CASE nAction == T_UNKNOWN;      lReturn := .F.
CASE nAction == T_UPDATE;       lReturn := .T.
CASE nAction == T_APPEND;       lReturn := AppendRec()
CASE nAction == T_DELETE;       lReturn := DeleteRec()
CASE nAction == T_RECALL;       lReturn := .T.
CASE nAction == T_PACK;         lReturn := .F.
CASE nAction == T_ZAP;          lReturn := .F.
CASE nAction == T_PUT;          lReturn := .T.
CASE nAction == T_OPEN;         lReturn := .T.
CASE nAction == T_CLOSE;        lReturn := .T.
ENDCASE

RETURN ( lReturn )


/*  Trigger Functions  */

/* AppendRec()
*
*  Appends records by recycling deleted ones
*  We will be using low-values because there is generally more activity at
*       the top of the file than at the bottom.
*/
STATIC FUNCTION AppendRec

LOCAL lAppend := .T., nOrder := IndexOrd()
LOCAL lSetDeleted := Set(_SET_DELETED, .F.)

DbSetOrder(1)                                   // Main key

IF DbSeek(Chr(0)) .and. Deleted()               // Low-values
    // If this record can be locked, recylcle it - don't append
    IF RLock()                                  // DbAppend() always locks
        RECALL
        lAppend := .F.
    ENDIF
ENDIF

Set(_SET_DELETED, lSetDeleted)
DbSetOrder(nOrder)

RETURN ( lAppend )


/* DeleteRec()
*
*  Deletes records and prepares them for recycling
*  In this version we will assume that the main key is always the first
*       field of the record.  In a generic implementation, the key field
*       could be obtained from a data dictionary or, using a class based
*       implementation, obtained from the workarea's instance (object).
*/
STATIC FUNCTION DeleteRec

local nCurrent, aFields := Array(FCount())

// Load the empty fields
nCurrent := RecNo()
DbGoTo( RecCount()+1 )                          // eof()
AEval(aFields, {|e,n| aFields[n] := FieldGet(n)})

// Write empty fields
DbGoTo(nCurrent)
AEval(aFields, {|e,n| FieldPut(n, e)})

// Write Low-Values to the key
FieldPut(1, Chr(0))                             // Assume chracter value!

RETURN ( .T. )                                  // Delete the record

// eof: Recycle.prg
