/*-------------------------------------------------------------------------*

             SWIS -- THE STARBUCK WINDOWED INFORMATION SYSTEM(c)
                        Copyright 1990, 1991, 1992
                  by Starbuck & Staff, Tarzana, California
                SWIS(tm) is the trademark of Starbuck & Staff

                                  NOTICE:

    Place in the public domain by the author on 03/18/92.  No warranties
    of any kind expressed or implied.  Freeware.


*--------------------------------------------------------------------------*

    Title.........:  PUSH-BUTTONS WITH POP-OUT ACTION

    Description...:  Keyboard activated push-buttons which respond to
                     cursor movement, enter-key and trigger key input.
                     When a button is selected, the selection is
                     displayed on screen with a push-down, pop-up action.

    Author........:  Wendy Starbuck


*--------------------------------------------------------------------------*/
/*

  Revision by Rey Bango 70312,2175. 5/20/92

  I just expanded a little on Wendy's initial routines so buttons of any
  length could be specified.  The problem before was that you had to make
  sure that the button strings were the same length.  If they weren't, it
  would cause the offset value to be incorrect and the buttons would not
  be spaced properly.

  I also added the ability to attach an action to a button.  The action is
  is a codeblock that will be evaluated when the button is pressed.  The
  actions are specified are specified in a seperate array.

      Ex:
	  Buttons:   		    Actions:

	  aArray1 := { ;            aArray2 := { ;
		       "Yes", ;                  { || disp_yes() }, ;
		       "No"   :                  { || disp_no() }   ;
		     }                         }


  If you don't want to specify an action for a specific button, simply define
  the action with "" or NIL.  The actions are an optional feature.

  Lastly, I created a UDC to make implementing the pushbuttons a little
  easier.

                                  NOTICE:

  Place in the public domain by the author on 05/20/92.  No warranties
  of any kind expressed or implied.  Freeware.

*/

// Include standard headers
#include "box.ch"
#include "inkey.ch"

// Define constants
#define  K_SPACE  32
#xcommand @<nRow>, <nCol> GET <nSele> AS PUSHBUTTON START WITH <nStart> ;
                          BUTTONS <aButtons> [ ACTIONS <aActions> ] =>  ;
          <nSele> := Button_Horizon( <nRow>, <nCol>, <nStart>, <aButtons>, <aActions> )

func main

    local cSaveColor, nAction := 0, nStart := 1

    clear screen
    set scoreboard off

    cSaveColor := SetColor( "W+/B" )
    @ 0, 0, maxrow(), maxcol() box B_SINGLE + chr(176)
    SetColor( "W+/R" )
    @ 09, 20, 15, 59 box B_SINGLE + space(1)
    @ 11, 25 say "Do you want to push a button?"

    while nAction <> 3 .and. lastkey() <> K_ESC

       @ 12, 27 get nAction as pushbutton start with nStart      ;
    		        buttons { "YES", "MAYBE", "NO" }         ;
    		        actions { { || dispmsg( "Say Yes" )   } ,;
                                  "",                            ;
			          { || getout() } }

       nStart := nAction

    end //while

    SetColor( cSaveColor )
    set cursor on
    clear screen
    quit

return nil


/*-------------------------------------------------------------------------*
    Function....:  Button_Horizon
    Description.:  Display push buttons horizontally.
*--------------------------------------------------------------------------*/

function Button_Horizon ( nRow, nCol, nSelect, aButton, aActions )

    local cColor1, cColor2, cColor3, cSaveColor
    local x, nNum, nOffSet, cSelect, lExit, nPop, cTrigger := ""

    // Setup the button colors
    cSaveColor := setcolor()
    cColor1 := "N/" + Substr( cSaveColor, At( "/", cSaveColor )+1 )
    cColor2 := "W+/G"
    cColor3 := "N/W"

    // Define the trigger keys
    x    := 0
    nNum := len( aButton )
    while nNum > x++
        cTrigger := cTrigger + substr( alltrim( aButton[ x ] ), 1, 1 )
    end

    // Define the controls
    set cursor off

    nOffSet := 0
    cSelect := ""
    lExit   := .F.
    x       := 1

    while .T.

        // Display buttons until end of array
        SetColor( cColor1 )
        if nNum >= x

            // Graphic Style Button
            if nSelect == x
		// This is the highlighted option...
                Button_Push( nRow, nCol, nOffSet, aButton[ x ], cColor2, lExit )
		// Store the selected button's offset position...
                nPop := nOffSet
            else
		// These are the unhighlighted option...
                Button_Push( nRow, nCol, nOffSet, aButton[ x ], cColor3 )
            endif

	    // Modified this so that variable length strings could be used
	    // as buttons.  I just added the old offset value to the new one
	    // that's being created...r.b.
            nOffSet += ( Len( aButton[ x ] ) + 4 )
            x++

        else

            // If a button has been selected, exit
            if lExit
                // Handle pop-out action
                Inkey(.3)
                Button_Push( nRow, nCol, nPop, aButton[ nSelect ], cColor2 )
                Inkey(.1)
		// Make sure that they have passed an action array...
		if len( aActions ) >= nSelect .and. valtype( aActions[ nSelect ] ) == "B"
		   eval( aActions[ nSelect ] )
		endif
                exit
            else
                // Wait for a button press
                cSelect := Chr( Inkey(0) )
            endif

            do case
               case LastKey() == K_ENTER .or. ;
                    LastKey() == K_SPACE
                   lExit := .T.
               case Upper( cSelect ) $ cTrigger // Select trigger key
                   lExit := .T.
                   nSelect := At( Upper( cSelect ), cTrigger )
               case LastKey() == K_LEFT         // Left arrow pressed
                   nSelect := if( nSelect - 1 < 1, nNum, nSelect - 1 )
               case LastKey() == K_RIGHT        // Right arrow pressed
                   nSelect := if( nSelect + 1 > nNum, 1, nSelect + 1 )
               case LastKey() == K_ESC          // Escape out
                   nSelect := 0
                   exit
            endcase

            nOffSet := 0
            x       := 1

        endif

    end

    // Housekeeping
    SetColor( cSaveColor )

return nSelect


/*-------------------------------------------------------------------------*
    Function....:  Button_Push
    Description.:  Button - push button style.
*--------------------------------------------------------------------------*/

function Button_Push ( nRow, nCol, nOffSet, cName, cColor, lExit )

    local nLen := Len( cName )

    cColor := if( cColor == NIL, SetColor(), cColor )
    lExit := if( lExit == NIL, .F., lExit )
    nCol := nCol + nOffSet

    if lExit
        // push state
        @ nRow+1, nCol+2 say space( nLen + 3 )
        @ nRow+2, nCol+2 say space( nLen + 3 )
        SetColor( cColor )
        @ nRow+1, nCol+3 say " " + cName + " "
    else
        // normal state
        @ nRow+1, nCol+nLen+4 say ""
        @ nRow+2, nCol+3 say Replicate( "", nLen + 2 )
        SetColor( cColor )
        @ nRow+1, nCol+2 say " " + cName + " "
    endif

return (.T.)


/* EOF: PUSHBU.PRG -----------------------------------------------------*/

func dispmsg( x )
local cVar := savescreen( 08, 20, 10, 40 )
local cSaveColor := setcolor()
@ 08, 20, 10, 40 box B_SINGLE + space(1) color "W+/G,GR+/G"
@ 09, 22 say x
inkey( 3 )
restscreen( 08, 20, 10, 40, cVar )
setcolor(cSaveColor)
return nil

func getout
local cVar := savescreen( 08, 20, 10, 40 )
local cSaveColor := setcolor()
@ 08, 20, 10, 40 box B_SINGLE + space(1) color "W+/G,GR+/G"
@ 09, 22 say "I'm oughta here!!"
inkey( 3 )
restscreen( 08, 20, 10, 40, cVar )
setcolor(cSaveColor)
return nil
