 /*
   PRB_HOTP.PRG
   
   Author     : Phil Barnett
   
   Written    : 08/02/93
   
   Function   : Data Selector
   
   Purpose    : Pop a window, and allow choice by arrows or typing.
   
   Syntax     : HOTPICK( nRow, nCol, aFieldList, cDbfName, [mIndex], [cBoxColor] )
   
   Parameters : nRow       : is the top row     (of box)

                nCol       : is the left column (of box)

                aFieldList : is the field array to display
                             (in the correct order)
                             (first element must be key of index)

                cDbfName   : is the file to pick from
                             if the table is not already open, it will
                             not be left open.

                mIndex     : is the Index file to use 
                             (only if not already open)
                             if passed as numeric, it specifies the index
                             order to set. If not passed, the keypick
                             function is disabled.

                cBoxColor  : defaults to bright yellow if not passed
   
   Returns    : Selected data or "" if not selected.
   
   Example    : use account

                clear screen

                if !file("ACCT.NTX")
                  INDEX ON upper(BUSINESS) TO ACCT
                ELSE
                  SET INDEX TO ACCT
                ENDIF
                
                choice := hotpick(5,5,{"BUSINESS","ACCOUNT"},"ACCOUNT","ACCT")
   
   Compile    : Clipper PRB_HOTP /N
   
   Warning    : aFieldList[ 1 ] must be the key of the selected index
   
   Released to Public Domain by Author
   
*/

#xcommand DEFAULT <foo> TO <bar> => IF <foo> == NIL ; <foo> := <bar> ; ENDIF

#include "dbstruct.ch"
#include "dbedit.ch"
#include "inkey.ch"
#include "box.ch"

static nShowRow, nShowCol, nMaxShow

*!*****************************************************************************
*!
*!       Function: HOTPICK()
*!
*!*****************************************************************************
function HOTPICK(nRow, nCol, aFieldList, cDbfName, mIndex, cBoxColor)

/*

 function will return '' if selection is not made, and the value of
 the first field in the aFieldList array if selection is made.

*/

local inarea := select()
local nWidth := 0
local lHotPickOK := .F.
local cReturnValue, nPtr, nArea, nBottRow, nRightCol
local x, aStruct, nFieldCount, cSaveScrn
local incolor := setcolor()

if pcount() < 4
  alert( 'HOTPICK: The first four parameters are mandatory.' )
  return ''
endif

default cBoxColor to "gr+/b"

// check to see if table is open, if not open it. If so, select it.

nArea := select( cDbfName )

if nArea == 0
  use ( cDbfName ) new
  if valtype( mIndex ) == 'C'
    set index to ( mIndex )
    lHotPickOK := .T.
  endif
else
  select( nArea )
  if valtype( mIndex ) == 'N'
    set order to ( mIndex )
    lHotPickOK := .T.
  endif
endif

GO TOP

// get the fieldnames from the table

aStruct := dbstruct()

nFieldCount := len( aFieldList )

// validate field list and calculate width of display
// Grab maxshow column width

for x := 1 to nFieldCount  // fields to display, not fields in table
  
  if fieldpos( aFieldList[ x ] ) == 0
    alert( 'Field List element ' + ltrim( str( x ) ) + ' ' + aFieldLIst[ x ] + ' is not in Table' )
    return ''
  endif
  
  if aFieldList[ 1 ] == aStruct[ x, 1 ]
  endif
  
  nWidth += aStruct[ fieldpos( aFieldList[ x ] ), DBS_LEN ]
  
next

nMaxShow := aStruct[ fieldpos( aFieldList[ 1 ] ), DBS_LEN ]

if nFieldCount == 1
  nWidth --
else
  nWidth += ( nFieldCount - 1 ) * 2
endif

nBottRow := min( maxrow(), reccount() + nRow + 2 )
nRightCol := min( maxcol(), nWidth + nCol + 2 )

cSaveScrn := savescreen( nRow, nCol, nBottRow, nRightCol )

setcolor( cBoxColor )

dispbox( nRow, nCol, nBottRow, nRightCol, B_DOUBLE + " " )

nShowRow := nBottRow
nShowCol := nCol

if lHotPickOK
  dbedit( nRow + 1, nCol + 1, nBottRow - 1, nRightCol - 1, aFieldList, "hotpickfun" )
else
  dbedit( nRow + 1, nCol + 1, nBottRow - 1, nRightCol - 1, aFieldList )
endif

restscreen( nRow, nCol, nBottRow, nRightCol, cSaveScrn )

if lastkey() == 13
  cReturnValue := fieldget( fieldpos( aFieldList[ 1 ] ) )
else
  cReturnValue := ""
endif

if nArea == 0
  close
  select( inarea )
endif

setcolor( incolor )

return cReturnValue

**********************************

function hotpickfun(PF1,PF2)

static seekstring := ""

local POINTER, Key, nRec

Key = lastkey()

do case
case PF1 == DE_IDLE
   return( DE_CONT )
case PF1 == DE_EMPTY
   alert( "No Choices Available." )
   seekstring := ""
   return( DE_ABORT )
case Key == K_UP .and. PF1 == DE_HITTOP
   return( DE_CONT )
case Key == K_DOWN .and. PF1 == DE_HITBOTTOM
   return( DE_CONT )
case Key == K_ENTER
   seekstring := ""
   return( DE_ABORT )
case Key == K_ESC
   seekstring := ""
   return( DE_ABORT )
CASE KEY > 31 .and. key < 123
   if len( seekstring ) < nMaxShow
     seekstring += chr( key )
     nRec := recno()
     seek( seekstring )
     if !found()
       seekstring := left( seekstring, len( seekstring ) - 1 )
       go nRec
       return( DE_CONT )
     else
       @ nShowRow,nShowCol say ""+seekstring+""
       return( DE_REFRESH )
     endif
   else
     return( DE_CONT )
   endif
case key == K_BS
   seekstring := left( seekstring, len( seekstring ) - 1 )
   if empty( seekstring )
     go top
   else
     seek( seekstring )
   endif
     @ nShowRow,nShowCol say ""+seekstring+""
   return( DE_REFRESH )
otherwise
   return( DE_CONT )
endcase

RETURN( DE_CONT )
