
* "TESTPACK.PRG"    PackMan(), the "safe" pack facility
* =====================================================
/*
    VERSION 1.0    10-03-91

    Compile with:  /m /n /w [/b /dDEBUG]
                            [to compile test procedure with line numbers]

    BACKGROUND:

    What does PackMan() do?
      It replaces Clipper's PACK, REINDEX, and ZAP commands with functions
      that may be invoked either by command or a "parameter-less" function of
      the same name.  Thus, PackMan() functions may be referenced within
      codeblocks and other constructs requiring function calls.

    Why PackMan()?  What does it do that standard Clipper facilities
    cannot do?

    o  First and foremost, PackMan() is safe.  It will not even begin the
       requested operation until it has determined that the current default
       disk drive has sufficient space available to contain all temporary
       files, at maximum potential size, and warning the user if available
       space may be insufficient.  Should something go awry during the file
       copy process, either the original file or the copy will always be
       available.  PackMan() never deletes the temporary file until the
       related original file has been successfully copied.  As an added
       precaution, PackMan() replaces all index files before it begins to
       replace the database [and memo] file(s).
    o  PackMan() operates on "open" .dbf and .ntx [.ndx] files, just as
       Clipper's related commands do.  But, PackMan() does not really
       "reindex" open index files.  It rebuilds them, using the key
       expressions retrieved from INDEXKEY(n) calls and DBCREATEINDEX().
       With the exception of the UNIQUE attribute, it does not rely on
       the existing index header record.  Should the index header/unique
       attribute be corrupted, PackMan() offers the user the option to
       default, i.e., create a non-unique index.
    o  PackMan() "compresses" database and memo files.  The new database
       file will be "sorted" on the controlling index expression, thus
       improving subsequent performance in many cases.
    o  Clipper's related commands require EXCLUSIVE database use.
       With PackMan(), files may be opened SHARED, as the test program
       illustrates.
    o  PackMan() provides an additional command/function, COMPRESS, which
       allows the user to re-create the database and index files on a
       FOR condition other than the defaults (".not. deleted()" for PACK and
       NIL for REINDEX.)  The condition may be expressed literally or as a
       codeblock.  Of course, this provides a user "hook" into the copy
       process.
    o  PackMan() is self-contained.  It delivers its various error messages
       through the user's ErrorSys.

    USAGE NOTES:  To permit the writing of temporary files on a path other
    than the current DOS subdirectory, PackMan() is coded to be responsive
    to the current SET DEFAULT TO .... condition.   To test this facility,
    TESTPACK.PRG, in its present form, does a SET DEFAULT TO E:, which
    just happens to be a RamDisk on the author's machine.  This spec should,
    of course, be changed (or eliminated) to suit the individual user's
    configuration and preferences.

    Also note that the following preprocessor directives REDEFINE STANDARD
    CLIPPER COMMANDS, permitting PackMan() to fulfill its primary mission
    transparently.  If this is not the user's preference, the command and/or
    function names should be changed accordingly.  If it IS the user's
    preference, the commands and translates should be copied to the user's
    "overall" .CH file to insure their pervasive use.

    Note also that the EXPRNBLOCK pseudo-function and TRUE/FALSE #define's
    are required by the test procedure, as well as PackMan() and its
    subordinate functions.  Thus, should the user decide to discard the
    "PROCEDURE testpack" code, these directives must be retained.  Stated
    another way, you may delete the #ifdef DEBUG and #endif, as well as
    everything in between.

    Special thanks to Ted Means and Jo French for their contributions to the
    development, evolution, and testing of PackMan().

    Chuck Friedel
    CIS [76467,706]

*/

#define   TRUE    .T.
#define   FALSE   .F.

#xtrans   EXPRNBLOCK(<cExpr>)  =>  &("{||"+<cExpr>+"}")

*===========================================

#command  PACK    [RETURN <lRet>]                        => ;
                  [<lRet> :=] PACKMAN( 1 )
#command  REINDEX [RETURN <lRet>]                        => ;
                  [<lRet> :=] PACKMAN( 2 )
#command  ZAP     [RETURN <lRet>]                        => ;
                  [<lRet> :=] PACKMAN( 3 )
#command  COMPRESS FOR <cCond> [RETURN <lRet>]           => ;
                  [<lRet> :=] PACKMAN( 4, <{cCond}> )
#command  COMPRESS [FOR] BLOCK <bCond> [RETURN <lRet>]   => ;
                  [<lRet> :=] PACKMAN( 4, <bCond> )

#xtrans   PACK()                   => PACKMAN( 1 )
#xtrans   REINDEX()                => PACKMAN( 2 )
#xtrans   ZAP()                    => PACKMAN( 3 )
#xtrans   COMPRESS( <bCopyFor> )   => PACKMAN( 4, <bCopyFor> )

*===========================================

*===========*
#ifdef DEBUG
*===========*

*-----------------
PROCEDURE testpack
*-----------------

LOCAL cOrigDbf, cOrigDbt, cFilePfx, cTestExpr, nAltFile, lStatus

cOrigDbf := "roster.dbf"
cOrigDbt := "roster.dbt"

SET DELETED off

USE  ( cOrigDbf )  EXCLUSIVE  NEW

/*  test use of default path  */
cFilePfx := "E:"                      // CHANGE THIS ??
SET DEFAULT TO  ( cFilePfx )

/*  let's not to screw around with the original file  */
COPY TO  testcopy.dbf
USE

IF ( nAltFile := ALERT( "Write Audit Report to TestPack.Log file?", { "Yes", "No" } ) ) = 1
   SET ALTERNATE TO  TestPack.Log
   SET ALTERNATE on
ENDIF

USE  testcopy.dbf  SHARED  NEW
/*  build some moderately useless indexes  */
INDEX ON  FIELD->name_last   TO testntx1
INDEX ON  FIELD->name_first  TO testntx2
INDEX ON  FIELD->city        TO testntx3  UNIQUE

SET INDEX TO testntx1, testntx2, testntx3

/*  test to insure new indexes retain original UNIQUE attributes  */
SET( _SET_UNIQUE, .T. )

/*  make sure we have some deleted records  */
FLOCK()
DELETE  FOR  FIELD->name_last > "M"
DBUNLOCK()

GO TOP
DBEVAL( { || QOUT( FIELD->name_last, FIELD->name_first, FIELD->city ) } )
?
? "  Records in orginal Database:", LASTREC()
? "  Original Database File Size: ", FILESIZE( cOrigDbf )
? "      Original Memo file Size: ", FILESIZE( cOrigDbt )
? REPLICATE( "-", 50 )

/*  first, let's try a "REINDEX" (and compress) only  */
lStatus := REINDEX()             //REINDEX RETURN lStatus
STATUS( cFilePfx, "REINDEX", lStatus )

USE  testcopy.dbf  SHARED  NEW
SET INDEX TO testntx1, testntx2, testntx3

/*  now, a "PACK" -- go get 'em, Bigmouth  */
PACK RETURN lStatus              //lStatus := PACK()
STATUS( cFilePfx, "PACK", lStatus )

USE  testcopy.dbf  SHARED  NEW
SET INDEX TO testntx1, testntx2, testntx3

/*  let's try a "Copy For" condition  */
//COMPRESS FOR  name_last > "F"  RETURN lStatus

/*  ...but, an end-user specified expression is more fun  */
cTestExpr := "name_last > 'F'"
//COMPRESS FOR BLOCK EXPRNBLOCK( cTestExpr )  RETURN lStatus

lStatus := COMPRESS( EXPRNBLOCK( cTestExpr ) )
STATUS( cFilePfx, "COMPRESS", lStatus )

USE  testcopy.dbf  SHARED  NEW
SET INDEX TO testntx1, testntx2, testntx3

/*  ok, the party's almost over - let's "ZAP" the thing  */
lStatus := ZAP()                 //ZAP RETURN lStatus
STATUS( cFilePfx, "ZAP", lStatus )

/*  clean 'er up and head for the barn  */
FERASE( cFilePfx + "testcopy.dbf" )
FERASE( cFilePfx + "testcopy.dbt" )
FERASE( cFilePfx + "testntx1.ntx" )
FERASE( cFilePfx + "testntx2.ntx" )
FERASE( cFilePfx + "testntx3.ntx" )

IF nAltFile = 1
   SET ALTERNATE off
   CLOSE ALTERNATE
   IF( ALERT( "Delete TestPack.Log file?", { "Yes", "No" } ) = 1,;
      FERASE( cFilePfx + "TestPack.Log" ), )
ENDIF

QUIT

RETURN

*--------------
STATIC FUNCTION status ( cFilePfx, cOperation, lStatus )
*--------------

IF lStatus
   GO TOP
   ?
   DBEVAL( { || QOUT( FIELD->name_last, FIELD->name_first, FIELD->city ) } )
   ?
   ? "  After " + cOperation + "  Number of Records:", LASTREC()
   USE
   ? "  After " + cOperation + " Database File Size: ", FILESIZE( cFilePfx + "testcopy.dbf" )
   ? "  After " + cOperation + "     Memo File Size: ", FILESIZE( cFilePfx + "testcopy.dbt" )
   ? REPLICATE( "-", 50 )
 ELSE
   USE
   ? "Error ocurred during " + cOperation + "...Operation not completed"
ENDIF

RETURN NIL

*=====*
#endif
*=====*

*==========================================

*---------------
FUNCTION PackMan ( nCommand, bCopyFor )
*---------------
/*
nCommand:
  1 = "PACK" the dbf in the current workarea
  2 = "REINDEX" (copy/compress) the database file
  3 = "ZAP" (remove all records from) the active database file
  4 = "COMPRESS" (copy) the database,
             adding the current record if the "bCopyFor" block returns TRUE
bCopyFor (optional):
  Codeblock required for, and used in conjunction with, COMPRESS (only).

  Note:  In all cases, the database is copied in primary index order (sorted)
         and compressed.  In addition, all active index files are rebuilt.
*/
*---------------

#include  "FILEIO.ch"

#define  MAX_NTXS      15
#define  NDX_UNIQUE    23       // header record offset
#define  NTX_UNIQUE   278       // header record offset
#define  NTX_MINLEN  2048
#define  DBT_MINLEN   512
#define  BLOCKSIZE   1024

LOCAL cFilePfx, cTmpFile, c1, c2, aFiles[0],;
      nOrgHandle, nLenFiles, n1, n2

IF( .NOT. ( cFilePfx := SET( _SET_DEFAULT ) ) == "",;
   IF( RAT( "\", cFilePfx ) > 0,;
      IF( .NOT. RIGHT(cFilePfx, 1) == "\", cFilePfx += "\", ), ), )
cFilePfx += "PAKMAN"

BEGIN SEQUENCE

   /*  grab the file, if we can  */
   FLOCK()

   nLenFiles := 0; n1 := 0; c2 := " "

   DO WHILE .NOT. ( ++n1 > MAX_NTXS .OR. EMPTY( ( c1 := INDEXKEY( n1 ) ) ) )
      cTmpFile := LTRIM( STR( n1 ) )
      IF( .NOT. ( nOrgHandle := NTXHANDLE( n1 ) ) > 0,;
         ERROR( "Retrieving Index File " + cTmpFile + " Handle" ), )
      n2 := FSEEK( nOrgHandle, 0, FS_RELATIVE )
      FSEEK( nOrgHandle, IF( INDEXEXT() == ".NDX", NDX_UNIQUE, NTX_UNIQUE ), FS_SET )
      FREAD( nOrgHandle, @c2, 1 )
      FSEEK( nOrgHandle, n2, FS_SET )
      IF( .NOT. ( FERROR() = 0 .AND. c2 $ CHR(0) + CHR(1) ),;
         ERROR( "Reading Index File " + cTmpFile + " Header",,, TRUE ), )
      cTmpFile  := cFilePfx + cTmpFile + INDEXEXT()
      nLenFiles += IF( nCommand = 3, NTX_MINLEN, FILESIZE( cTmpFile, nOrgHandle ) )
      AADD( aFiles, { cTmpFile, nOrgHandle, c1, IF( c2 == CHR(1), TRUE, FALSE ) } )
   ENDDO

   IF( .NOT. ( nOrgHandle := DBFHANDLE() ) > 0,;
      ERROR( "Retrieving DBF Handle" ), )

   nLenFiles += IF( nCommand = 3, HEADER(), FILESIZE( cTmpFile, nOrgHandle ) )
   IF( ( n2 := DBTHANDLE() ) > 0,;
      nLenFiles += IF( nCommand = 3, DBT_MINLEN, FILESIZE( cTmpFile, n2 ) ), )

   /*  compare available space to aggregate files length  */
   n1 := DISKSPACE( IF( AT( ":", cFilePfx ) = 2,;
         ASC( UPPER( LEFT( cFilePfx, 1 ) ) ) - 64, ) )
   IF( n1 < nLenFiles,;
      ERROR( "Disk Space May be Insufficient",,;
              LTRIM( STR( n1 ) ) + " Free, " +;
              LTRIM( STR( nLenFiles ) ) + " Reqd", TRUE,, 1 ), )
   n1 := 15
   DO WHILE .NOT. FLOCK()
      /*  can't proceed without lock  */
      INKEY(2)
      IF( n1 > 0, --n1,;
        ( ERROR( "Cannot Lock Database File",,,, TRUE ), n1 := 15 ) )
   ENDDO

   n1 := SELECT()
   cTmpFile := cFilePfx + ".dbf"

   IF nCommand = 3
      /*  "zap"  */
      COPY STRUCTURE TO  ( cTmpFile )
    ELSE
      /*  NOTE DIRECT USE of CLIPPER INTERNAL !!  */
      __DBCOPY( cTmpFile,, ;
                IF( nCommand = 1, { || .NOT. DELETED() }, ;
                IF( nCommand = 4, bCopyFor, ) ) )
   ENDIF

   USE  ( cTmpFile )  EXCLUSIVE  NEW

   AEVAL( aFiles, { |aRow, nEl, cKey| cKey := aRow[3],;
            DBCREATEINDEX( aRow[1], cKey, EXPRNBLOCK( cKey ), aRow[4] ) } )
   USE
   SELECT  ( n1 )

   AADD( aFiles, { cTmpFile, nOrgHandle } )

   cTmpFile := cFilePfx + ".dbt"
   IF( FILE( cTmpFile ),;
      ( IF( n2 < 1, ERROR( "Retrieving DBT Handle" ), ),;
        AADD( aFiles, { cTmpFile, n2 } ) ), )

   AEVAL( aFiles, { |aRow| COPYFILE( aRow[1], aRow[2] ) } )

   LASTREC()
   DBUNLOCK()

RECOVER
   DBUNLOCK()
   RETURN ( FALSE )

END   // sequence

RETURN ( TRUE )

*--------------
STATIC FUNCTION filesize ( cFile, nHandle )
*--------------

LOCAL nOrg, nSize
nSize := 0

IF nHandle = NIL
   /*  file is closed  */
   IF OPENFILE( cFile, @nHandle )
      nSize := FSEEK( nHandle, 0, FS_END )
      CLOSEFILE( cFile, @nHandle )
   ENDIF
 ELSE
   /*  file is open  */
   nOrg  := FSEEK( nHandle, 0, FS_RELATIVE )
   nSize := FSEEK( nHandle, 0, FS_END )
   FSEEK( nHandle, nOrg, FS_SET )
ENDIF

RETURN ( nSize )

*--------------
STATIC FUNCTION CopyFile ( cTmpFile, nOrgHandle )
*--------------
*  copy temporary file to original, using specified handle
*--------------

LOCAL cBuffer := SPACE( BLOCKSIZE )
LOCAL nTmpHandle, nBytesRead := 1

OPENFILE( cTmpFile, @nTmpHandle )

FSEEK( nOrgHandle, 0, FS_SET )
DO WHILE nBytesRead > 0
   nBytesRead := FREAD( nTmpHandle, @cBuffer, BLOCKSIZE )
   IF( FERROR() = 0,;
      IF( FWRITE( nOrgHandle, cBuffer, nBytesRead ) < nBytesRead,;
         ERROR( "Copying to Original File", cTmpFile ), ),;
      ERROR( "Reading from Temporary File", cTmpFile ) )
ENDDO

IF( CLOSEFILE( cTmpFile, nTmpHandle ),;
   IF( FERASE( cTmpFile ) # 0,;
      ERROR( "Erasing Temporary File", cTmpFile ), ), )

RETURN NIL

*--------------
STATIC FUNCTION error ( cDesc, cFile, cOper, lDefault, lRetry, nSeverity )
*--------------
LOCAL oErr

IF( cDesc = NIL, cDesc := "", )
IF( cFile = NIL, cFile := IF( EMPTY( cOper ), ALIAS(), "" ), )
IF( cOper = NIL, cOper := "", )
IF( lDefault = NIL, lDefault := FALSE, )
IF( lRetry = NIL, lRetry := FALSE, )
IF( nSeverity = NIL, nSeverity := 2, )

oErr             := ERRORNEW()
oErr:SubSystem   := "PackMan"
oErr:SubCode     := FERROR()
oErr:Description := cDesc
oErr:Filename    := cFile
oErr:Operation   := cOper
oErr:CanDefault  := lDefault
oErr:CanRetry    := lRetry
oErr:Severity    := nSeverity

EVAL( ERRORBLOCK(), oErr )

RETURN ( TRUE )

*--------------
STATIC FUNCTION OpenFile ( cFile, nHandle, nAccess )
*--------------

IF( nAccess = NIL, nAccess := FO_READ + FO_EXCLUSIVE, )

nHandle := FOPEN( cFile, nAccess )

RETURN ( IF( FERROR() = 0, TRUE, ERROR( "Opening File", cFile ) ) )

*--------------
STATIC FUNCTION CloseFile ( cFile, nHandle )
*--------------
LOCAL lRetVal

IF VALTYPE( nHandle ) == "N" .AND. .NOT. nHandle < 0
   FCLOSE( nHandle )
   lRetVal := IF( FERROR() = 0, TRUE, ERROR( "Closing File", cFile ) )
 ELSE
   lRetVal := FALSE
ENDIF

RETURN ( lRetVal )

