/*
 * File......: OUTPUTM.PRG
 * Author....: Berend M. Tober
 * CIS ID....: 70541,1030
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 * 
 * This is an original work by Berend M. Tober and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

/*  $DOC$
 *  $FUNCNAME$
 *     FT_OUTPUTM()
 *  $CATEGORY$
 *     To be assigned
 *  $ONELINER$
 *     Sends reports to screen/file/printer
 *  $SYNTAX$
 *     FT_OUTPUTM( <bReport>, [<cColors>] ) -> NIL
 *  $ARGUMENTS$
 *     <bReport>  - Codeblock which calls your function to perform
 *                  report.
 *     <cColors>  - Color string for destination menu.
 *  $RETURNS$
 *     NIL
 *  $DESCRIPTION$
 *     |--------------------------------------------------------------|
 *     This module is useful when you wish to create applications that
 *     produce reports and you wish to optionally send those reports
 *     to the screen, a disk file, or the printer.
 *
 *     FT_OUTPUTM pops up an screen-centered ACHOICE menu listing the
 *     three ouput destinations.  Depending on the user's selection
 *     ouput produced by the call to your report (via the codeblock)
 *     is sent to different places.
 *  $EXAMPLES$
 *
 *     // As a funtion call
 *     lFinished := FT_OUTPUTM()
 *
 *  $SEEALSO$
 *  $INCLUDE$
 *     box.ch
 *     inkey.ch
 *     common.ch
 *     outputm.ch
 *     setcurs.ch
 *  $END$
 *
 */

#include "box.ch"
#include "inkey.ch"
#include "common.ch"
#include "setcurs.ch"

/*
// File:    OUTPUTM.CH   // Command for FT_OUTPUTM()
// Author:  Berend M. Tober
// Date:    1994/07/12
#ifndef _OUTPUTM_CH

   #xcommand OUTPUTM BLOCK <b> [COLORS <c>] TO <r>;
             => <r> := FT_OUTPUTM(  <b>, <c> )

   #xcommand OUTPUTM BLOCK <b> [COLORS <c>] ;
             => FT_OUTPUTM(  <b>, <c> )

   #define _OUTPUTM_CH

#endif
*/

ANNOUNCE CLIPPER501

#ifdef FT_TEST

#include "outputm.ch"
PROCEDURE T_OUTPUTM    // Sample program
   * Example #1 - Command invocation
   OUTPUTM BLOCK {||Report1()}

   * Example #2 - Function call invocation
   ft_outputm({||Report2()})
   RETURN

STATIC FUNCTION Report1()
   @ 1,0 SAY "SAMPLE REPORT #1"
   @ 3,0 SAY "The OUTPUTM command really provides a substantial"
   @ 4,0 SAY "amount of flexibilily in how you use it."
   ?
   RETURN ALERT("Done")

STATIC FUNCTION Report2()
   @ 1,0 SAY "SAMPLE REPORT #2"
   @ 3,0 SAY "These two examples, however, were trivial..."
   RETURN ALERT("Done")

#endif

********************************* FT_OUTPUTM() *********************************
FUNCTION FT_OUTPUTM( bReport, cColors )
   * Prompts user for output destination of report information

   LOCAL nCursor  := SETCURSOR( SC_NONE )
   LOCAL cHeader  := "Select output destination..."
   LOCAL cFooter  := "Press <ESC> to exit"
   LOCAL nChoice  := 0

   LOCAL aMenuItems :=;
   {;
   "Screen"    ,;
   "Disk File" ,;
   "Printer"    ;
   }

   LOCAL aMenuBlocks :=;
   {;
   {|| _ftToSCR( bReport )},;
   {|| _ftToFIL( bReport )},;
   {|| _ftToPRN( bReport )} ;
   }

   // Center menu on screen
   LOCAL nHigh := LEN( aMenuItems )
   LOCAL nWide := MAX(11,MAX(LEN(cHeader),LEN(cFooter)))
   LOCAL nBoxT := INT((MAXROW()-nHigh)/2)
   LOCAL nBoxL := INT((MAXCOL()-nWide)/2)
   LOCAL nBoxB := nBoxT + nHigh + 1
   LOCAL nBoxR := nBoxl + nWide + 1

   DEFAULT bReport TO {|| TRUE }
   DEFAULT cColors TO SETCOLOR("N/W, W/N")

   @ nBoxT-2, nBoxL+0, nBoxT+0, nBoxR BOX B_SINGLE
   @ nBoxB+0, nBoxL+0, nBoxB+2, nBoxR BOX B_SINGLE

   @ nBoxT-1, nBoxL+1 SAY PADC(cHeader, nWide )
   @ nBoxB+1, nBoxL+1 SAY PADC(cFooter, nWide )
   @ nBoxT++, nBoxL++, nBoxB--, nBoxR-- BOX "Ĵó "

   nChoice := ACHOICE( nBoxT, nBoxL, nBoxB, nBoxR, aMenuItems )

   IF nChoice != 0
      EVAL( aMenuBlocks[nChoice] )
   ENDIF

   SETCURSOR( nCursor )
   SETCOLOR( cColors )
   RETURN NIL
*  end of FT_OUTPUTM()

************************** STATIC FUNCTION _ftToSCR() **************************
STATIC FUNCTION _ftToSCR( bReport )
   * Sends report info to console
   LOCAL lBlink   := SETBLINK( .F. )
   LOCAL cColor   := SETCOLOR("N/W*")
   LOCAL nBoxT    := 0, nBoxL := 0, nBoxB := MAXROW(), nBoxR := MAXCOL()
   LOCAL cMsg     := "Use arrow keys to navigate.  <ESC> to quit"
   LOCAL cFile    := FT_TEMPFIL(".\")

   SET CONSOLE OFF
   SET PRINTER TO (cFile)
   SET PRINTER ON
   SET DEVICE TO PRINTER

   EVAL( bReport )

   cFile := SET(_SET_PRINTFILE)
   SET DEVICE TO SCREEN
   SET PRINTER TO
   SET PRINTER OFF
   SET CONSOLE ON

   DISPBOX( nBoxT++, nBoxL++, nBoxB--, nBoxR--, SPACE(8), "W/B")
   DISPBOX( nBoxT--, nBoxL++, nBoxB++, nBoxR--,   SPACE(9), "N/W*")
   @ nBoxB, nBoxL SAY PADC(cMsg, nBoxR )  COLOR "W/B"

   MEMOEDIT(MEMOREAD(cFile), ++nBoxT, nBoxL, --nBoxB, nBoxR, .F.)
   ERASE (cFile)

   SETCOLOR( cColor )
   SETBLINK( lBlink )

   RETURN NIL
*  end of STATIC FUNCTION _ftToSCR()

************************** STATIC FUNCTION _ftToFIL() **************************
STATIC FUNCTION _ftToFIL( bReport )
   * Sends report info to file
   #define MSG_FILENAME "Enter destination file name: "
   LOCAL cFile    := SPACE(32)
   LOCAL nHigh    := 1
   LOCAL nWide    := LEN(MSG_FILENAME+cFile)+2
   LOCAL nBoxT    := INT((MAXROW()-nHigh)/2)
   LOCAL nBoxL    := INT((MAXCOL()-nWide)/2)
   LOCAL nBoxB    := nBoxT + nHigh + 1
   LOCAL nBoxR    := nBoxl + nWide + 1
   LOCAL GetList  := {}

   DO WHILE ( ALLTRIM(cFile) == "" ) .AND. ( LASTKEY() != K_ESC )
      DISPBOX( nBoxT++, nBoxL++, nBoxB--, nBoxR--, B_SINGLE+" ")
      @ nBoxB, nBoxL SAY MSG_FILENAME GET cFile
      READ
   ENDDO
   IF LASTKEY() <> K_ESC
      cFile := ALLTRIM( cFile )

      SET CONSOLE OFF
      SET PRINTER TO (cFile)
      SET PRINTER ON
      SET DEVICE TO PRINTER

      EVAL( bReport )

      SET DEVICE TO SCREEN
      SET PRINTER TO
      SET PRINTER OFF
      SET CONSOLE ON

      ALERT("Application printed to file "+cFile)
   ENDIF
   RETURN NIL
*  end of STATIC FUNCTION _ftToFIL()

************************** STATIC FUNCTION _ftToPRN() **************************
STATIC FUNCTION _ftToPRN( bReport )
   * Sends report info to printer

   SET CONSOLE OFF
   SET PRINTER ON
   SET PRINTER TO
   SET DEVICE TO PRINTER
   DO WHILE (LASTKEY(0) <> K_ESC) .AND. !ISPRINTER()
      ALERT("PRINTER NOT READY")
   ENDDO
   IF LASTKEY() <> K_ESC
      EVAL( bReport )
      EJECT
      ALERT("Done")
   ENDIF
   SET CONSOLE ON
   SET PRINTER OFF
   SET DEVICE TO SCREEN
   RETURN NIL
*  end of STATIC FUNCTION _ftToPRN()

