/****
*   Program Name : OPEN.PRG                                                            
*
*   Authors      : Barry Ehret, Michael Abadjiev
*   Last Update  : 05/12/92                                                       
*
*   Language      : Clipper 5.01
*   Purpose       : Open files & optionally recreate, network compatible.
*
*   Compile       : Use the OPEN.RMK 
*                   RMAKE OPEN.RMK         - Blinker is the default linker
*                   RMAKE OPEN.RMK /dRTL   - Rtlink will link.
*
*   Functions:
*       OpenDbf()   --> LOGICAL
****/

/* -=- NOTE:

   Function assumes:
      1. Never indexing in SHARE mode.
      2. Never recreate .DBF in SHARE mode.
      3. Function checks data integrity of .DBF and .NTX files
      4. Function handles the problem running out of DOS file handles.
      5. Function we look at the file and if it's allready opened,
         first will close it and then open it again.
      6. I'm using this function all the time it'written by me and
         Barry Ehret who I want to thank.

      SPECIAL NOTE:
      If you find the function usefull ..../no no no no donations/
      go ahead feel free to use it.

      Thanks,
      Michael

***/

#include "fileio.ch"

#xcommand DEFAULT <var> TO <val>        ;
=>  if valtype(<var>) <> valtype(<val>) ;
    ; <var> := <val>                    ;
    ; endif

#xcommand BEEP ERROR    ;
=>  if set(_SET_BELL)   ; 
    ; tone(1200, .1)    ;
    ; tone(1400, .1)    ;
    ; tone(1600, .1)    ;
    ; endif

#xcommand BEEP ATTENTION    ;
=>  if set(_SET_BELL)       ;
    ; tone(1600, .1)        ;
    ; tone(1600, .1)        ;
    ; endif

#define NO_ERROR          0
#define DBF_ERROR       100
#define NTX_ERROR       101
#define DBT_ERROR       102
#define INDEXKEY_ERR    103
#define PROGRAMMER_ERR 1000


STATIC nPercent, aGauge


/*---------------------------- Module Test Code ------------------------------*/

// To test: uncomment the following line & recompile.
//#define TEST
#ifdef TEST

#include "open.ch"

FUNCTION Test()

    LOCAL aSplit, ok, aOk := {}

    set bell on
    set cursor off
    cls

    open test new index TEST1 key "NAME+PHONE"     ;
       unique UNIQUE                               ;
       structure {{"NAME", "C", 30, 0}, {"PHONE", "C", 10, 0}} ;
       result ok

    IF !Ok       
       aadd(aOk,ok)
    ENDIF

    open Junk1 new indexlist {"junk1"} keylist {"(WORKSHEET)"}     ;
       structure {|| GetStruct()}                                     ;
       result ok

    IF !Ok       
       aadd(aOk,ok)
    ENDIF

    open Junk2 new index junk2 key "(WORKSHEET)"                      ;
       structure  {{"WORKSHEET", "C", 6, 0}, {"PHONE", "C", 10, 0}}   ;
       result ok

    IF !Ok       
       aadd(aOk,ok)
    ENDIF

    dbcloseall()
    IF file("junk1.ntx")
       ferase("junk1.ntx")
    ENDIF           

    IF file("junk2.ntx")
       ferase("junk2.ntx")
    ENDIF


    alert(IF(ascan(aOk,.f.) == 0, "OPEN test passed", "OPEN test failed"))

    set cursor on

RETURN nil

STATIC FUNCTION GetStruct()

RETURN {{"WORKSHEET", "C", 6, 0}, {"PHONE", "C", 10, 0}}

#endif

/*---------------------------- End of Test Code ------------------------------*/


/****
*   Function OpenDbf() --> LOGICAL
*
*   Purpose: Try to open a database in exclusive or shared mode
*            on a network. On failure, return false & reset work
*            areas to original status. Optionally recreate missing
*            database files, or index files.    
****/

FUNCTION OpenDbf( lNew, cFile, cAlias, lShared, lReadOnly, ;
                            aIndex, aExprs, struct, lQuiet, aUnique )

    LOCAL   lResult := .t.,        ;
            i,                     ;
            nIndexes,              ;
            cMsg,                  ;
            OldError,              ;
            aStruct,               ;
            cPath,                 ;
            cName,                 ;  
            aIndErr := {},         ;
            oldArea := select(),   ;
            nError,                ;
            nHandle,               ;
            bTest, errobj1

    // set local error handler
    OldError := errorblock({|| break("")})

    BEGIN SEQUENCE

        // validate parameters
        IF valtype(cFile) <> "C" .or. empty(cFile)
            BREAK "ERROR: An invalid filename specified!"
        ENDIF
        cFile := upper(trim(cFile))
        cName := GetFileName(cFile)
        default cAlias to cName
        cAlias := upper(trim(cAlias))
        default lQuiet to .f.
        default aIndex to {}

        nIndexes := len(aIndex)
        IF valtype(aExprs) <> "A" .or. len(aIndex) <> len(aExprs)
            aExprs := array(nIndexes)
         ENDIF

        /*-------------------------------------------------------------*/
        // UNIQUE index support...
        // Attention DON'T change the order...!
        IF valtype(aUnique) == "A"
           FOR i := 1 TO len(aUnique)
              IF valtype(aUnique[i]) == "C"
                 IF alltrim(upper(aUnique[i])) == "UNIQUE"
                    aUnique[i] := .t.
                 ELSE
                    aUnique[i] := nil
                 ENDIF
              ENDIF
           NEXT
        ENDIF            
        IF valtype(aUnique) <> "A" .or. len(aIndex) <> len(aUnique)
            aUnique := array(nIndexes)
        ENDIF
        /*-------------------------------------------------------------*/

        // Check DBF existance & integrity
        IF (nError := CheckDbf(cFile)) <> 0

            // failed in share mode: must abort
            IF !set(_SET_EXCLUSIVE)
                BREAK cFile + ".DBF; ERROR:" + GetDosErr(nError)

            // failed in exclusive mode: attempt to recreate
            ELSEIF valtype(struct) $ "AB"

                // allow confirmation
                IF !lQuiet
                    beep attention
                    IF alert("ERROR: "+ cFile +".DBF;"+;
                    GetDosErr(nError) + ";Create " + cFile +".DBF"+ ;
                    " - are you sure?", {"No", "Yes"}) <> 2
                        BREAK nil
                    ENDIF
                ENDIF

                // get the file structure
                IF valtype(struct) == "B"
                    aStruct := eval(struct)
                ELSEIF valtype(struct) == "A"
                    aStruct := struct
                ENDIF

                // attempt to create the file
                IF !CreateDbf(cPath, cFile, aStruct)
                    BREAK "ERROR: Unable to recreate " + cFile + "!"
                ENDIF
            ELSE
                BREAK "ERROR: Unable to open " + cFile + "!"
            ENDIF
         ENDIF
        // Check out for the file handle available
        IF (nHandle := fopen("NUL",0)) < 0
           BREAK "ERROR: Not enough file handles!"
        ELSE
           fclose(nHandle)
        ENDIF

        // prevent twice opening the same file
        IF select(cFile) > 0
           close cFile
        ENDIF

        // now, attempt to open the database
        dbusearea(lNew,, cFile, cAlias, lShared, lReadonly)
        IF neterr() .or. alias() <> cAlias
            BREAK "ERROR: Unable to open " + cFile + "!"
        ENDIF

        // attempt to open each index
        FOR i = 1 TO nIndexes
            // Check NTX existance & integrity
            IF (nError := CheckNtx(aIndex[i], aExprs[i],, cPath)) <> 0
                IF !set(_SET_EXCLUSIVE) .or. valtype(aExprs[i]) <> "C"
                    BREAK aIndex[i] + ".NTX ; ERROR:" + GetDosErr(nError)
                ENDIF
                aadd(aIndErr, {aIndex[i], aExprs[i], aUnique[i]})
            ENDIF
        NEXT

        // failed ... recreate if in single user mode
        IF len(aIndErr) <> 0
            FOR i = 1 TO len(aIndErr)
               // Check out for the file handle available
               IF (nHandle := fopen("NUL",0)) < 0
                  BREAK "ERROR: Not enough file handles!"
               ELSE
                  fclose(nHandle)
               ENDIF

/*------------------------------------------------------------------------*/
//
//               IF !CheckExpr(aIndErr[i, 2],.f.)
//                  BREAK "ERROR: Wrong Index Expression was passed!;" +;
//                        GetDosErr(1000)
//               ENDIF
/*-------------------------------------------------------------------------*/


               IF !CreateNtx(aIndErr[i, 1], cPath, aIndErr[i, 2],aIndErr[i,3])
                    BREAK "ERROR: Unable to recreate " + aIndex[i] + "!"
               ENDIF                                                   
            NEXT    
            aIndErr := {}
            dbclearindex()      
        ENDIF

         // ok, open the index
        FOR i = 1 TO nIndexes
            // Check out for the file handle available
            IF (nHandle := fopen("NUL",0)) < 0
               BREAK "ERROR: Not enough file handles!"
            ELSE
               fclose(nHandle)
            ENDIF
            dbsetindex(aIndex[i])
            IF neterr()
                BREAK "ERROR: Unable to open " + aIndex[i] + "!"
            ENDIF
        NEXT

    RECOVER USING cMsg

        IF !lQuiet .and. valtype(cMsg) == "C"
            IF empty(cMsg)
                cMsg := "An unknown error occurred;" + ;
                        "while opening " + cFile + ".DBF !"
            ENDIF
            beep error
            alert(cMsg)
        ENDIF
        select(OldArea)
        IF select(cFile) > 0
            close (cFile)
        ENDIF
        lResult := .f.

    END SEQUENCE

    // reset error handler
    errorblock(OldError)

RETURN lResult


/****
*   Function GetFileName() --> CHARACTER
*
*   Purpose: Determine the bare filename.
****/

FUNCTION GetFileName( cFile )
RETURN if("\" $ cFile, right(cFile, rat(cFile, "\") - 1), cFile)


/****
*   Function CheckDbf(DbfFile,nMode,cPath) --> nResult
*
*   Purpose     : Check the integrity of Data Base Files
****/

FUNCTION CheckDBF( Dbfname, nMode, cPath )

    LOCAL   nResult     := DBF_ERROR,  ;
            nHandle,                   ;
            nHandleDbt,                ;
            cLeader,                   ;
            nLeaderLenght := 32,       ;
            cHeader,                   ;
            nheaderLenght,             ;
            nFirstByte  := 0,          ;
            cFileName,                 ;
            nHeaderEnd,                ;
            nTemp
        
    DEFAULT nMode TO if(set(_SET_EXCLUSIVE),FO_READ,FO_READ+FO_SHARED)
    DEFAULT cPath TO ""

    cLeader   := space(nLeaderLenght)         // DBF header       
    Dbfname   := if(at(".",Dbfname) == 0, upper(Dbfname+".DBF"),upper(Dbfname))
    cFileName := left(Dbfname,at(".",Dbfname)-1)

    BEGIN SEQUENCE

        IF valtype(Dbfname) <> "C" .or. len(alltrim(Dbfname)) == 0
            alert("Wrong or missing parameter!;Function;CheckDbf(Dbfname)")
            nResult := PROGRAMMER_ERR
            BREAK                                                       
        ENDIF

        nHandle := fopen(cPath+Dbfname,nMode)
        IF ferror() <> 0           
            nResult := ferror()
            BREAK                               // Open failed                       
        ENDIF                                

        IF nLeaderLenght <> fread(nHandle,@cLeader,nLeaderLenght)
            BREAK                               // Can't read first 32 bytes
        ENDIF

        nFirstByte := asc(cLeader)
        IF nFirstByte <> 3 .and. nFirstByte <> 131
            BREAK                               // Not a dBase-III-type file
        ENDIF

        nHeaderLenght := asc(substr(cLeader,9,1)) + ;
                                    asc(substr(cLeader,10,1))*256
        nTemp := nHeaderLenght % 32

        IF nTemp < 1 .or. nTemp > 2             // 1 if dBase, 2 if dbu
            BREAK                               // Bad header length
        ENDIF

        nHeaderLenght := nHeaderLenght - 32      // Header after leader
        cHeader := space(nHeaderLenght)          // Buffer for header
        nHeaderEnd := nHeaderLenght - nTemp + 1

        IF nHeaderLenght <> fread( nHandle, @cHeader, nHeaderLenght )
            BREAK                                  // File is too short
        ENDIF

        IF substr(cHeader,nHeaderEnd,1) <> chr(13)
            BREAK                                  // Last char should be <cr>
        ENDIF

        IF nFirstByte == 131 
            nHandledbt := fopen(cPath+cFileName + ".DBT",nMode)
            IF ferror() <> 0
                nResult :=  DBT_ERROR
                BREAK
            ELSE
                fclose(nHandledbt)
            ENDIF
        ENDIF

        // So far so good....
        nResult := NO_ERROR

    END SEQUENCE              

    IF nHandle > 0
        fclose(nHandle)
    ENDIF        

RETURN nResult 


/****
*   CheckNtx( cIndexName,cIndexKey,nMode,cPath )
*
*   Purpose     : Check integrity of NTX files
****/

FUNCTION CheckNTX(cIndexName,cIndexKey,nMode,cPath)

    LOCAL   nSearch     := 0,        ;
            cKey        := "",       ;
            nResult     := NTX_ERROR,;  // Until all tests are passed.
            nHandle ,                ;
            nHeaderSize := 1024,     ;  // NTX must be at least this long.
            nKeyOffset  := 23           // Where Clipper stuffs the key.

    LOCAL   cHeader :=  space(nHeaderSize) ,    ;
            lExact  := set(_SET_EXACT)     ,    ;
            lIndex
            
    default nMode to if(set(_SET_EXCLUSIVE), FO_READ, FO_READ+FO_SHARED)
    default cPath to ""

    set(_SET_EXACT,.T.)

    BEGIN SEQUENCE

        IF valtype(cIndexName) <> "C" .or. len(alltrim(cIndexName)) == 0
            alert("Wrong or missing parameter!;Function;CheckNtx(cIndexName)")
            nResult := PROGRAMMER_ERR
            BREAK                                                       
        ENDIF

        IF valtype(cIndexkey) <> "C" .or. len(alltrim(cIndexKey)) == 0
            lIndex := .f.
        ELSE
            cIndexKey := upper(cIndexKey)             
            lIndex := .t.
        ENDIF

        cIndexName := if(at(".",cIndexName) == 0, cIndexName+".NTX",cIndexname)
        nHandle := fopen(cPath+cIndexName,nMode)    // Try to open the index.
        IF ferror() <> 0         
            nResult := ferror()
            BREAK
        ENDIF

        IF nHeaderSize <> fread(nHandle,@cHeader,nHeaderSize)
            BREAK                               // Error reading the header.
        ENDIF

        IF left(cHeader,2) <> chr(6)+chr(0)
            BREAK                               // Must be: 06 00.
        ENDIF    

        cKey := substr(cHeader,nKeyOffset)      //  Key in the current index.
        nSearch := at(chr(0),cKey)
        IF  nSearch == 0                    
            BREAK
        ENDIF

        IF lIndex .and. left(cKey,nSearch-1) <> cIndexKey
            nResult := INDEXKEY_ERR
            BREAK                               // Key is wrong.
        ENDIF

        nResult := NO_ERROR

    END SEQUENCE

    IF nHandle > 0
        fclose(nHandle)
    ENDIF

    set(_SET_EXACT,lExact)

RETURN nResult                                 


/****
*   Function   CreateDbf()-->nil
*
*   Purpose     : Create any DBF using dbcreate()
****/

FUNCTION CreateDbf(cPath,cDbname,aStructure)    

    LOCAL lResult := .t.,cMsg

    default cPath to ""

    BEGIN SEQUENCE

        IF !CheckPath(cPath)                   
            //" Wrong Path was specified !"
            BREAK ""
        ENDIF

        IF valtype(cDbName) <> "C"
            BREAK " Wrong parameter Function: CreateDbf(,cDbname) !"
        ENDIF

        IF valtype(aStructure) <> "A"
            BREAK  " Wrong parameter Function: CreateDbf(,,aStructure) !"
        ENDIF
    
        dbcreate(cPath+cDbname,aStructure)
    
    RECOVER USING cMsg

        Beep Error
        alert(cMsg)
        lResult := .f.                        

    END SEQUENCE

RETURN lResult


/****
*   Function   CheckPath()-->lResult
*
*   Purpose     : Check out for existance of the PATH
****/

FUNCTION CheckPath(cPath,lDisplay)

    LOCAL nHandle, lResult := .f.

    default lDisplay to .t.

    BEGIN SEQUENCE

        IF valtype(cPath) <> "C"                             
            Beep Error
            alert("ERROR: Wrong parameter Function: CheckPath(cPath)!")
            BREAK
        ENDIF

        nHandle := fcreate(cPath+"_TEMP",if(set(_SET_EXCLUSIVE),2,64))

        IF ferror() <> 0
            IF(lDisplay,alert(GetDosErr(ferror())),nil)
            BREAK                      
        ELSE
            fclose(nHandle)
            ferase(cPath+"_TEMP")
            lResult := .t.
        ENDIF

    END SEQUENCE

RETURN lResult          


/****
*   Function CreateNtx()-->LOGICAL
*
*   Purpose     : Create NTX file if corrupted or missing
****/

FUNCTION CreateNtx(cFile, cPath, cIndexKey, lUnique)

    LOCAL ntxblock,                    ;
          nRow := 18,                  ;
          nCol := 14,                  ;
          cBackClr := "W/B",           ;
          cHiClr := "GR+/B",           ;
         cMsg,lResult := .t.

     LOCAL OldError := errorblock({|| break("")})

    default cPath to ""                            
    default lUnique to .f.

    BEGIN SEQUENCE

    aGauge := GaugeNew( nRow,nCol,nRow+2,nCol+54,"W/B","W+/B")

    GaugeDisplay(aGauge) 
    nPercent := 0                   

    ntxBlock := &("{ || UpdateProg()," + cIndexKey + "}" )
    dbcreateind(cPath + cFile, cIndexKey, ntxBlock, lUnique)

    RECOVER USING cMsg
       IF empty(cMsg)
           cMsg := "ERROR: Unknown error occurred!"
       ENDIF
       beep error
       alert(cMsg)
       lResult := .f.

    END SEQUENCE
    inkey(.5)
    GaugeRelease(aGauge)

RETURN lResult

/****
*   Function UpdateProg() --> nil
*
*   Purpose     : Update status bar
****/

FUNCTION UpdateProg()

    IF recno() < reccount() + 1
        nPercent += 1/reccount()
        GaugeUpdate(aGauge,nPercent)
     ENDIF                    

    // Put a Clock updating routine...
    // ClockUpdate()

RETURN nil

STATIC FUNCTION CheckHandles()

   LOCAL lResult :=.t., nHandle

    IF (nHandle := fopen("NUL",0)) < 0
        lResult := .f.
     ELSE
        fclose(nHandle)
    ENDIF

RETURN lResult


**** EOF open.prg

