/***
*  Gauge.prg
*
*  Sample functions to create, display, and update a percentage completed
*  progress gauge.  This function can be used for creating user interface 
*  options such as a status bar to indicate the current status of a process.
*
*  Copyright (c) 1990, Nantucket Corp.  All rights reserved.
*  David R. Alison
*  Portions: Michael Abadjiev
*
*  Note: Compile with /W/N options
*
*/

// Box array definitions
#define B_TOP           1
#define B_LEFT          2
#define B_BOTTOM        3
#define B_RIGHT         4
#define B_BACKCOLOR     5
#define B_BARCOLOR      6
#define B_DISPLAYNUM    7
#define B_BARCHAR       8
#define B_PERCENT       9
#define B_IMAGE         10

#define B_LEN           10

#define B_BOXLINES      "Ŀ"

#xcommand DEFAULT <var> TO <val>        ;
=>  if valtype(<var>) <> valtype(<val>) ;
    ; <var> := <val>                    ;
    ; endif


/***
*  GaugeNew( <nRowTop>, <nColumnTop>, <nRowBottom>, <nColumnBottom>, 
*     [<cBackgroundColor>], 
*     [<cGaugeColor>], 
*     [<cGaugeCharacter>] ) --> aGauge
*
*  Create a new gauge array
*
*/
FUNCTION GaugeNew( nTop, nLeft, nBottom, nRight, ;
                 cBackColor, cBarColor, cBarCharacter )

   LOCAL aHandle[ B_LEN ]

   // Default values
   default nTop to 19
   default nLeft to 10
   default nBottom to 21
   default nRight to 69
   default cBarColor to "W+/N"
   default cBackColor to cBarColor
   default cBarCharacter to chr(219)

   // Assign default values
   aHandle[ B_TOP ]        := nTop
   aHandle[ B_LEFT ]       := nLeft
   aHandle[ B_BOTTOM ]     := nBottom
   aHandle[ B_RIGHT ]      := nRight
   aHandle[ B_BACKCOLOR ]  := cBackColor
   aHandle[ B_BARCOLOR ]   := cBarColor
   aHandle[ B_DISPLAYNUM ] := .T.
   aHandle[ B_BARCHAR ]    := cBarCharacter
   aHandle[ B_PERCENT ]    := 0

   // OK, the defaults are set, now let's make sure it will fit on the
   // screen correctly
   IF aHandle[ B_RIGHT ] < aHandle[ B_LEFT ] + 4
      aHandle[ B_RIGHT ] := aHandle[ B_LEFT ] + 4
   ENDIF

   IF aHandle[ B_BOTTOM ] < aHandle[ B_TOP ] + 2
      aHandle[ B_BOTTOM ] := aHandle[ B_TOP ] + 2
   ENDIF

   // Determine if we can fit the bracketed number on top of the graph
   IF aHandle[ B_RIGHT ] < aHandle[ B_LEFT ] + 9
      aHandle[ B_DISPLAYNUM ] := .F.
   ENDIF

RETURN aHandle 

/***
*  GaugeDisplay( aGauge ) --> aGauge
*  Display a gauge array to the screen
*
*/ 

FUNCTION GaugeDisplay( aHandle )

   LOCAL cOldColor := SETCOLOR( aHandle[ B_BACKCOLOR ] )
   LOCAL nCenter := ROUND((aHandle[ B_RIGHT ] - aHandle[ B_LEFT ]) / 2, 0 ) + 1

   // If the screen portion under neat of the gauge is not saved
   // just save it..
   IF valtype(aHandle[B_IMAGE]) <> "C"
      aHandle[B_IMAGE] := savescreen( aHandle[ B_TOP ], aHandle[ B_LEFT ],;
                          aHandle[ B_BOTTOM ], aHandle[ B_RIGHT ])
   ENDIF

   dispbegin()

   @ aHandle[ B_TOP ], aHandle[ B_LEFT ] CLEAR TO ;
     aHandle[ B_BOTTOM ], aHandle[ B_RIGHT ]

   @ aHandle[ B_TOP ], aHandle[ B_LEFT ], ;
     aHandle[ B_BOTTOM ], aHandle[ B_RIGHT ] BOX B_BOXLINES

   IF aHandle[ B_DISPLAYNUM ]
      @ aHandle[B_TOP], aHandle[B_LEFT] + 1             ;
        say padc( " " + str(aHandle[B_PERCENT]*100,3) + "% ",  ;
        aHandle[B_RIGHT] - aHandle[B_LEFT] - 1, chr(196) )       ;
        color aHandle[ B_BACKCOLOR ]
   ENDIF

   SETCOLOR( cOldColor )

   // Draw bar to show current percent
   GaugeUpdate( aHandle, aHandle[ B_PERCENT ] )

   dispend()

RETURN aHandle 

/***
*  GaugeUpdate( aGauge, nPercent ) --> aGauge
*  Updates a gauge with a new progress value and redisplays the gauge 
*  to the screen to the screen
*
*/

FUNCTION GaugeUpdate( aHandle, nPercent )

   LOCAL nBarRatio := (aHandle[ B_RIGHT ]) - (aHandle[ B_LEFT ] + 1)
   LOCAL nRow := 0, nCols := 0

   default nPercent to aHandle[B_PERCENT]

   aHandle[B_PERCENT] := nPercent

   dispbegin()
   IF aHandle[ B_DISPLAYNUM ]
      @ aHandle[B_TOP], aHandle[B_LEFT] + 1             ;
        say padc( " " + str(aHandle[B_PERCENT]*100,3) + "% ",  ;
        aHandle[B_RIGHT] - aHandle[B_LEFT] - 1, chr(196) )       ;
        color aHandle[ B_BACKCOLOR ]
   ENDIF

   IF nPercent > 1
      nPercent := 1
   ENDIF

   IF nPercent < 0
      nPercent := 0
   ENDIF

   nCols := round( nPercent * nBarRatio, 0 )

   dispbox( aHandle[ B_TOP ] + 1, aHandle[ B_LEFT ] + 1, ;
     aHandle[ B_BOTTOM ] - 1, aHandle[ B_RIGHT ] - 1, "         ",;
     aHandle[ B_BARCOLOR] )

   FOR nRow := 1 TO (aHandle[ B_BOTTOM ] - aHandle[ B_TOP ] - 1)

      @ nRow + aHandle[B_TOP], aHandle[B_LEFT] + 1   ;
      say replicate( aHandle[ B_BARCHAR ], nCols - 5) ;
      color aHandle[ B_BARCOLOR ] 
      @ row(),col() say  ;
      alltrim(str(aHandle[ B_PERCENT ] *100,3)) + "%" ;
      color aHandle[ B_BARCOLOR ]
   NEXT
   dispend()


RETURN aHandle 


/****
*   Function   GaugeRelease()-->NIL
*
*   Purpose     : Kill the gauge pseudo-object.
****/

FUNCTION GaugeRelease(aHandle)

   BEGIN SEQUENCE

   IF valtype(aHandle) <> "A"
      alert("ERROR: Invalid parameter!;Function GaugeRelease().")
      BREAK
   ENDIF

   IF valtype(aHandle[B_IMAGE]) == "C"
      restscreen( aHandle[ B_TOP ], aHandle[ B_LEFT ],;
                 aHandle[ B_BOTTOM ], aHandle[ B_RIGHT ],aHandle[B_IMAGE])
   ENDIF

   END SEQUENCE

RETURN aHandle := nil

// **** EOF: gauge.prg
