/********
*
*         Program
*
*     Application: Generic Application
*       File Name: EVNTMGR.PRG
*          Author: Ashin S.Wimalajeewa   CompuServe : 100351,650
*
*    Date created: 12-08-94              Date updated: 12-08-94
*       Copyright: (c) 1994 by Virtual/Ash-Tec Development
*     Acknowledge: Transit Computer System Pty Ltd. (Australia)
*
*     Description: Definition of class methods
*
**/


#ifndef _INKEY_CH
  #include "Inkey.ch"
  #define _INKEY_CH
#endif

#ifndef _CLASSY2_CH
  #include "Class(y).ch"
  #define _CLASSY2_CH
#endif

#ifndef _EVNTMGR_CH
  #include "EvntMgr.ch"
#endif


/**
*
*          Method
*
*            Name: Init()
*
*       Arguments: nWait    - Numeric time delay to wait for a double click
*                : nDblWait - Numeric time delay after the DblClick event detected
*                : oMD      - Object of root type TMouseDevice
*    Return Value: self
*        See Also: TObject
*
*     Description:
**/

method Init( nWait, nDblWait, oDispCtxt, oMseDevice ), ( 0, 0, MaxRow(), MaxCol() )
  ::FormStack   := {}
  ::FormCount   := 0
  ::HiddenLst   := {}
  ::HiddenCount := 0
  ::Evnt        := TEvent():New( nWait, nDblWait, oMseDevice )
  ::Color       := SetColor()

  ::Quit        := .f.
  ::Col         := Col()
  ::Row         := Row()
  ::oDC         := oDispCtxt
  ::oMD         := oMseDevice
return ( self )


/**
*
*          Method
*
*            Name: ActiveForm()
*
*       Arguments: nNameID   - Numeric ID for the form
*                : nFormPos  - Numeric Position for the form in form stack
*    Return Value: nPos      - New or current active form position
*
*     Description: make a form active or return the currently active form.
**/

method ActiveForm( nNameID, nFormPos )
  Local nPos     := 0
  Local oFormObj := NIL

  if ( nNameID == NIL .and. nFormPos == NIL )
     nPos := ::FormCount
  else
     if ( nFormPos == 0 )
        nPos := nFormPos
     else
        nPos := iif( nFormPos == NIL, ::SrchForm(nNameID), nFormPos )
        if ( nPos != 0 .and. nPos != ::FormCount )
           ::FormStack[ ::FormCount ]:LostFocus()

           ::ArrangeForms( nPos )
           oFormObj := ::FormStack[ nPos ]
           aDel( ::FormStack, nPos )
           ::FormStack[ ::FormCount ] := oFormObj
           nPos                       := ::FormCount

           ::FormStack[ ::FormCount ]:SetFocus()
           ::FormStack[ ::FormCount ]:GotFocus()
        endif
     endif
  endif
return ( nPos )


/**
*
*          Method
*
*            Name: AddForm()
*
*       Arguments: oFormObj - Object of root type TForm
*    Return Value: oFormObj - Object of root type TForm
*        See Also: ::AppendForm(), ::DelForm(), ::SrchForm()
*
*     Description: Add the form to the hidden form list
**/

method AddForm( oFormObj )
  oFormObj:Parent := self
  oFormObj:oDC    := ::oDC
  oFormObj:oMD    := ::oMD
  aAdd( ::HiddenLst, oFormObj )
  ::HiddenCount++
return ( oFormObj )


/**
*
*          Method
*
*            Name: AppendForm()
*
*       Arguments: oFormObj - Object of root type TForm
*    Return Value: oFormObj - Object of root type TForm
*        See Also: ::AddForm(), ::DelForm(), ::SrchForm(), etc...
*
*     Description: Append a form to the Form Stack (Active Stack)
**/

method AppendForm( oFormObj )
  oFormObj:Parent := self
  oFormObj:oDC    := ::oDC
  oFormObj:oMD    := ::oMD
  aAdd( ::FormStack, oFormObj )
  ::FormCount++
return ( oFormObj )


/**
*
*          Method
*
*            Name: DelForm()
*
*       Arguments: nFormIndex - Numeric Index Position
*                : nFormID    - Numeric Form ID value
*                : oForm      - Object of root type TForm
*    Return Value: oFormObj   - Object of root type TForm
*        See Also: ::AddForm(), ::AppendForm(), ::SrchForm(), etc...
*
*     Description: Deletes a form from the Formstack.
**/

method DelForm( nFormIndex, nFormID, oForm )
  Local oFormObj := NIL

  do case
     case ( nFormIndex != NIL ); oFormObj   := ::FormStack[ nFormIndex ]
     case ( nFormID    != NIL ); nFormIndex := ::SrchForm( nFormID ); oFormObj := ::FormStack[ nFormIndex ]
     case ( oForm      != NIL ); nFormIndex := ::SrchForm( oForm:NameID ); oFormObj := oForm
     otherwise                 ; return ( NIL )
  endcase

  ::ArrangeForms( nFormIndex, .t. )
  // ::ArrangeForms( nFormIndex )     // causes the lost focus event to be
  // oFormObj:LostFocus()             // issued.
  // oFormObj:TObject:Hide()
  oFormObj:UnLoad()
  aDel ( ::FormStack, nFormIndex )
  aSize( ::FormStack, --::FormCount )
  if ( ::FormCount > 0 )
     ::FormStack[ ::FormCount ]:SetFocus()
     ::FormStack[ ::FormCount ]:GotFocus()
  endif
return ( oFormObj )


/**
*
*          Method
*
*            Name: DelHiddenForm()
*
*       Arguments: nNameID   - Name ID of the Hidden form to delete
*    Return Value: ( self )  - Object of root type TForm
*        See Also: ::getHiddenForm()
*
*     Description: Deletes a form from the hidden form list.
**/

method DelHiddenForm( nNameID )
  Local nCntr := 0
  Local nPos  := 0

  for nCntr := 1 to ( ::HiddenCount )
      if ( ::HiddenLst[ nCntr ]:NameID == nNameID )
         nPos := nCntr
      endif
  next
  if ( nPos > 0 )
     aDel ( ::HiddenLst, nPos )
     aSize( ::HiddenLst, --::HiddenCount )
  endif
return ( self )


/**
*
*          Method
*
*            Name: Exec()
*
*    Return Value: self - Object of root type TForm
*        See Also: TEvent:Exec()
*
*     Description: Loop control for the Event Manager
**/

method Exec()
  if ( ::HiddenCount > 0 )
     ::Show()
     While ( !::Quit )
           if ( ::FormCount == 0 )
              ::Quit := .t.
              exit
           else
              ::Evnt:Exec()
           endif
           if ( ::ActiveForm(NIL, ::ManageEvent()) > 0 )
              ::SendMsg()
           endif
     enddo
     if ( ::FormCount > 0 )
        ::Kill()
     endif
     ::Hide()
     SetPos( ::Row, ::Col )
  endif
return ( self )


/**
*
*          Method
*
*            Name: FrmCount()
*
*       Arguments: None
*    Return Value: Numeric ::FormCount value
*        See Also: ::FormStack
*
*     Description: This is really used for internal use only.
**/

method FrmCount()
return ( ::FormCount )


/**
*
*          Method
*
*            Name: FrmStack()
*
*       Arguments: None
*    Return Value: Array of Root type TForm ::FormStack
*        See Also: ::FormCount
*
*     Description: This os really for internal use only.
**/

method FrmStack()
return ( ::FormStack )


/**
*
*          Method
*
*            Name: InsForm()
*
*       Arguments: nPos      - Insertion Position
*                : oFormObj  - Object of root type TForm
*    Return Value: oFormObj  - Object of root type TForm
*        See Also: ::AddForm(), ::DelForm(), ::SrchForm(), etc...
*
*     Description: Inserts a form into the Hidden Form List
**/

method InsForm( nPos, oFormObj )
  oFormObj:Parent := self
  aSize( ::HiddenLst, ++::HiddenCount )
  aIns ( ::HiddenLst, nPos )
  ::HiddenLst[ nPos ] := oFormObj
return ( oFormObj )


/**
*
*          Method
*
*            Name: getForm()
*
*       Arguments: nFormIndex - Form Stack Index position
*    Return Value: oFormObj   - Object of root type TForm
*        See Also: ::SrchForm()
*
*     Description: Get a form from the Form Stack given and Index Positon
**/

method getForm( nFormIndex )
return ( ::FormStack[ nFormIndex ] )


/**
*
*          Method
*
*            Name: getHiddenForm()
*
*       Arguments: nNameID   - Numeric ID of Form
*    Return Value: oHForm    - Object of root type TForm
*        See Also: ::DelHiddenForm()
*
*     Description: Gets and Object from the Hidden Form List.
**/

method getHiddenForm( nNameID )
  Local nFormCntr := 0
  Local oHForm    := NIL

  for nFormCntr := 1 to ( ::HiddenCount )
      if ( ::HiddenLst[ nFormCntr ]:NameID == nNameID )
         oHForm := ::HiddenLst[ nFormCntr ]
      endif
  next
return ( oHForm )


/**
*
*          Method
*
*            Name: Msg()
*
*       Arguments: cMsg - Character String to String to Display
*    Return Value: NIL
*
*     Description: Display a message to the Desktops message area.
**/

method Msg( cMsg )
  Local cEnhClr := StrExtract(::Color, ",", 2)
  Local lmCsrON := ::oMD:Set( .f. )

  ::oDC:print( ::Bottom(), 0, PadR(" "+cMsg, ::Right()+1-20), cEnhClr )
  ::oMD:Set( lmCsrON )
return NIL


/**
*
*          Method
*
*            Name: Show()
*
*    Return Value: NIL
*        See Also: TForm:fShow(), TObject:Show()
*
*     Description: Shows all forms that are have visibility enabled.
**/

method Show()
  Local aHidden     := {}
  Local cEnhClr     := StrExtract(::Color, ",", 2)
  Local lmCsrON     := ::oMD:Set( .f. )
  Local nFormCntr   := 0
  Local nHiddenCntr := 0

  ::TObject:Show()
  for nFormCntr := 1 to ( ::FormCount )
      ::FormStack[ nFormCntr ]:fShow()           // Internal Show
  next
  for nFormCntr := 1 to ( ::HiddenCount )
      if ( ::HiddenLst[ nFormCntr ]:Visible )
         ::AppendForm( ::HiddenLst[ nFormCntr ] ):fShow()  // Internal Show
      else
         nHiddenCntr++
         aAdd( aHidden, ::HiddenLst[ nFormCntr ] )
      endif
  next

  ::HiddenCount := nHiddenCntr
  ::HiddenLst   := aHidden

  if ( ::FormCount > 0 )
     ::FormStack[ ::FormCount ]:SetFocus()
     ::FormStack[ ::FormCount ]:GotFocus()
  endif

  ::oDC:print( ::Bottom(),0, PadR(" ", ::Right()+1), cEnhClr )
  ::oMD:Set( lmCsrON )
return NIL


/**
*
*          Method
*
*            Name: SrchForm()
*
*       Arguments: nNameID   - Numeric ID of the form to Search for
*    Return Value: None
*
*     Description: Searchs for a form in the form stack
**/

method SrchForm( nNameID )
  Local nFormCntr := 0
  Local nPos      := 0

  for nFormCntr := ( ::FormCount ) to 1 step -1
      if ( ::FormStack[ nFormCntr ]:NameID == nNameID )
         nPos := nFormCntr
      endif
  next
return ( nPos )


/***************************************************************************
*
*   PROTECTED METHODS
*
*/


/**
*
*          Method
*
*            Name: ArrangeForms()
*
*       Arguments: nPos   - Numeric Position of form in form stack to make active
*                : lDelIt - Make it active and then delete it.
*    Return Value: NIL
*
*     Description: Arranges the forms on the desktop when forms are shuffled
**/

method ArrangeForms( nPos, lDelIt )
  Local aReDisplay := {}
  Local nFormCntr  := nPos
  Local nMaxLen    := 0

  for nFormCntr := ( nPos + 1 ) to ( ::FormCount )
      if ( ::Intersect(nPos, nFormCntr) )
         aAdd( aReDisplay, nFormCntr )
      endif
  next
  nMaxLen := Len( aReDisplay )
  if ( nMaxLen == 0 )
      if ( lDelIt != NIL )
         if ( lDelIt )
            ::FormStack[ nPos ]:TObject:Hide()
         endif
      endif
  elseif ( nMaxLen > 0 )
     aAdd( aReDisplay, nPos )
     ::ReDisplay( aReDisplay, nMaxLen + 1, lDelIt )
  endif
return NIL


/**
*
*          Method
*
*            Name: Kill()
*
*    Return Value: NIL
*        See Also: TForm:Unload
*
*     Description: Removes a form from the screen and form stack.
**/

method Kill()
  Local nCntr := 0

  for nCntr := ( ::FormCount ) to 1
      ::FormStack[ nCntr ]:UnLoad()
  next
return NIL


/**
*
*          Method
*
*            Name: ManageEvent()
*
*    Return Value: nPos - Numeric Pos of the Form in the Form Stack
*        See Also: ::ArrangeForms(), ::Exec()
*
*     Description: Determines which Form the message goes to.
**/

method ManageEvent()
  Local nFormCntr := 0
  Local nPos      := 0
  Local oFormObj  := NIL

  if ( !::FormStack[ ::FormCount ]:Modal )
     if ( !::Evnt:KeyPress() )
        for nFormCntr := ( ::FormCount ) to 1 step -1
            oFormObj  := ::FormStack[ nFormCntr ]
            if ( ::Evnt:mRow >= oFormObj:Top()  .and. ::Evnt:mRow <= oFormObj:Bottom() .and. ;
                 ::Evnt:mCol >= oFormObj:Left() .and. ::Evnt:mCol <= oFormObj:Right()        )
               nPos := nFormCntr
               exit
            endif
        next
     else
        if ( ::Evnt:Key == K_ALT_TAB )
           nPos := 1
        else
           nPos := ::FormCount
        endif
     endif
  else
     if ( !::Evnt:KeyPress() )
        oFormObj  := ::FormStack[ ::FormCount ]
        if ( ::Evnt:mRow >= oFormObj:Top()  .and. ::Evnt:mRow <= oFormObj:Bottom() .and. ;
             ::Evnt:mCol >= oFormObj:Left() .and. ::Evnt:mCol <= oFormObj:Right()        )
           nPos := ::FormCount
        endif
     endif
  endif
return ( nPos )


/**
*
*          Method
*
*            Name: ReDisplay()
*
*       Arguments: aIndex  - An Array of Numerics of form pos in the form stack
*                : nMaxLen - Numeric len of aIndex
*                : lDelIt  - Logical value of whether the selected Form will removed
*    Return Value: NIL
*        See Also: ::ArrangeForms()
*
*     Description: Redisplays all the required Forms above the selected Form
**/

method ReDisplay( aIndex, nMaxLen, lDelIt )
  Local aScrn    := {}
  Local aScreens := {}
  Local nCntr    := 0
  Local lExit    := .f.
  Local lmCsrON  := ::oMD:Set( .f. )
  Local oForm    := NIL

  for nCntr := ( nMaxLen - 1 ) to 0 step -1
      if ( nCntr == 0 )
         nCntr := nMaxLen
      endif
      oForm    := ::FormStack[ aIndex[nCntr] ]
      aScrn    := { oForm:Top(), oForm:Left(), oForm:Bottom(), oForm:Right(), NIL, oForm }
      aScrn[5] := ::oDC:SaveScreen( aScrn[1], aScrn[2], aScrn[3], aScrn[4] )
      oForm:TObject:Hide()
      aAdd( aScreens, aScrn )
      if ( nCntr == nMaxLen )
         exit
      endif
  next

  for nCntr := ( nMaxLen - 1 ) to 0 step -1
      if ( nCntr == 0 )
         if ( lDelIt == NIL )
            nCntr := nMaxLen
         else
            exit
         endif
      endif
      aScreens[nCntr, 6]:TCtrlObj:Show()
      ::oDC:RestScreen( aScreens[nCntr,1], aScreens[nCntr,2], aScreens[nCntr,3], aScreens[nCntr,4], aScreens[nCntr,5])
      if ( nCntr == nMaxLen )
         exit
      endif
  next
  ::oMD:Set( lmCsrON )
return NIL


/**
*
*          Method
*
*            Name: SendMsg()
*
*    Return Value: None
*        See Also: ::Exec()
*
*     Description: Sends the event message to the chosen Form.
**/

method SendMsg()
  do case
     case ( ::Evnt:Click() )
          ::FormStack[ ::FormCount ]:Click( ::Evnt )

     case ( ::Evnt:DblClick() )
          ::FormStack[ ::FormCount ]:DblClick( ::Evnt )

     case ( ::Evnt:DragDown() )
          ::FormStack[ ::FormCount ]:DragDown( ::Evnt )

     case ( ::Evnt:KeyPress() )
          ::FormStack[ ::FormCount ]:KeyPress( ::Evnt )
  endcase
return NIL


/***************************************************************************
*
*   PRIVATE METHODS
*
*/


/**
*
*          Method
*
*            Name: BoxCross()
*
*       Arguments: nTop1    - Numeric Positions
*                : nLeft1   -         ....
*                : nBottom1 -         ....
*                : nRight1  -         ....
*                : nTop2    -         ....
*                : nLeft2   -         ....
*                : nBottom2 -         ....
*                : nRight2  -         ....
*    Return Value: Logical of whether the two boxes intersect.
*        See Also: ::InterSect()
*
*     Description: This is the condition to determine whether one boxes
*                  corner is within another box
**/

method BoxCross( nTop1, nLeft1, nBottom1, nRight1, nTop2, nLeft2, nBottom2, nRight2 )
  Local lIntersect := .f.

  if ( ((nTop2  >= nTop1  .and. nTop2  <= nBottom1) .or. (nBottom2 >= nTop1  .and. nBottom2 <= nBottom1)) .and. ;
       ((nLeft2 >= nLeft1 .and. nLeft2 <= nRight1 ) .or. (nRight2  >= nLeft1 .and. nRight2  <= nRight1 ))       )
       lIntersect := .t.
  endif
  if ( !lIntersect )
     if ( ((nTop1  >= nTop2  .and. nTop1  <= nBottom2) .or. (nBottom1 >= nTop2  .and. nBottom1 <= nBottom2)) .and. ;
          ((nLeft1 >= nLeft2 .and. nLeft1 <= nRight2 ) .or. (nRight1  >= nLeft2 .and. nRight1  <= nRight2 ))       )
          lIntersect := .t.
     endif
  endif
return ( lIntersect )


/**
*
*          Method
*
*            Name: Intersect()
*
*       Arguments: nForm1 - Numeric Index Position in Form Stack
*                : nForm2 - Numeric Index Position in Form Stack
*    Return Value: Logical if whether the Forms Intersect.
*        See Also: ::BoxCross()
*
*     Description: This method determines whether one of the two forms are
*                  overlapping the other.
**/

method Intersect( nForm1, nForm2 )
  Local lIntersect := .f.

  Local oForm1     := ::FormStack[ nForm1 ]
  Local oForm2     := ::FormStack[ nForm2 ]

  Local nTop1      := oForm1:Top()
  Local nLeft1     := oForm1:Left()
  Local nBottom1   := oForm1:Bottom()
  Local nRight1    := oForm1:Right()

  Local nTop2      := oForm2:Top()
  Local nLeft2     := oForm2:Left()
  Local nBottom2   := oForm2:Bottom()
  Local nRight2    := oForm2:Right()

  lIntersect := ::BoxCross( nTop1, nLeft1, nBottom1, nRight1, nTop2, nLeft2, nBottom2, nRight2 )
  if ( !lIntersect )
     nTop1      := max( nTop1   , nTop2    )
     nLeft1     := max( nLeft1  , nLeft2   )
     nBottom1   := min( nBottom1, nBottom2 )
     nRight1    := min( nRight1 , nRight2  )
     lIntersect := ::BoxCross( nTop1, nLeft1, nBottom1, nRight1, nTop2, nLeft2, nBottom2, nRight2 )
  endif

return ( lIntersect )
