/***
*  Torment2.prg
*  The Data Entry Torment Version 2.1
*
*  NOTE:  Two functions added, Torment() and TormentOff().  Two settings
*         were added to the system variables that are saved and restored
*         also.  Search for NEW! to find changes to system.
*
*	Copyright (c) 1990, 1991 Nantucket Corp.  All rights reserved.
*
*	NOTE: compile with /m/n/w
*/

#include "Set.ch"
#include "Inkey.ch"
#include "Getexit.ch"

#define K_UNDO         K_CTRL_U


// state variables for active READ
STATIC lBumpTop
STATIC lBumpBot
STATIC nLastExit
STATIC nLastPos
STATIC oActiveGet
STATIC cReadProcName
STATIC nReadProcLine

// Tormentor state variables (NEW!)
STATIC lDisabled   := .F.
STATIC lShouldMove := .F.
STATIC lMoved      := .F.
STATIC nLastColor  := 1

// format of array used to preserve state variables
#define TSV_KILLREAD     1
#define TSV_BUMPTOP      2
#define TSV_BUMPBOT      3
#define TSV_LASTEXIT     4
#define TSV_LASTPOS      5
#define TSV_ACTIVEGET    6
#define TSV_READVAR      7
#define TSV_READPROCNAME 8
#define TSV_READPROCLINE 9
#define TSV_GETSMOVED    10 // Have the gets been moved? (NEW!)
#define TSV_TORMENTCOLOR 11 // What was the last color?  (NEW!) 
#define TSV_COUNT        11

#define T_CYCLEON        5  // How often? (NEW!)


/*************************
*
*  User READ Layer
*
*/

/***
*	ReadTorment() (NEW!)
*/
FUNCTION ReadTorment( aGetList, aColors, cBack )
	LOCAL oGet
	LOCAL nPos
	LOCAL aSavedGetSysVars
   LOCAL lTorment := .F.
   LOCAL lDoMove := lShouldMove

   STATIC nCycle := 1

	IF (VALTYPE(READFORMAT()) == "B")
		EVAL(READFORMAT())
	ENDIF

	IF (EMPTY(aGetList))
		// S87 compat.
		SETPOS(MAXROW() - 1, 0)
		RETURN (.F.) // NOTE
	ENDIF

	// preserve state vars
	aSavedGetSysVars := ClearGetSysVars()

	// set these for use in SET KEYs
	cReadProcName := PROCNAME(1)
	nReadProcLine := PROCLINE(1)

   // determine if it is time to torment
   IF (nCycle >= T_CYCLEON)
      lTorment := .T.
      nCycle := 1

      // Toggle torment mode
      lShouldMove := ! lShouldMove
   ELSE
      nCycle++
   ENDIF

	// set initial GET to be read
	nPos := Settle(aGetList, 0)

	DO WHILE (nPos <> 0)

		// get next GET from list and post it as the active GET
		oGet := aGetList[nPos]
		PostActiveGet(oGet)

      // if it is time to torment, by all means torment
      IF (lTorment .AND. ! lDisabled)
         Torment( aGetList, aColors, cBack, lDoMove )
      ENDIF

		// read the GET
		IF (VALTYPE(oGet:reader) == "B")
			EVAL(oGet:reader, oGet) // use custom reader block
		ELSE
			GetReader(oGet) // use standard reader
		ENDIF

		// move to next GET based on exit condition
		nPos := Settle(aGetList, nPos)
	ENDDO

   // Move gets back if they are currently shifted
   IF (lMoved)
      Torment( aGetList, aColors, cBack, lDoMove )
   ENDIF

	// restore state vars
	RestoreGetSysVars(aSavedGetSysVars)

	// S87 compat.
	SETPOS(MAXROW() - 1, 0)

	RETURN (READUPDATED())


/***
*  TormentOff()  (NEW!)
*  Turn off tormentor
*
*/
PROCEDURE TormentOff()
   lDisabled := .T.
   RETURN


/**************************
*
*	READ services
*
*/


/***
*  Torment( @<aGets>, <aColors>, <cBackColor> ) (NEW!)
*  Routine to generally cause havoc with data entry
*
*/
STATIC PROCEDURE Torment( aGets, aColors, cBackColor, lDoMove )
   
   // Buffer screen for instaneous update
   DISPBEGIN()

   IF lDoMove
      MoveAndErase( aGets, cBackColor, IF(lMoved, -1, 1) )
      lMoved := ! lMoved
   ELSE
      IF nLastColor > LEN(aColors)
         nLastColor := 1
      ENDIF
      ColorGets( aGets, aColors[nLastColor++] )
   ENDIF

   DISPEND()

   RETURN

/***
*  MoveAndErase() (NEW!)
*  Erase gets then move 'em
*
*/
STATIC PROCEDURE MoveAndErase( aGets, cBackColor, nMoveBy )
   LOCAL nGet
   LOCAL nLen := LEN(aGets)
   LOCAL lHasFocus
   LOCAL nBuffLen

   // Erase current gets
   FOR nGet := 1 TO nLen
      // Save focus
      lHasFocus := aGets[nGet]:hasFocus 

      // Set focus so I can find out buffer length (get width)
      aGets[nGet]:setFocus()
      nBuffLen := LEN(aGets[nGet]:buffer)

      // Restore focus
      IF !lHasFocus
         aGets[nGet]:killFocus()
      ENDIF

      // Blank out current get
      @ aGets[nGet]:row,aGets[nGet]:col SAY SPACE(nBuffLen);
         COLOR cBackColor
   NEXT

   // Move 'em
   MoveGets( aGets, 0, nMoveBy )
   RETURN


/***
*	Settle()
*
*	Returns new position in array of Get objects, based on
*
*		- current position
*		- exitState of Get object at current position
*
*	NOTE return value of 0 indicates termination of READ
*	NOTE exitState of old Get is transferred to new Get
*/
STATIC FUNCTION Settle( aGetList, nPos )
	LOCAL nExitState

	IF (nPos == 0)
		nExitState := GE_DOWN
	ELSE
		nExitState := aGetList[nPos]:exitState
	ENDIF


	IF (nExitState == GE_ESCAPE .OR. nExitState == GE_WRITE)
		RETURN (0) // NOTE
	ENDIF

	IF (nExitState <> GE_WHEN)
		// reset state info
		nLastPos := nPos
		lBumpTop := .F.
		lBumpBot := .F.
	ELSE
		// re-use last exitState, do not disturb state info
		nExitState := nLastExit
	ENDIF

	/***
	*	move
	*/
	DO CASE
	CASE (nExitState == GE_UP)
		nPos--

	CASE (nExitState == GE_DOWN)
		nPos++

	CASE (nExitState == GE_TOP)
		nPos := 1
		lBumpTop := .T.
		nExitState := GE_DOWN

	CASE (nExitState == GE_BOTTOM)
		nPos := LEN(aGetList)
		lBumpBot := .T.
		nExitState := GE_UP

	CASE (nExitState == GE_ENTER)
		nPos++

	ENDCASE


	/***
	*	bounce
	*/
   // bumped top
	IF (nPos == 0)

		IF (! READEXIT() .AND. ! lBumpBot)
			lBumpTop := .T.
			nPos := nLastPos
			nExitState := GE_DOWN
		ENDIF

	ELSEIF (nPos == LEN(aGetList) + 1) // bumped bottom

		IF (! READEXIT() .AND. nExitState <> GE_ENTER .AND. ! lBumpTop)
			lBumpBot := .T.
			nPos := nLastPos
			nExitState := GE_UP
		ELSE
			nPos := 0
		ENDIF
	ENDIF


	// record exit state
	nLastExit := nExitState

	IF (nPos <> 0)
		aGetList[nPos]:exitState := nExitState
	ENDIF

	RETURN (nPos)



/***
*	PostActiveGet()
*	Post active GET for ReadVar(), GetActive().
*/
STATIC PROCEDURE PostActiveGet( oGet )

	GETACTIVE(oGet)
	READVAR(GetReadVar(oGet))

	ShowScoreBoard()

	RETURN



/***
*	ClearGetSysVars()
*	Save and clear READ state variables. Return array of saved values.
*
*	NOTE: 'Updated' status is cleared but not saved (S87 compat.).
*/
STATIC FUNCTION ClearGetSysVars()
	LOCAL aSaved[TSV_COUNT]


	aSaved[TSV_KILLREAD] := READKILL()
	READKILL(.F.)

	aSaved[TSV_BUMPTOP] := lBumpTop
	lBumpTop := .F.

	aSaved[TSV_BUMPBOT] := lBumpBot
	lBumpBot := .F.

	aSaved[TSV_LASTEXIT] := nLastExit
	nLastExit := 0

	aSaved[TSV_LASTPOS] := nLastPos
	nLastPos := 0

	aSaved[TSV_ACTIVEGET] := GETACTIVE(NIL)

	aSaved[TSV_READVAR] := READVAR("")

   // New Torment values (NEW!)
	aSaved[TSV_READPROCNAME] := cReadProcName
	cReadProcName := ""

	aSaved[TSV_READPROCLINE] := nReadProcLine
	nReadProcLine := 0

   aSaved[TSV_GETSMOVED] := lMoved
   lMoved := .F.

   aSaved[TSV_TORMENTCOLOR] := nLastColor
   nLastColor := 1

	READUPDATED(.F.)

	RETURN (aSaved)


/***
*   RestoreGetSysVars()
*	Restore READ state variables from array of saved values.
*
*	NOTE: 'Updated' status is not restored (S87 compat.).
*/
STATIC PROCEDURE RestoreGetSysVars( aSaved )

	READKILL(aSaved[TSV_KILLREAD])

	lBumpTop := aSaved[TSV_BUMPTOP]

	lBumpBot := aSaved[TSV_BUMPBOT]

	nLastExit := aSaved[TSV_LASTEXIT]

	nLastPos := aSaved[TSV_LASTPOS]

	GETACTIVE(aSaved[TSV_ACTIVEGET])

	READVAR(aSaved[TSV_READVAR])

	cReadProcName := aSaved[TSV_READPROCNAME]

	nReadProcLine := aSaved[TSV_READPROCLINE]

   // New Torment Values (NEW!)
   lMoved := aSaved[TSV_GETSMOVED]
   nLastColor := aSaved[TSV_TORMENTCOLOR]
	RETURN


/***
*	GetReadVar()
*	Set READVAR() value from a GET.
*/
STATIC FUNCTION GetReadVar( oGet )
	LOCAL cName := UPPER(oGet:name)
	LOCAL i

	/***
	*	The following code includes subscripts in the name returned by
	*	this function, if the get variable is an array element.
	*
	*	Subscripts are retrieved from the get:subscript instance variable.
	*
	*	NOTE: incompatible with Summer 87
	*/

	IF (oGet:subscript <> NIL)
		FOR i := 1 TO LEN(oGet:subscript)
			cName += "[" + LTRIM(STR(oGet:subscript[i])) + "]"
		NEXT
	ENDIF

	RETURN (cName)


/**********************************
*
*	wacky compatibility services
*
*/


// display coordinates for SCOREBOARD
#define SCORE_ROW      0
#define SCORE_COL      60

/***
*   ShowScoreboard()
*/
STATIC PROCEDURE ShowScoreboard()
	LOCAL nRow, nCol

	IF (SET(_SET_SCOREBOARD))
		nRow := ROW()
		nCol := COL()

		SETPOS(SCORE_ROW, SCORE_COL)
		DISPOUT(IF(SET(_SET_INSERT), "Ins", "   "))
		SETPOS(nRow, nCol)
	ENDIF

	RETURN



/***
*	DateMsg()
*/
STATIC PROCEDURE DateMsg()
	LOCAL nRow, nCol

	IF (SET(_SET_SCOREBOARD))
		nRow := ROW()
		nCol := COL()

		SETPOS(SCORE_ROW, SCORE_COL)
		DISPOUT("Invalid Date")
		SETPOS(nRow, nCol)

		DO WHILE (NEXTKEY() == 0)
		ENDDO

		SETPOS(SCORE_ROW, SCORE_COL)
		DISPOUT("            ")
		SETPOS(nRow, nCol)

	ENDIF

	RETURN

