/***
* MsgRead.prg
* READ Layer that prints messages on the message line for each
* Get if a message is supplied.
*
* Copyright (c) 1990, 1991 Nantucket Corp.  All rights reserved.
*
* NOTE: compile with /m/n/w
*/

#include "Set.ch"
#include "Inkey.ch"
#include "Getexit.ch"

#define K_UNDO         K_CTRL_U


// state variables for active READ
STATIC BumpTop
STATIC BumpBot
STATIC LastExit
STATIC LastPos
STATIC ActiveGet
STATIC ReadProcName
STATIC ReadProcLine


// format of array used to preserve state variables
#define GSV_KILLREAD     1
#define GSV_BUMPTOP      2
#define GSV_BUMPBOT      3
#define GSV_LASTEXIT     4
#define GSV_LASTPOS      5
#define GSV_ACTIVEGET    6
#define GSV_READVAR      7
#define GSV_READPROCNAME 8
#define GSV_READPROCLINE 9

#define GSV_COUNT        9

/*************************
*
*  READ MESSAGES
*
*/

/***
*	ReadMessage()
*/
FUNCTION ReadMessage( aGetList )
	LOCAL oGet
	LOCAL nPos
	LOCAL aSavedGetSysVars
   LOCAL cMsg, lCenter, nRow
   LOCAL nSaveRow, nSaveCol
   LOCAL cSaveScr

	IF (VALTYPE(ReadFormat()) == "B")
		EVAL(ReadFormat())
	ENDIF

	IF (EMPTY(aGetList))
		// S87 compat.
		SETPOS(MAXROW() - 1, 0)
		RETURN (.F.) // NOTE
	ENDIF

   // Conform to normal message system
   nRow := SET(_SET_MESSAGE)
   IF nRow == NIL
      nRow := 0
   ENDIF

   lCenter := SET(_SET_MCENTER)
   IF lCenter == NIL
      lCenter := .F.
   ENDIF

	// preserve state vars
	aSavedGetSysVars := ClearGetSysVars()

	// set these for use in SET KEYs
	ReadProcName := PROCNAME(1)
	ReadProcLine := PROCLINE(1)

	// set initial GET to be read
	nPos := Settle(aGetList, 0)

	DO WHILE (nPos <> 0)
		// get next GET from list and post it as the active GET
		oGet := aGetList[nPos]
		PostActiveGet(oGet)

      // if a message is supplied and messages are enabled
      // display it
      cSaveScr := NIL
      IF (nRow > 0) .AND. (oGet:cargo != NIL)

         // Retrieve message from dictionary
         cMsg := DictAt(oGet:cargo, "cMessage")

         IF (cMsg != NIL)
            // Save message row and cursor settings
            cSaveScr := SAVESCREEN(nRow, 0, nRow, MAXCOL() - 1)
            nSaveRow := ROW()
            nSaveCol := COL()

            IF lCenter
               SETPOS(nRow, ((MAXCOL()/2) - (LEN(cMsg)/2)))
               DISPOUT(cMsg)
            ELSE
               SETPOS(nRow, 0)
               DISPOUT(cMsg)
            ENDIF

            // Restore cursor
            SETPOS(nSaveRow, nSaveCol)
         ENDIF
      ENDIF

		// read the GET
		IF (VALTYPE(oGet:reader) == "B")
			EVAL(oGet:reader, oGet) // use custom reader block
		ELSE
			GetReader(oGet) // use standard reader
		ENDIF

      // if line was saved, restore it
      IF (cSaveScr != NIL)
         RESTSCREEN(nRow, 0, nRow, MAXCOL() - 1, cSaveScr)
      ENDIF

		// move to next GET based on exit condition
		nPos := Settle(aGetList, nPos)
	ENDDO

	// restore state vars
	RestoreGetSysVars(aSavedGetSysVars)

	// S87 compat.
	SETPOS(MAXROW() - 1, 0)

	RETURN (UPDATED())



/**************************
*
*	READ services
*
*/



/***
*	Settle()
*
*	Returns new position in array of Get objects, based on
*
*		- current position
*		- exitState of Get object at current position
*
*	NOTE return value of 0 indicates termination of READ
*	NOTE exitState of old Get is transferred to new Get
*/
STATIC FUNCTION Settle( aGetList, nPos )
	LOCAL nExitState

	IF (nPos == 0)
		nExitState := GE_DOWN
	ELSE
		nExitState := aGetList[nPos]:exitState
	ENDIF

	IF (nExitState == GE_ESCAPE .OR. nExitState == GE_WRITE)
		RETURN (0) // NOTE
	ENDIF

	IF (nExitState <> GE_WHEN)
		// reset state info
		LastPos := nPos
		BumpTop := .F.
		BumpBot := .F.
	ELSE
		// re-use last exitState, do not disturb state info
		nExitState := LastExit
	ENDIF

	/***
	*	move
	*/
	DO CASE
	CASE (nExitState == GE_UP)
		nPos--

	CASE (nExitState == GE_DOWN)
		nPos++

	CASE (nExitState == GE_TOP)
		nPos := 1
		BumpTop := .T.
		nExitState := GE_DOWN

	CASE (nExitState == GE_BOTTOM)
		nPos := LEN(aGetList)
		BumpBot := .T.
		nExitState := GE_UP

	CASE (nExitState == GE_ENTER)
		nPos++

	ENDCASE


	/***
	*	bounce
	*/
  // bumped top
	IF (nPos == 0)
		IF (! READEXIT() .AND. ! BumpBot)
			BumpTop := .T.
			nPos := LastPos
			nExitState := GE_DOWN
		ENDIF

	ELSEIF (nPos == LEN(aGetList) + 1) // bumped bottom

		IF (! READEXIT() .AND. nExitState <> GE_ENTER .AND. ! BumpTop)
			BumpBot := .T.
			nPos := LastPos
			nExitState := GE_UP
		ELSE
			nPos := 0
		ENDIF
	ENDIF


	// record exit state
	LastExit := nExitState

	IF (nPos <> 0)
		aGetList[nPos]:exitState := nExitState
	ENDIF

	RETURN (nPos)



/***
*	PostActiveGet()
*	Post active GET for ReadVar(), GetActive().
*/
STATIC PROCEDURE PostActiveGet( oGet )

	GETACTIVE(oGet)
	READVAR(GetReadVar(oGet))

	ShowScoreBoard()

	RETURN



/***
*	ClearGetSysVars()
*	Save and clear READ state variables. Return array of saved values.
*
*	NOTE: 'Updated' status is cleared but not saved (S87 compat.).
*/
STATIC FUNCTION ClearGetSysVars()
	LOCAL aSaved[GSV_COUNT]

	aSaved[GSV_KILLREAD] := ReadKill()
	ReadKill(.F.)

	aSaved[GSV_BUMPTOP] := BumpTop
	BumpTop := .F.

	aSaved[GSV_BUMPBOT] := BumpBot
	BumpBot := .F.

	aSaved[GSV_LASTEXIT] := LastExit
	LastExit := 0

	aSaved[GSV_LASTPOS] := LastPos
	LastPos := 0

	aSaved[GSV_ACTIVEGET] := GETACTIVE(NIL)

	aSaved[GSV_READVAR] := READVAR("")

	aSaved[GSV_READPROCNAME] := ReadProcName
	ReadProcName := ""

	aSaved[GSV_READPROCLINE] := ReadProcLine
	ReadProcLine := 0

	READUPDATED(.F.)

	RETURN (aSaved)



/***
*   RestoreGetSysVars()
*	Restore READ state variables from array of saved values.
*
*	NOTE: 'Updated' status is not restored (S87 compat.).
*/
STATIC PROCEDURE RestoreGetSysVars( aSaved )

	ReadKill(aSaved[GSV_KILLREAD])

	BumpTop := aSaved[GSV_BUMPTOP]

	BumpBot := aSaved[GSV_BUMPBOT]

	LastExit := aSaved[GSV_LASTEXIT]

	LastPos := aSaved[GSV_LASTPOS]

	GETACTIVE(aSaved[GSV_ACTIVEGET])

	READVAR(aSaved[GSV_READVAR])

	ReadProcName := aSaved[GSV_READPROCNAME]

	ReadProcLine := aSaved[GSV_READPROCLINE]

	RETURN



/***
*	GetReadVar()
*	Set READVAR() value from a GET.
*/
STATIC FUNCTION GetReadVar( oGet )
	LOCAL cName := UPPER(oGet:name)
	LOCAL i

	/***
	*	The following code includes subscripts in the name returned by
	*	this function, if the get variable is an array element.
	*
	*	Subscripts are retrieved from the get:subscript instance variable.
	*
	*	NOTE: incompatible with Summer 87
	*/

	IF (oGet:subscript <> NIL)
		FOR i := 1 TO LEN(oGet:subscript)
			cName += "[" + LTRIM(STR(oGet:subscript[i])) + "]"
		NEXT
	ENDIF

	RETURN (cName)


/**********************************
*
*	wacky compatibility services
*
*/


// display coordinates for SCOREBOARD
#define SCORE_ROW      0
#define SCORE_COL      60


/***
*   ShowScoreboard()
*/
STATIC PROCEDURE ShowScoreboard()
	LOCAL nRow, nCol

	IF (SET(_SET_SCOREBOARD))
		nRow := ROW()
		nCol := COL()

		SETPOS(SCORE_ROW, SCORE_COL)
		DISPOUT(IF(SET(_SET_INSERT), "Ins", "   "))
		SETPOS(nRow, nCol)
	ENDIF

	RETURN

/***
*	DateMsg()
*/
STATIC PROCEDURE DateMsg()
	LOCAL nRow, nCol

	IF (SET(_SET_SCOREBOARD))
		nRow := ROW()
		nCol := COL()

		SETPOS(SCORE_ROW, SCORE_COL)
		DISPOUT("Invalid Date")
		SETPOS(nRow, nCol)

		DO WHILE (NEXTKEY() == 0)
		ENDDO

		SETPOS(SCORE_ROW, SCORE_COL)
		DISPOUT("            ")
		SETPOS(nRow, nCol)
	ENDIF

	RETURN

