******************************************************************************
*									      *
*	FUNCTION:  NUM2TEXT						      *
*									      *
*	  SYNTAX:  NUM2TEXT (<expN>[,<expL>])				      *
*									      *
*	 PURPOSE:  To convert a numeric value in to an English string,	      *
*		   suitable for check printing. 			      *
*									      *
*     PARAMETERS:  <expN> -- the numeric value to be converted. 	      *
*		   <expL> -- whether or not to print the decimal portion.     *
*			     If omitted, the default is .T., resulting in the *
*			     output containing info on cents.		      *
*									      *
*   RETURN VALUE:  <expC> -- character expression containing the English      *
*			     representation of the number in <expN>.	      *
*									      *
*	  AUTHOR:  John Wright						      *
*									      *
*    REVISION BY:  Chris Borneman					      *
*		   Borneman Systems					      *
*		   28 Coy Park Drive					      *
*		   Newark, IL 60541					      *
*									      *
*	   NOTES:  The previous version by John Wright did not allow for      *
*		   decimals(cents).  The routine handles both positive and    *
*		   negative numbers.  However, they must be under a trillion  *
*		   in either direction.  This is a recursive function.	      *
*									      *
*   Examples:								      *
*	? num2text(34521.43)						      *
*	    Thirty-Four Thousand, Five Hundred Twenty-One 43/100	      *
*	? num2text(34521.43,.F.)					      *
*	    Thirty-Four Thousand, Five Hundred Twenty-One		      *
*	? num2text(562) 						      *
*	    Five Hundred Sixty-Two No/100				      *
*	? num2text(-612,.F.)						      *
*	    Negative Six Hundred Twelve 				      *
*									      *
*******************************************************************************
FUNCTION NUM2TEXT

PARAMETERS parm1,parm2

PRIVATE chk_dec,add_dec,chk_num,chk_list,chk_it,chk_out,treturn

chk_dec=IIF(TYPE('parm1')='N',parm1,0)
add_dec=IIF(TYPE('parm2')='L',parm2,.t.)
chk_num=INT(parm1)

DO CASE
    CASE chk_num < 0
	treturn="Negative "+NUM2TEXT(chk_dec*-1)

    CASE chk_num < 1 .AND. add_dec
	chk_it=int((chk_dec*100)-(chk_num*100))
	treturn=IIF(chk_it=0,'No',pad(ltrim(str(chk_it,2)),2,'R','0'))+'/100 '

    CASE chk_num < 1
	treturn=''
	* do nothing

    CASE chk_num < 11
	chk_list="Zero One  Two  ThreeFour Five Six  SevenEightNine Ten  "
	treturn=TRIM(SUBSTR(chk_list,(chk_num*5)+1,5))+" "+                   ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 20
	chk_list="Eleven   Twelve   Thirteen Fourteen Fifteen  "+;
	   "Sixteen  SeventeenEighteen Nineteen "
	treturn=TRIM(SUBSTR(chk_list,((chk_num-11)*9)+1,9))+" "+              ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 30
	treturn="Twenty"+IIF(chk_num>20,"-"+NUM2TEXT(chk_num-20,.f.)," ")+    ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 40
	treturn="Thirty"+IIF(chk_num>30,"-"+NUM2TEXT(chk_num-30,.f.)," ")+    ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 50
	treturn="Forty"+IIF(chk_num>40,"-"+NUM2TEXT(chk_num-40,.f.)," ")+     ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 60
	treturn="Fifty"+IIF(chk_num>50,"-"+NUM2TEXT(chk_num-50,.f.)," ")+     ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 70
	treturn="Sixty"+IIF(chk_num>60,"-"+NUM2TEXT(chk_num-60,.f.)," ")+     ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 80
	treturn="Seventy"+IIF(chk_num>70,"-"+NUM2TEXT(chk_num-70,.f.)," ")+   ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 90
	treturn="Eighty"+IIF(chk_num>80,"-"+NUM2TEXT(chk_num-80,.f.)," ")+    ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 100
	treturn="Ninety"+IIF(chk_num>90,"-"+NUM2TEXT(chk_num-90,.f.)," ")+    ;
		    NUM2TEXT(chk_dec-chk_num,add_dec)

    CASE chk_num < 1000
	chk_it=INT(chk_num/100)
	chk_out=chk_dec-(chk_it*100)
	treturn=NUM2TEXT(chk_it,.f.)+"Hundred "+NUM2TEXT(chk_out,add_dec)

    CASE chk_num < 1000000
	chk_it=INT(chk_num/1000)
	chk_out=chk_dec-(chk_it*1000)
	treturn=NUM2TEXT(chk_it,.f.)+"Thousand"+                              ;
		IIF(chk_out>=1,", ",' ')+NUM2TEXT(chk_out,add_dec)

    CASE chk_num < 1000000000
	chk_it=INT(chk_num/1000000)
	chk_out=chk_dec-(chk_it*1000000)
	treturn=NUM2TEXT(chk_it,.f.)+"Million"+                               ;
		IIF(chk_out>=1,", ",' ')+NUM2TEXT(chk_out,add_dec)

    CASE chk_num < 1000000000000
	chk_it=INT(chk_num/1000000000)
	chk_out=chk_dec-(chk_it*1000000000)
	treturn=NUM2TEXT(chk_it,.f.)+"Billion"+                               ;
		IIF(chk_out>=1,", ",' ')+NUM2TEXT(chk_out,add_dec)

    OTHERWISE		 &&if not handled above then display a message
	treturn="** Number not translated, out of range **"
ENDCASE

RETURN treturn
