/* ------------------------------------------------------------------------

NGTitle.C

NGTitle.Exe reads the title from the NG file(s) specified on the command
line.  Author: Dave Gordon.

Version history:

  Rel.  Date     Notes
  ----  ----     -----
  1.00  8/26/94  Initial release.

Copyright 1994 by Blue Ox Software, all rights reserved.

*/

#define Opening_Message   "NGTitle - Norton Guides database title reader"
#define Current_Version   "Version 1.00 - released 8/26/94"
#define Copyright_Message "Copyright (c) 1994 by Blue Ox Software"

#define Prefix_Size   3    /* First 3 bytes of the database has signature */
#define Title_Size  128    /* Rough guess on maximum size */

#include <stdlib.h>
#include <stdio.h>
#include <dir.h>
#include <dos.h>


/* ------------------------------------------------------------------------
   Main. If there are no file parameters, look for "*.ng" in the current
   directory.  Otherwise, seek each parameter in turn.
*/

main(int argc, char *argv[])
{
  int n;
  int count = 0;

  fprintf(stderr, "\n%s\n", Opening_Message);
  fprintf(stderr, "%s\n", Current_Version);
  fprintf(stderr, "%s\n", Copyright_Message);

  if (argc == 1)
    count = readNGfile("*.ng");
  else
    for (n = 1; n < argc; n++)
      count = count + readNGfile(argv[n]);

  if (count == 0)
    fprintf(stderr, "\nNo NG database files found\n");
  else
    fprintf(stderr, "\n%d NG database files found\n", count);

  return (min(count, 100));

} /* main() */


/* ------------------------------------------------------------------------
   readNGfile().  Qualify the file name received from above, and then dig
   out the title with GetTitle()
*/

int readNGfile(char *filename)
{
  char fpath[MAXPATH];       /* Reconstructed search file path     */
  char drive[MAXDRIVE];      /* fnsplit drive                      */
  char dir[MAXDIR];          /* fnsplit directory                  */
  char name[MAXFILE];        /* fnsplit file name                  */
  char ext[MAXEXT];          /* fnsplit file extension             */
  char readfile[MAXPATH];    /* Reconstructed NG file name buffer  */
  int result;                /* return holders for various things  */
  int count = 0;             /* Tally of valid NG files found      */
  struct ffblk fblock;       /* findfirst structure                */

  /* split up the file name; if no extension, use ".ng" */
  result = fnsplit(filename, drive, dir, name, ext);
  if (!(result & EXTENSION))
     strcpy(ext, ".ng");

  /* combine the parts back into a full qualified path name */
  fnmerge(fpath, drive, dir, name, ext);

  /* look for our new file name */
  result = findfirst(fpath, &fblock, 0);
  while (!result)
  {
    /* assemble a complete read file name */
    result = fnsplit(fblock.ff_name, NULL, NULL, name, ext);
    fnmerge(readfile, drive, dir, name, ext);

    /* get title, increment counter, and seek another database */
    count = count + GetTitle(readfile, fblock.ff_name);
    result = findnext(&fblock);
  }
  return (count);  /* number of valid database files actually found */

}  /* readNGfile() */


/* --------------------------------------------------------------------
  GetTitle(). Opens file, validates, reads title, displays file name
  and title. Returns 1 if successful, otherwise 0.
*/

int GetTitle(char *readfile, char *fname)
{
  char prefix[Prefix_Size];  /* NG database buffer for prefix       */
  char title[Title_Size];    /* NG database title buffer            */
  FILE *fNG;                 /* File handle structure               */
  int wrongformat;           /* return holders for various things   */
  int success = 0;           /* 0 if failed, 1 if read successfully */

  /* open file; if valid handle, continue */
  fNG = fopen(readfile, "rb");
  if (fNG)
  {
    /* read the first three bytes */
    fread(prefix, Prefix_Size, 1, fNG);

    /* prefix should be "NG" or "EH"; if so, read title */
    wrongformat = strcmp(prefix, "NG");
    if (wrongformat)
      wrongformat = strcmp(prefix, "EH");

    if (wrongformat)
      printf("%s is not a Norton Guides file!\n", fname);
    else
    {
      /* reset the title buffer, point to the right byte, and read it in */
      strcpy(title, "<UNTITLED>");
      fseek(fNG, 8, SEEK_SET);
      fread(title, Title_Size, 1, fNG);

      /* print file name and title */
      printf("%-12s -> %s\n", fname, title);
      success++;

    }  /* if (wrongformat) */

    /* close the file ... */
    fclose(fNG);

  } /* if (fNG) */

  return (success);

}  /* GetTitle() */


