/*
 * File......: WINDOW.PRG
 * Author....: Jeffrey M. Lewis [CIS ID: >internet:IPWO400@INDYCMS.IUPUI.EDU]
 *                              [CIS ID: 70711,3030]
 * Date......: $Date:  02 Dec 1991 12:00:00     $
 * Revision..: $Revision: 1.1 $
 * Log file..: $Logfile:  E:/nanfor/src/window.prv $
 * 
 * This is an original work by Jeffrey M. Lewis and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log:  E:/nanfor/src/window.prv $
 *
 * Orig version 03/15/91 (Before submitting to Nanforum Toolkit)
 * Converted to array handling and added more parameter checking
 *                        12/91  -  Jeffrey M. Lewis
 *  Submitted to Nanforum 01/92  -  Jeffrey M. Lewis
 *
 * NOTE: This file must be compiled with the /n switch to assure that
 *       the global variables are restricted to this object file only.
 */

#include "box.ch"            // For window border characters

#define MAXELEM       4096   // Max array elements

STATIC xWinStack[0]          // Window Parameters
STATIC xWinCont[0]           // Window Contents
STATIC xWinCurs[0]           // Window Cursor Positions

#ifdef FT_TEST

FUNCTION WINDOW
    LOCAL nCounter, nCounter2

* Test RETURN Codes
    CLEAR SCREEN
    ? "FT_ClosWin() Should return F: ",FT_ClosWin()
    ? "FT_PrtWin()  Should return F: ",FT_PrtWin("testing")
    WAIT
* ------ Test the <nRow> , <nCol> params as 0 -----------
    FT_OpenWin(1 , 1, 20,70,"Window to test Line wrapping.","Press AnyKey",.T.)
    FOR nCounter := 1 TO 10
        FT_PrtWin(0,0,"Hi Michelle!  This is Test Line "+STR(nCounter,2))
        INKEY(0)
    NEXT nCounter
    INKEY(0)
    FT_ClosWin()
* -------------------------------------------------------
    SET COLOR TO "GR+/B"
    FOR nCounter := 1 TO 15
        FT_OpenWin(nCounter,nCounter,nCounter*2+10,nCounter*2+60,;
            "Window "+STR(nCounter,2),"Press AnyKey",.T.)
        FOR nCounter2 := 1 TO 15
            FT_PrtWin("Test Line #"+STR(nCounter2,2))
        NEXT nCounter2
        SET COLOR TO "R/B"
        FT_PrtWin(3,5,"#Test#")
        FT_PrtWin(4,0,"Next Test")
        SET COLOR TO "GR+/B"
        INKEY(0)
    NEXT nCounter
    FT_ClosWin("ALL")
RETURN NIL

#endif

/*  $DOC$
 *  $FUNCNAME$
 *     FT_OpenWin()
 *  $CATEGORY$
 *     Video
 *  $ONELINER$
 *    Open a new window to the current screen
 *  $SYNTAX$
 *    FT_OpenWin( <nTop>, <nLeft>, <nBottom>, <nRight>, ;
 *                [ <cWinTitle> ], [ <cWinBottom> ],    ;
 *                [ <lBorder> ]) -> lStatus
 *  $SEEALSO$
 *     FT_ClosWin()   FT_PrtWin()
 *  $ARGUMENTS$
 *
 *     <nTop>, <nLeft>, <nBottom>, <nRight> are the coordinates of
 *     the area to display the Window in. 
 *
 *     <cWinTitle> is the Title to be displayed in the title bar of the
 *        window.  Defaults to NIL if not passed.
 *
 *     <cWinBottom> is the window title to be displayed in the
 *        window status bar (bottom of window).  Defaults to NIL if
 *        not passed.
 *
 *     <lBorder> is .T. if you want to have a border around your windows
 *        or .F. if not.
 *
 *  $RETURNS$
 *     lStatus is a .T. if the window was successfully opened or a .F.
 *     if to many windows are currently open.
 *
 *  $DESCRIPTION$
 *     This function will open a new window on the screen while preserving
 *     the contents of what's underneath.  You can not have more that
 *     4096 windows open at any given time.  
 *  $EXAMPLES$
 *    LOCAL nCounter, nCounter2
 *
 *    SET COLOR TO "GR+/B"
 *    FOR nCounter := 1 TO 15
 *        FT_OpenWin(nCounter,nCounter,nCounter*2+10,nCounter*2+60,;
 *            "Window "+STR(nCounter,2),"Press AnyKey",.T.)
 *        FOR nCounter2 := 1 TO 15
 *            FT_PrtWin("Test Line #"+STR(nCounter2,2))
 *        NEXT nCounter2
 *        SET COLOR TO "R/B"
 *        FT_PrtWin(5,3,"#Test#")
 *        SET COLOR TO "GR+/B"
 *        INKEY(0)
 *    NEXT nCounter
 *    FT_ClosWin("ALL")
 *  $END$
 */

FUNCTION FT_OpenWin(nTop,nLeft,nBottom,nRight,cWinTitle, cWinBottom, lBorder)
    LOCAL xTmpWin, nCntr

    IF LEN(xWinStack) >= MAXELEM
       RETURN .F.                      // To many windows are open?
    ENDIF
    nLeft   := MIN(MAXCOL(),nLeft)
    nRight  := MIN(MAXCOL(),nRight)
    nTop    := MIN(MAXROW(),nTop)
    nBottom := MIN(MAXROW(),nBottom)

    lBorder := IF(lBorder==NIL,.F.,lBorder)
       
    // Save Screen to global stack
    AADD(xWinStack,chr(nLeft)+chr(nTop)+chr(nRight)+chr(nBottom) ;
                +IIF(lBorder,"1","0"))
    xTmpWin := savescreen(nTop, nLeft, nBottom, nRight)
    AADD(xWinCont, xTmpWin)
    AADD(XWinCurs, CHR(0))
    IF lBorder
        FT_SHADOW(nTop,nLeft,nBottom-1,nRight-2)  // Draw the shadows
        nRight  := IF(nRight<MAXCOL(),nRight -1,nRight)
        nBottom := IF(nBottom<MAXROW(),nBottom -1,nBottom)
    ENDIF
    cWinTitle  := LEFT(cWinTitle,nRight-nLeft-2)
    cWinBottom := LEFT(cWinBottom,nRight-nLeft-2)
    @nTop, nLeft CLEAR TO nBottom, nRight
    @nTop, nLeft, nBottom, nRight BOX B_DOUBLE
    @nTop, (((nRight-nLeft)/2+nLeft)-(LEN(cWinTitle)/2))  SAY (cWinTitle)
    @nBottom, (((nRight-nLeft)/2+nLeft)-(LEN(cWinBottom)/2)) SAY (cWinBottom)
RETURN .T.

/*  $DOC$
 *  $FUNCNAME$
 *     FT_ClosWin()
 *  $CATEGORY$
 *     Video
 *  $ONELINER$
 *    Closes the window(s) opened with FT_OpenWin.
 *  $SYNTAX$
 *    FT_ClosWin( [ "ALL" ] ) -> lStatus
 *  $SEEALSO$
 *     FT_OpenWin()   FT_PrtWin()
 *  $ARGUMENTS$
 *
 *     If no arguments are passed, only the last opened window will be
 *     closed.  If the "ALL" argument is passed, all windows that were
 *     opened with the FT_OpenWin() function will be closed
 *     automatically.  The contents under the window being closed are
 *     restored to their original state.
 *
 *     Passing any arguments that are invalid (not "ALL") will be
 *     ignored and not result in an error.
 *
 *  $RETURNS$
 *     lStatus is a .T. if the window was successfully closed or a .F.
 *     if there are no windows to close.
 *
 *  $DESCRIPTION$
 *     This function will close a window on the screen that was opened
 *     with the FT_ClosWin() function.
 *  $EXAMPLES$
 *
 *    FT_OpenWin(1 , 1, 20,70,"Window","Press AnyKey",.T.)
 *    FT_PrtWin("Test Line")
 *    INKEY(0)
 *    FT_ClosWin()
 *  $END$
 */

FUNCTION FT_ClosWin(cParam)
    LOCAL xTmpWin, nLeft, nRight, nTop, nBottom, lComplete, nLast
    IF LEN(xWinStack) == 0
       RETURN .F.            // No windows to close
    ENDIF
    cParam := IIF(EMPTY(cParam),"NONE",cParam)     // need something incase
    lComplete = .F.                                // it's not passed.
    DO WHILE (.NOT. lComplete)
       nLast   := LEN(xWinStack)
       nLeft   := asc(SUBSTR(xWinStack[nLast],1,1))
       nTop    := asc(SUBSTR(xWinStack[nLast],2,1))
       nRight  := asc(SUBSTR(xWinStack[nLast],3,1))
       nBottom := asc(SUBSTR(xWinStack[nLast],4,1))
       xTmpWin := SUBSTR(xWinCont[nLast],1,(nRight-nLeft+1)*(nBottom-nTop+1)*2)     
       restscreen(nTop,nLeft,nBottom,nRight,xTmpWin)
       ASIZE(xWinStack,nLast-1)
       ASIZE(xWinCont,nLast-1)
       ASIZE(xWinCurs,nLast-1)
       lComplete := IIF(UPPER(cParam) == "ALL" .AND. nLast <> 1,.F.,.T.)
   ENDDO
RETURN .T.

/*  $DOC$
 *  $FUNCNAME$
 *     FT_PrtWin()
 *  $CATEGORY$
 *     Video
 *  $ONELINER$
 *    Print TEXT in the Current window opened with FT_OpenWin().
 *  $SYNTAX$
 *     FT_PrtWin( [ <nRow>, <nCol> ], <cMsg> ) -> lStatus
 *  $SEEALSO$
 *     FT_OpenWin()   FT_ClosWin()
 *  $ARGUMENTS$
 *
 *     <nRow> is the row to position the text to be printed.
 *
 *     <cCol> is the column to position the text to be printed.
 *
 *     <cMsg> is the text to be printed.
 *     
 *  $RETURNS$
 *     lStatus is a .T. if the window was successfully printed to or a .F.
 *     if there are no windows open.
 *
 *  $DESCRIPTION$
 *     This function will print text to the most recent window opened
 *     with the FT_OpenWin() function.
 *
 *     This function does consider the borders by wrapping around text
 *     when needed and scrolling text up when reacing the bottom of the
 *     window.
 *
 *     There are two seperate formats that can be passed.  The most used
 *     would be passing only <cMsg>.  This will automatically continue
 *     printing on the next line from where that last FT_PrtWin() took
 *     place.  
 *
 *     The second format is to specify the <nRow> and <nCol>.
 *     FT_PrtWin() will position the cursor on the coordinates related
 *     to the window borders.  (ie.  If you pass 3, 4 and the upper
 *     left corner is at 10,10.  The cursor will be positioned to 
 *     13, 14 and the <cMsg> will be printed.  This helps prevent the
 *     programmer from having a screen map of some sort.
 *
 *     The <nRow> and <nCol> parameters can be passed to position the
 *     <cMsg> at those coordinates offset from the upper left corner
 *     of the most recent window opened.
 *
 *     Passing a 0 for <nRow> results is staying on the same
 *     row the cursor is currently on (unless you go beyond the
 *     window border, then the line wraps to the next line).
 *
 *     NOTE: If <nCol> is passed as 0, the last column
 *           printed to will be continued from.
 *
 *           ie.:
 *                   FT_PrtWin(5, 5, "Test 1")
 *                   FT_PrtWin(5, 0, "Test 2")
 *
 *     (The 0 will be treated as a 11 since that's the next print
 *      position is  (5 + "Test 1" = 11, 5 + 6 characters = 11) )
 *
 *     If <nRow> is greater than the number of lines in the window,
 *     the window is scrolled up one and <nRow> is automatically
 *     assumed to be the last row (now blank).
 *
 *     You are permitted to just pass <cMsg> and let the FT_PrtWin()
 *     function print the text on the next line in the window (based
 *     on the last call to FR_PrtWin() ).
 *  $EXAMPLES$
 *    LOCAL nCounter
 *
 *    FT_OpenWin(1 , 1, 20,70,"Window","Press AnyKey",.T.)
 *    FOR nCounter := 1 TO 20
 *        FT_PrtWin("Test Line "+STR(nCounter,2))
 *    NEXT nCounter
 *    INKEY(0)
 *    FT_ClosWin()
 *
 *  OR
 *    LOCAL nCounter
 *
 *    FT_OpenWin(1 , 1, 20,70,"Window","Press AnyKey",.T.)
 *    FOR nCounter := 1 TO 20
 *        FT_PrtWin(0,0,"Test Line "+STR(nCounter,2))
 *    NEXT nCounter
 *    INKEY(0)
 *    FT_ClosWin()
 * 
 *  $END$
 */

FUNCTION FT_PrtWin(nRow, nCol, cMsg)
    LOCAL nLeft, nRight, nTop, nBottom, nLineSize, lLoaded, lBorder, nLast
    nLast   := LEN(xWinStack)
    IF nLast == 0
       RETURN .F.
    ENDIF
    nLeft   := asc(SUBSTR(xWinStack[nLast],1,1))
    nTop    := asc(SUBSTR(xWinStack[nLast],2,1))
    nRight  := asc(SUBSTR(xWinStack[nLast],3,1))
    nBottom := asc(SUBSTR(xWinStack[nLast],4,1))
    lBorder := IIF(SUBSTR(xWinStack[nLast],5,1)=="1",.T.,.F.)

    IF VALTYPE(nRow) == 'C'
       cMsg  := nRow
       nCol := 1
       nRow := ASC(SUBSTR(xWinCurs[nLast],2,1))+1
    ENDIF
    IF lBorder
        nRight  := IF(nRight<MAXCOL(),nRight -1,nRight)
        nBottom := IF(nBottom<MAXROW(),nBottom -1,nBottom)
    ENDIF
    IF LEN(xWinCurs[nLast])>0
       IF (nCol == 0)
          lLoaded := .T.
          nCol    := ASC(SUBSTR(xWinCurs[nLast],1,1))
       ENDIF
       IF (nRow == 0)
          lLoaded := .T.
          nRow    := ASC(SUBSTR(xWinCurs[nLast],2,1))
       ENDIF
    ENDIF

    nCol := MAX(1,MIN(MAXCOL(),nCol))
    nRow := MAX(1,MIN(MAXROW(),nRow))

    // Error Checking 
    IF nCol >= (nRight - nLeft)  // Wrap around
        nCol := nLeft +1
        IF lLoaded == .T.   // from cursor read.
            nRow := nRow +1
        ENDIF
    ELSE
        nCol := nCol+nLeft
    ENDIF
    IF nRow >= (nBottom - nTop)   
        nRow := nBottom -1
        SCROLL(nTop+1,nLeft+1,nBottom-1,nRight-1,1)   // 1 is up one
    ELSE
        nRow := nRow+nTop
    ENDIF
    nLineSize := (nRight-nCol) - LEN(cMsg)     

    IF nLineSize < 0
       @ nRow, nCol SAY SUBSTR(cMsg,1,(nRight-nCol))
       FT_PrtWin(nRow-nTop+1, 1, SUBSTR(cMsg,(nRight-nCol)+1)) // Recursion
    ELSE
       @ nRow, nCol SAY (cMsg)
    ENDIF
    xWinCurs[nLast] := chr(COL()-nLeft)+chr(ROW()-nTop)
RETURN .T.
