/*
 * File......: EOJ.PRG
 * Author....: Steve Larsen
 * CIS ID....: 76370,1532
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Steve Larsen and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "ftint86.ch"
#include "netto.ch"

/*  $DOC$
 *  $FUNCNAME$
 *      fn_EOJ()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      EndOfJob
 *  $SYNTAX$
 *
 *      fn_EOJ( [ <lProcesses> ] ) -> NIL
 *
 *  $ARGUMENTS$
 *
 *      <lProcesses> is a logical indicating whether the EOJ applies to
 *      all processes on the workstation (.T.) or to the current process
 *      only.  If omitted the default is .F..
 *
 *  $RETURNS$
 *
 *      Nothing.  Any errors can be detected with fn_error().
 *
 *  $DESCRIPTION$
 *
 *      This function cause an EndOfJob (EOJ) command to be sent to the
 *      file server and causes the shell to reset all internal tables for
 *      the application.  The EOJ at the file server releases all resources
 *      (open files, file and record locks, logical locks, semaphores, etc.)
 *      allocated on the file server to the current task.
 *
 *      A program may issue this call whenever the need arises to reset the
 *      network environment to a beginning-of-program state with no files
 *      or records logged or locked.  After issuing this call, a station
 *      must not use any files without first reopening, relogging and
 *      relocking them.
 *
 *      NOTE:  The shell automatically generates the same EOJ action when
 *      an application terminates, and control returns to COMMAND.COM
 *      (all resources allocated to all tasks on the workstation are
 *      released, same as setting <lProcesses> == .T.).
 *
 *  $EXAMPLES$
 *
 *      /* close all active files, locks and network resources */
 *
 *      fn_EOJ( .T. )
 *
 *  $SEEALSO$
 *      fn_setEOJ()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_EOJ( lProcesses )
	LOCAL aRegs[ INT86_MAX_REGS ]
	DEFAULT lProcesses TO .F.

	aRegs[ AX ] := ft_HEX2DEC( "F102" )   // Netware Service F1h, function 2h
	aRegs[ BX ] := IIF( lProcesses, -1, 0 )

	IF !ft_int86( INT21, aRegs )          // logout
		_fnSetErr( EINT86 )
	ENDIF
RETURN(NIL)

/*  $DOC$
 *  $FUNCNAME$
 *      fn_setEOJ()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      SetEndOfJobStatus
 *  $SYNTAX$
 *
 *      fn_setEOJ( [ <lEnable> ] ) -> lPreviousEOJStatus
 *
 *  $ARGUMENTS$
 *
 *      <lEnable> is a logical indicating whether the shell should
 *      automatically set the End Of Job (EOJ) when control returns to the
 *      the root COMMAND.COM (or when all processes end).  If omitted,
 *      the default is .T. (set EOJ).
 *
 *  $RETURNS$
 *
 *      The status of the EOJ flag prior to this function call.
 *
 *  $DESCRIPTION$
 *
 *      Whenever control is returned to COMMAND.COM from a process or
 *      application, the shell will attempt to determine if control has
 *      been returned to the root COMMAND.COM (the one started by DOS at
 *      bootup time).  If so, the shell assumes all processes and
 *      applications have terminated, so sends an EOJ command to the
 *      server that will release all file server resources allocated to
 *      the workstation (open files, physical and logical locks, semphores,
 *      etc.).
 *
 *      This function allows an application to enable or disable the
 *      generation of the EOJ command when control returns the the root
 *      COMMAND.COM.
 *
 *  $SEEALSO$
 *      fn_EOJ()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_setEOJ( lStatus )
	LOCAL aRegs[ INT86_MAX_REGS ]
	DEFAULT lStatus TO .T.

	aRegs[ AX ] := ft_Hex2Dec( "BB" + IIF( lStatus, "01", "00" ) ) // Netware Service BBh

	IF !ft_int86( INT21, aRegs )                  // Call Netware
		_fnSetErr( EINT86 )
  	ENDIF

RETURN ( UNSIGNED( LowByte( aRegs[ AX ] ) ) == 1 )
