/* ===============================================
||
||    Source File: mousmenu.prg
||
||         System: My Mouse Menu Program
||         Author: David Barrett
||   Copyright (c) 1993 Clarity Computer Consultants
||
||      Comments : Source code file for Mouse Menu.
||                 Compile with Clipper 5.x using the /n parameter.
||                 Link with Nanfor and Frankie libraries.
||
||  Procs & Funcs: main()
||               : Sub1()
||               : DispMenu()
||               : EditMenu()
||               : EditItem()
||               : ColorString()
||               : EditColors()
||               : DefaultColors()
||               : ExpColors()
||               : Center()
||               : DispCenter()
||               : CenterNonTrim()
||               : PauseBox()
||               : MakeBox()
||               : DelEmptyArray()
||
|| Documented 09/07/93 at 07:51:27 DOC Version 3.10
=============================================== */

/*           MOUSMENU.EXE has to be called from a batch file called
             MENU.BAT which must look like (assuming that MOUSMENU.EXE
             is in the directory c:\mousmenu):
                 c:
                 cd \mousmenu
                 mousmenu
                 _menu
             ( Menu.bat must be in a directory in the DOS search path,
             or in c:\mousmenu.)
             Mousmenu.exe creates a batch file called _menu.bat which
             performs some desired action, such as running an application
             or some DOS command and then calls MENU.BAT, returning to
             the menu.
                When MOUSMENU.EXE is first run the opening menu will be
             blank.  To create a menu entry, press the F5 key, select
             a line, type in a menu prompt, press N to the submenu
             question, and then type in the DOS batch file commands
             that should be executed when the user selects that menu entry.
                To create sub menus answer Y to the submenu question.
                Menu entries can be selected by clicking on them with
             the mouse, moving the cursor to the entry with the arrow
             keys and pressing ENTER or by typing the first character
             in the menu prompt.
                You are free to use this source code and the resulting
             application but you are forbidden from selling it or using
             any of this source code to create commerical applications.
*/

# include "inkey.ch"
# include "set.ch"
# define cCRLF chr(13) + chr(10)

#define StdColor 1
#define EnhColor 2
#define UnsColor 3
#define BoxColor 4
#define Norm_Color 5
#define OptionST 6
#define OptionSB 7
#define OptionET 8
#define OptionEB 9
#define OptionUT 10
#define OptionUB 11
#define OptionBT 12
#define OptionBB 13
#define InitColor 14
#define SetColors 15
#define ForceBW 16

#command DEFAULT <p> TO <val> [,<pn> TO <valn>] ;
         => ;
         <p> = iif(<p> = NIL, <val>, <p>) ;
         [;<pn> = iif(<pn> = NIL, <valn>, <pn>)]


memvar nExitCode, lRet

/* ===============================================
||
||             FUNCTION main
||
||     Calls : ColorString()
||           : FT_Default()          (function in nanfor.lib)
||           : FT_RestArr()          (function in nanfor.lib)
||           : FT_SaveArr()          (function in nanfor.lib)
||           : PauseBox()
||           : Sub1()
||
||   Purpose : Main routine, saves data to and from disk, calls sub1()
||
||   Returns : nil
||
=============================================== */
function main
  local cFileName, aMenu, nErrorCode := 0, lSelected, aSubs := {}, ;
        nDoSubs := 0, cCurDir := '\' + curdir(), nCurrent := 1, ;
        cOrig_Color
  set( _SET_SCOREBOARD, .F.)
  set cursor off
  release GetList
  cOrig_Color := setcolor( ColorString( Norm_Color ) )
  cls
  if ! file('MENU.DAT')
    aMenu := { { space( 10 ), .F. },;
               { space( 10 ), .F. } ;
             }
  else
    aMenu := FT_RestArr('Menu.dat',@nErrorCode)
    if nErrorCode # 0
      PauseBox('Error reading menu data')
    endif
  endif
  if file( 'menulevl.dat' )
    aSubs := FT_RestArr( 'menulevl.dat', @nErrorCode )
    if nErrorCode # 0
      PauseBox('Error reading level data')
    endif
  endif
  if file( 'current.dat')
    nCurrent := FT_RestArr( 'current.dat', @nErrorCode )[ 1 ]
    if nErrorCode # 0
      PauseBox('Error reading current data')
    endif
  endif
  aMenu := Sub1( aMenu, 'Clarity Computer Consultants','SELECT PROGRAM',;
                 FT_Default(), cCurDir, @lSelected, @aSubs, @nDoSubs, ;
                 @nCurrent )
  FT_SaveArr( aSubs, 'menulevl.dat', @nErrorCode )
  if nErrorCode # 0
    PauseBox('Error writing level data')
  endif
  FT_SaveArr( aMenu, 'Menu.Dat', @nErrorCode)
  if nErrorCode # 0
    PauseBox('Error writing menu data')
  endif
  FT_SaveArr( { nCurrent }, 'current.dat', @nErrorCode )
  if nErrorCode # 0
    PauseBox( 'Error writing current data')
  endif
  setcolor( cOrig_Color )
  set cursor on
return nil


/* ===============================================
||
||             FUNCTION Sub1
||
|| Called by : main()
||           : Sub1()
||
||     Calls : ColorString()
||           : DispMenu()
||           : EditMenu()
||
||   Purpose : Function which displays, executes or edits one menu level.
||             When a submenu is selected it calls itself recursively
||             with subarray for that level.
||
|| Parameters: aMenu - array containing menu info including menu prompts,
||                     commands to be written to batch file for each
||                     prompt, and subarrays containing all info for any
||                     sub menus.
||           : cTitle - Title to be displayed at top of screen.
||           : cMenuName - menu name for selecting screen (not editing
||                         screen)
||           : cDrive - drive on which to create _MENU.BAT.
||           : cPath - directory on cDrive where _MENU.BAT is to be
||                     created.
||           : lSelected - pass by reference to return whether this menu
||                         level is exiting because choice selected and
||                         _MENU.BAT created, rather than aborted.  Used
||                         when returning from Sub1() to Sub1() rather
||                         than on main menu level when Sub1() returns
||                         to main().
||           : aSubs - array keeping track of submenu levels so that we
||                     can return to correct submenu after executing
||                     choice from that submenu.
||           : nDoSubs - submenu count.  Will be less than len( aSubs )
||                       when we are returning to correct submenu after
||                       executing choice from submenu.  At all other
||                       times then nDoSubs == len( aSubs)
||           : nCurrent - current menu choice so we can highlight
||                        correct menu choice after executing that
||                        choice.
||
||   Returns : aMenu - will be different from passed aMenu if editing
||                     has been done.
||
=============================================== */
static function Sub1( aMenu, cTitle, cMenuName, cDrive, cPath, ;
                      lSelected, aSubs, nDoSubs, nCurrent )
  local nK, ThisLen, nReturnChoice, cOrig_Color, cBottom, ;
        cScreen, cString, lDone := .F., lDoSubs := .F.
  DEFAULT lSelected to .F.
  cBottom := 'RETURN - run / ESC - quit / F5 - edit / F9 - set colors'
  do while ! lDone
    if nDoSubs  < len( aSubs ) // we must be initially returning to submenu
      nDoSubs ++
      nReturnChoice := aSubs[ nDoSubs ]  // select submenu
      lDosubs := .T.     // set flag so we don't add this level to aSubs
    else                 // we are not initially returning to submenu
      lDoSubs := .F.     // reset flag, display this menu level
      nReturnChoice := DispMenu( aMenu, cMenuName, cTitle, cBottom, ;
                              @nCurrent )
    endif
    if nReturnChoice = 0        // Esc key or right menu button pressed
      cString := '@echo off' + cCRLF + 'echo .' + cCRLF + ;
                 'echo Menu Program Terminated' + cCRLF + 'echo .' + cCRLF
      memowrit('_MENU.BAT',cString)
      lDone := .T.           // return to previous level or
      lSelected := .F.       //  DOS if this is mail level
      nCurrent := 1
    elseif nReturnChoice = K_F5         // F5 key pressed
      aMenu := EditMenu( aMenu, @nCurrent )
    elseif nReturnChoice = K_F9        // F9 key pressed
      ColorString(SetColors)
    elseif  nReturnChoice > 0          // choice made
      if aMenu[ nReturnChoice, 2 ]  // submenu chosen
        if ! lDosubs             // if not returning to previous submenu
          aadd( aSubs, nReturnChoice )  // keep aSubs current
          nDoSubs ++                   // keep nDoSubs current
          nCurrent := 1           // going to submenu, not initial return
        endif
        aMenu[ nReturnChoice, 3 ] := Sub1( aMenu[ nReturnChoice, 3 ], ;
                       cTitle, cMenuName, cDrive, cPath, @lSelected, ;
                       @aSubs, @nDoSubs, @nCurrent )
        if lSelected // menu item chosen in submenu -- exit
          lDone := .T.
        else    // submenu escaped, remove last level from aSubs
          asize( aSubs, len( aSubs ) - 1 )
          nDosubs --
        endif
      else      // choice was not submenu, create _MENU.BAT
        cString := ''
        for nk := 3 to 13
          if len( aMenu[ nReturnChoice ] ) >= nk
            if ! empty( aMenu[ nReturnChoice, nk ] )
              cString += aMenu[ nReturnChoice, nk ] + cCRLF
            endif
          endif
        next
        cString += cDrive + ':' + cCRLF + 'cd ' + cPath + cCRLF + ;
                   'MENU' + cCRLF
        memowrit('_MENU.BAT',cString)
        lDone := .T.
        lSelected := .T.
        nCurrent := nReturnChoice
      endif
    endif
  enddo
return aMenu


/* ===============================================
||
||             FUNCTION DispMenu
||
|| Called by : Sub1()
||           : EditMenu()
||
||     Calls : adverMenu()           (function in frankie.li)
||           : advm_Abort()          (function in frankie.li)
||           : advm_BoxAttr()        (function in frankie.li)
||           : advm_Color()          (function in frankie.li)
||           : advm_Current()        (function in frankie.li)
||           : advm_InitSel()        (function in frankie.li)
||           : advm_Keys()           (function in frankie.li)
||           : Center()
||           : CenterNonTrim()
||           : ColorString()
||           : FT_Shadow()           (function in nanfor.lib)
||
||   Purpose : Displays current menu level, returns if aborted, choice
||             selected or function key pressed.
||
|| Parameters: aMenu - array containing all info for this level and all
||                     submenus below.
||           : cMenuName - menu name to display
||           : cTitle - program title to diplay at top of screen.
||           : cBottom - prompt to display at bottom of screen.
||           : nCurrent - must be passed by reference, returns current
||                        menu choice.
||
||   Returns : nReturnChoice - 0 if aborted, menu choice or function key
||
=============================================== */
function DispMenu( aMenu, cMenuName, cTitle, cBottom, nCurrent )
  local nMaxLenPrompt := 0, nNumPrompts,lDoubleSpaced, PromptSRow, ;
        PromptSCol, BoxRow, BoxCol, nReturnChoice, cOrig_Color, ;
        cScreen, nk, nRow, aPrompts := {} , bConfig
  private nExitCode
  nExitCode := 0
  bConfig := { || advm_Color( {  "B/W" , ;    // standard color
                              "N/BG" , ;   // enhanced color
                              "N/W" , ;    // box color
                              "R" ;        // trigger color
                              } ;
                            ), ;
                  advm_InitSel( nCurrent ) , ;
                  advm_Keys( { K_F2, K_F3, K_F4, K_F5, K_F6, K_F7, ;
                              K_F8, K_F9, K_F10 }, ;
                             { |n,k| nCurrent := advm_Current(), ;
                              advm_Abort(), nExitCode := k };
                           ) , ;
                  advm_BoxAttr( { ;
                              "         " ,; // "͸Գ "  frame
                              .f.,;    // no shadow
                              .f.,;    // do not explode
                              4 ;      // pad with 1 spaces
                              } ) ;
             }
  save SCREEN to cScreen
  cls
  cOrig_Color := setcolor( ColorString( BoxColor ) )
  nNumPrompts := if( len( aMenu ) > 13, 13, len( aMenu ) )
  lDoubleSpaced := .F.
  aeval( aMenu, { | aField | nMaxLenPrompt := if( ( len( aField[ 1 ] ) > nMaxLenPrompt ),;
                              len( aField[ 1 ] ), nMaxLenPrompt ) } )
  PromptSRow := 11 - int( nNumPrompts / 2 )
  PromptSCol = int( ( 66 - nMaxLenPrompt ) / 2)
  BoxRow := PromptSRow
  BoxCol := if( PromptSCol > 23, 23, PromptSCol - 1 )
  if ( BoxRow > 0 ) .AND. ( BoxCol > 0 )
    BoxRow -=  2
    BoxCol -= 4
    if ( BoxRow > 0 ) .AND. ( BoxCol > 0 )
      setcolor(ColorString(BoxColor))
      @ BoxRow, BoxCol clear to ( 25 - BoxRow), ( 80 - BoxCol )
      @ BoxRow, BoxCol to (25 - BoxRow), (80 - BoxCol) DOUBLE
      if BoxRow > 1
        FT_Shadow( BoxRow, BoxCol, 25 - BoxRow, 80 - BoxCol)
      endif
      setcolor(ColorString(EnhColor))
      CenterNonTrim( 23 - BoxRow, 'Use arrow keys to move light bar')
      CenterNonTrim(25 - BoxRow, ' Press Enter, letter or mouse to select')
      CenterNonTrim(BoxRow, ' ' + alltrim(cMenuName) + ' ')
      BoxRow -= 2
      BoxCol -= 4
      if (BoxRow > 0) .AND. (BoxCol > 0)
        setcolor(ColorString(BoxColor))
        @ BoxRow, BoxCol to (25 - BoxRow), (80 - BoxCol) DOUBLE
        if BoxRow > 1
          FT_Shadow( BoxRow, BoxCol, 25 - BoxRow, 80 - BoxCol)
        endif
        setcolor(ColorString(EnhColor))
        CenterNonTrim(BoxRow, ' ' + alltrim(cTITLE) + ' ')
      endif
    endif
  endif
  setcolor( ColorString( BoxColor ) )
  for nK = 1 to nNumPrompts
    aadd( aPrompts, chr( nk + 64 ) + ') ' + aMenu[ nK, 1 ] )
  next
  Center( 24, cBottom )
  nReturnChoice := adverMenu( PromptSRow, PromptSCol, aPrompts, , , bConfig )
  if nReturnChoice # 0
    nCurrent := nReturnChoice
  endif
  nReturnChoice := iif( nExitCode == 0, nReturnChoice, nExitCode )
  setcolor( cOrig_Color )
  RESTORE SCREEN from cScreen
return nReturnChoice


/* ===============================================
||
||             FUNCTION EditMenu
||
|| Called by : Sub1()
||           : EditItem()
||
||     Calls : DelEmptyArray()
||           : DispMenu()
||           : EditItem()
||
||   Purpose : Allows editing of current menu level and any submenus
||             which are selected.
||
|| Parameters: aMenu - array containing all info for this menu level and
||                     any submenus
||           : nCurrent - pass by reference returns last choice edited so
||                        that choice can be highlighted in Sub1() when
||                        edit screen exited.
||
||   Returns : aTemp - edited array
||
=============================================== */
function EditMenu( aMenu, nCurrent )
  local nLenArray := len(aMenu), nChoice
  local aTemp := {}, nk, cScreen, cOrig_Color, lDone := .F.
  local cBottom := 'RETURN-Edit / ESC-Quit / F3 - Insert /F7 - Del'
  local cMenuName := 'Clarity Computer Consultants', cTitle := 'Editing'
  for nk := 1 to 13
    if nk <= nLenArray
      aadd( aTemp, aMenu[ nk ] )
    else
      aadd( aTemp, { space( 10 ) } )
    endif
  next
  do while ! lDone
    nChoice := DispMenu( aTemp, cMenuName, cTitle, cBottom, @nCurrent )
    do case
    case nChoice > 0             // Choice made
      aTemp[ nChoice ] := EditItem( aTemp[ nChoice ], @nCurrent )
    case nChoice = K_F3         // Insert
      for nk := 13 to nCurrent + 1 STEP -1
        aTemp[ nk ] := aTemp[nk - 1]
      next
      aTemp[ nCurrent ] := { space( 10 ) }
    case nChoice = K_F7    // Delete
      for nk := nCurrent to 12
        aTemp[ nk ] := ATemp[ nk + 1 ]
      next
      aTemp[ 13 ] := { space( 10 ) }
    case nChoice = 0        // Esc pressed
      lDone := .T.
    endcase
  enddo
return DelEmptyArray( aTemp )


/* ===============================================
||
||             FUNCTION EditItem
||
|| Called by : EditMenu()
||
||     Calls : Center()
||           : ColorString()
||           : EditMenu()
||           : MakeBox()
||
||   Purpose : Edit menu item selected.  If submenu recursively calls
||             EditMenu().  If not submenu, edit aEntry lines.
||
|| Parameters: aEntry - subarray from aMenu for this entry.
||           : nCurrent - current menu choice.  Pass by reference.
||
||   Returns :
||
=============================================== */
function EditItem( aEntry, nCurrent )
  local nLenArray, nk, cScreen, lSave := .T., GetList := {}
  local aTemp := aclone( aEntry ), cOrig_Color
  nLenArray = len( aTemp )
  if len( aTemp ) < 1
    aadd( aTemp, space( 25 ) )
  else
    aTemp[ 1 ] := padr( aTemp[ 1 ], 25)
  endif
  if len( aTemp ) < 2
    aadd( aTemp, .F. )
  endif
  cScreen := savescreen( 5, 10, 20, 72 )
  cOrig_Color := MakeBox( 5, 10, 19, 70, ColorString( BoxColor ) )
  Center(5,'Edit Item')
  @ 8, 15 SAY 'PROMPT             :' GET aTemp[1]
  @ 9, 15 SAY 'Is this a SubMenu ? ' GET aTemp[2] PICTURE 'Y'
  set cursor on
  read
  set cursor off
  if lastkey() # K_ESC
    Center(6, 'Enter batch file commands')
    aTemp[1] = trim(aTemp[1])
    if aTemp[2]
      if len( aTemp ) < 3
        aadd( aTemp, {} )
      endif
      aTemp[ 3 ] := EditMenu( iif( valtype( aTemp[ 3 ] ) = 'A', ;
                              aTemp[ 3 ],{} ), @nCurrent )
    else
      for nk := 3 to 8
        if len( aTemp ) < nk
          aadd( aTemp, space( 30 ) )
        endif
        aTemp[ nk ] := padr( aTemp[ nk ], 30 )
        @ 7 + nk, 25  GET aTemp[ nk ]
      next
      set cursor on
      read
      set cursor off
      lSave := iif(lastkey() = K_ESC,.F.,.T.)
    endif
    if lSave
      aEntry := aclone( aTemp )
    endif
  endif
  setcolor( cOrig_Color )
  restscreen( 5, 10, 20, 72, cScreen )
return aEntry

/* ===============================================
||
||             FUNCTION ColorString
||
|| Called by : main()
||           : Sub1()
||           : DispMenu()
||           : EditItem()
||           : PauseBox()
||
||     Calls : DefaultColors()
||           : EditColors()
||           : FT_RestArr()          (function in nanfor.lib)
||           : FT_SaveArr()          (function in nanfor.lib)
||           : PauseBox()
||
||   Purpose : stores color strings in static array, returns color
||             string when called, also can invoke color string
||             editing screen.
||
|| Parameters: nMode
||
||   Returns : cRet - color string to pass to setcolor()
||
=============================================== */
function ColorString( nMode )
  static aColorString[ OptionBB ]
  local cScreen, cOrig_Color, cRet, nError := 0 ,;
       cFileName := getenv( "COLORPATH" ) + 'COLORS.DAT'
  setblink(.F.)
  if ( nMode = ForceBW ) .OR. ( empty( aColorString[ Norm_Color ] ) )
    if (file(cFileName)) .AND. ( nMode # ForceBW )
      aColorString := FT_RestArr(cFileName, @nError)
      if nError # 0
        PauseBox('Error reading colors from disk.  DOS error ' + ;
                  str(nError))
        return NIL
      endif
    else
      aColorString := DefaultColors( aColorString, nMode )
    endif
  endif
  if nMode = SetColors
    aColorString := EditColors( aColorString )
    FT_SaveArr( aColorString, cFileName, @nError )
    if nError # 0
      PauseBox( 'Error saving colors to disk.  DOS error ' + ;
                str( nError ) )
    endif
  elseif (nMode # ForceBW ) .AND. ( len( aColorString ) >= nMode )
    cRet := aColorString[ nMode ]
  endif
return cRet

/* ===============================================
||
||             FUNCTION EditColors
||
|| Called by : ColorString()
||
||     Calls : ExpColors()
||           : MakeBox()
||
||   Purpose :
||
|| Parameters: aColorString
||
||   Returns :
||
=============================================== */
static function EditColors( aColorString )
  local cScreen, cOrig_Color, GetList := {}
  cScreen = savescreen( 5, 18, 21, 64 )
  cOrig_Color := MakeBox( 5, 18, 20, 62, aColorString[ BoxColor ] )
  @ 7, 20 SAY ' BLACK      N         BLUE       B'
  @ 8, 20 SAY ' GREEN      G         CYAN       BG'
  @ 9, 20 SAY ' RED        R         MAGENTA    RB'
  @ 10,20 SAY ' BROWN      GR        WHITE      W'
  @ 11,20 SAY ' GRAY       N+        YELLOW     GR+'
  @ 12,20 SAY ' BLANK      X         UNDERLINE  U'
  @ 13,20 SAY ' INVERSE    I'
  @ 15, 20 SAY 'Standard Color ' GET aColorString[ OptionST ]
  @ 15, 40 SAY '/' GET aColorString[ OptionSB ]
  @ 16, 20 SAY 'Enhanced Color ' GET aColorString[ OptionET ]
  @ 16, 40 SAY '/' GET aColorString[ OptionEB ]
  @ 17, 20 SAY 'Unselect Color ' GET aColorString[ OptionUT ]
  @ 17, 40 SAY '/' GET aColorString[ OptionUB ]
  @ 18, 20 SAY 'Box Color      ' GET aColorString[ OptionBT ]
  @ 18, 40 SAY '/' GET aColorString[OptionBB]
  read
  if lastkey() # 27
    aColorString := ExpColors(aColorString)
  endif
  setcolor(cOrig_Color)
  restscreen(5,18,21,64,cScreen)
return aColorString

/* ===============================================
||
||             FUNCTION DefaultColors
||
|| Called by : ColorString()
||
||     Calls : ExpColors()
||
||   Purpose :
||
|| Parameters: aColorString
||           : nMode
||
||   Returns :
||
=============================================== */
static function DefaultColors(aColorString, nMode)
  if iscolor() .AND. ( nMode # ForceBW )
    aColorString[ OptionST ] = 'GR+'
    aColorString[ OptionSB ] = 'B  '
    aColorString[ OptionET ] = 'GR+'
    aColorString[ OptionEB ] = 'R  '
    aColorString[ OptionUT ] = 'B  '
    aColorString[ OptionUB ] = 'BG '
    aColorString[ OptionBT ] = 'N  '
    aColorString[ OptionBB ] = 'W  '
  else
    aColorString[ OptionST ] = 'W  '
    aColorString[ OptionSB ] = 'N  '
    aColorString[ OptionET ] = 'N  '
    aColorString[ OptionEB ] = 'W  '
    aColorString[ OptionUT ] = 'W  '
    aColorString[ OptionUB ] = 'N  '
    aColorString[ OptionBT ] = 'N  '
    aColorString[ OptionBB ] = 'W  '
  endif
  aColorString := ExpColors( aColorString )
return aColorString

/* ===============================================
||
||             FUNCTION ExpColors
||
|| Called by : EditColors()
||           : DefaultColors()
||
||   Purpose :
||
|| Parameters: aColorString
||
||   Returns :
||
=============================================== */
static function ExpColors( aColorString )
  aColorString[ StdColor ] := trim( aColorString[ OptionST ] ) + '/' + ;
       trim( aColorString[ OptionSB ] )
  aColorString[ EnhColor ] := trim( aColorString[ OptionET ] ) + '/' + ;
       trim(aColorString[OptionEB])
  aColorString[ UnsColor ] := trim(aColorString[OptionUT]) + '/' + ;
       trim(aColorString[OptionUB])
  aColorString[ BoxColor ] := trim(aColorString[OptionBT]) + '/' + ;
       trim(aColorString[OptionBB]) + ','+ ;
       aColorString[ UnsColor ] + ',,,' + aColorString[ EnhColor ]
  aColorString[ Norm_Color ] := aColorString[ StdColor ] + ',' + ;
       aColorString[ EnhColor ] + ',,,' + aColorString[ UnsColor ]
return aColorString


/* ===============================================
||
||             FUNCTION Center
||
|| Called by : DispMenu()
||           : EditItem()
||
||   Purpose : @ nRow, Mcol SAY TRIM(cString) centered
||
|| Parameters: nRow
||           : cString
||
||   Returns :
||
=============================================== */
function Center( nRow, cString )
  local nL
  cString := alltrim( cString )
  nL := len( cString )
  if nL > 58
    cString := left( cString, 58 )
    nL := 58
  endif
  @ nRow, ( maxcol() - nL) / 2 SAY cString
return nL

/* ===============================================
||
||             FUNCTION DispCenter
||
|| Called by : PauseBox()
||
||   Purpose : Displays centered string regardless of SET DEVICE setting.
||
|| Parameters: nRow
||           : cString
||
||   Returns :
||
=============================================== */
function DispCenter(nRow, cString)
  local nL, nSaveRow, nSaveCol
  cString := alltrim(cString)
  nL := len(cString)
  if nL > 58
    cString := left(cString,58)
    nL := 58
  endif
  nSaveRow := row()
  nSaveCol := col()
  setpos( nRow, ( maxcol() - nL ) / 2 )
  dispout( cString )
  setpos( nSaveRow, nSaveCol)
return nL

/* ===============================================
||
||             FUNCTION CenterNonTrim
||
|| Called by : DispMenu()
||
||   Purpose : @ MRow, Mcol SAY Mstring centered
||
|| Parameters: nRow
||           : cString
||
||   Returns :
||
=============================================== */
function CenterNonTrim( nRow, cString )
  local nL
  nL := len( cString )
  if nL > 58
    cString := left( cString, 58 )
    nL := 58
  endif
  @ nRow, ( maxcol() - nL ) / 2 SAY cString
return nL


/* ===============================================
||
||             FUNCTION PauseBox
||
|| Called by : main()
||           : ColorString()
||
||     Calls : ColorString()
||           : DispCenter()
||           : MakeBox()
||
||   Purpose : Pops up box, displays message, waits for key press,
||             restores screen
||
|| Parameters: cMessage
||
||   Returns :
||
=============================================== */
function PauseBox( cMessage )
  local cScreen, cOrig_Color
  cScreen = savescreen( 10, 10, 16, 71 )
  while inkey() <> 0   // Remove any pending keypresses
  end
  cOrig_Color := MakeBox( 10, 10, 15, 70, ColorString( BoxColor ) )
  DispCenter( 12, cMessage )
  DispCenter( 13, 'Press any key to continue' )
  inkey( 0 )
  setcolor( cOrig_Color )
  restscreen( 10, 10, 16, 71, cScreen )
return .T.


/* ===============================================
||
||             FUNCTION MakeBox
||
|| Called by : EditItem()
||           : EditColors()
||           : PauseBox()
||
||     Calls : FT_Shadow()           (function in nanfor.lib)
||
||   Purpose : Draws shadowed box on screen using passed coordinates
||             and color string, Returns original color setting so
||             it can be restored by calling function.
||
|| Parameters: nUp
||           : nLeft
||           : nDown
||           : nRight
||           : cColor
||
||   Returns :
||
=============================================== */
function MakeBox( nUp, nLeft, nDown, nRight, cColor )
  local cOrig_Color
  @ nUp, nLeft CLEAR to nDown + 1, nRight + 1
  cOrig_Color := iif( cColor # NIL, setcolor(cColor), setcolor() )
  @ nUp, nLeft, nDown, nRight  BOX "͸Գ "
  FT_Shadow( nUp, nLeft, nDown, nRight)
return cOrig_Color

/* ===============================================
||
||             FUNCTION DelEmptyArray
||
|| Called by : EditMenu()
||
||   Purpose : Returns array in which empty elements have been removed.
||
|| Parameters: aArray
||
||   Returns :
||
=============================================== */
function DelEmptyArray( aArray )
  local nk, lDel, aTemp := {}
  for nk := 1 to len( aArray )
    lDel := .F.
    if valtype( aArray[ nk ] ) = 'A'
      if ( len( aArray[ nk ] ) = 0 ) .OR. ( empty( aArray[ nk, 1 ] ) )
        lDel := .T.
      endif
    elseif valtype( aArray[ nk ] ) = 'C'
      if empty( aArray[ nk ] )
        lDel := .T.
      endif
    elseif valtype( aArray[ nk ] ) = 'U'
      lDel := .T.
    endif
    if ! lDel
      aadd( aTemp, aArray[ nk ] )
    endif
  next
return aTemp

*:   EOF: mousmenu.prg
