
/* -- MENU.PRG 

   AUTHOR : Rick Hellewell                                   
   DATE   : June 17, 1993  11:44 am
   NOTICE : Copyright (c) 1991 - 1993 by Rick Hellewell
            CIS ID 71331,514 or 70724,3300
            All Rights Reserved                           
   VERSION: Clipper 5.01                                      
   PURPOSE: Menuing program using MENU.DBF                   
   Notes  : Released into the public domain at no charge 
            to users.  
            Comments via CIS welcome .... IAP/BIW 
            (It Ain't Perfect, But It Works!)
*/ 

procedure menu()

/* builds an ACHOICE menu from a MENU.DBF file

   MENU.DBF structure is:
      #  FIELD_NAME FIELD_TYPE FIELD_LEN FIELD_DEC FIELD_DESC      D_TEXT           D_LEN D_LEN_DEC D_ORDER
      1  L1         N                  2         0 Level 1         Lev 1             2         0        
      2  L2         N                  2         0       2             2             2         0        
      3  L3         N                  2         0       3             3             2         0        
      4  L4         N                  2         0       4             4             2         0        
      5  PROGNAME   C                 30         0 Program Name    Program Function 30         0        
      6  PROGDESC   C                 40         0         Desc.   Description       0         0        
      7  SECURITY   C                 10         0 Security Level  Security         10         0        

    Notes:  FIELD_DESC and D_* field names are used with my Generic TBrowsing 
               routine, and are not required by MENU.PRG 
            PROGNAME is left blank in database if there are additional levels          
            PROGDESC is the description that is shown on the screen
            SECURITY required only if you want security levels for menu choices

   Index file (required) for MENU.DBF is MENU.NTX;
      MENU.NTX = str(L1,2) + str(L2,2) + str(L3,2) + str(L4,2)

   This program is built to allow a menu that is up to 3 levels deep.  
   To have a deeper menu structure, these changes are required:
      MENU.DBF: add additional field name(s) "Lx" (x = 5 or more)
      variable MENUFILTER: add more "0"'s to var to reflect additional levels
      function BLDFILTER: change line "if len(newmenflt) > 4" to number of levels

   Example data:

L1 L2 L3 L4 PROGNAME-- PROGDESC-- SECURITY--
 0  0  0  0                        Main Menu Title        0
 0  1  0  0                        Menu Choice 1          0
 0  1  1  0 PROG1()                   Menu Choice 1-1     0
 0  1  2  0 PROG2()                   Menu Choice 1-2     0
 0  1  3  0 PROG3()                   Menu Choice 1-3     0
 0  1  5  0 PROG4()                   Menu Choice 1-4     0
 0  2  0  0                        Menu Choice 2          0
 0  2  1  0                           Menu Choice 2-1     0
 0  2  1  1 PROG4()                      Choice 2-1-1     0
 0  2  1  2 PROG5()                      Choice 2-1-2     0
 0  2  1  3 PROG6()                      Choice 2-1-3     0
 0  2  1  4 PROG7()                      Choice 2-1-4     0
          
The PROGDESC data is indented in the database so that I can keep track
of the levels.  The leading spaces are trimmed off when the items are 
displayed.

Note that menu choices that call submenus do not have any PROGNAME
values.  The PROGDESC text is used as the menu's heading.

When the program runs, you can press the first letter of the PROGDESC
text to highlight that choice, or use the arrow keys.  The <Esc> key
is used to return to the previous menu.  <Esc> at the main (top) menu
exits MENU.PRG.

*/

# include "inkey.ch"

// compiles the PROGNAME expression and runs it
#ifndef COMPILE
   #define COMPILE(cExpr)  & ( ' { | | ' + cExpr + ' } ' )
#endif

// variable declarations
local i, maxchoice
local menuhelp      := "Select choice, press <Enter>.  Press <Esc> to exit."
local progtorun     := ""         // name of program to run
private lfilter     := ""         // logical filter for MENU.DBF
private menufilter := "0*00"      // menu filter pattern, "*" is wildcard
private acProg      := {}         // program to run array
private acDesc      := {}         // menu items (program descriptions)
private acSecure    := {}         // program security level flag
private menuchoice  := 0          // menu choice returned by ACHOICE
#ifndef userlevel
   private userlevel   := "0"     // current user's security level  
                                  // should be declared in a parent program
#endif

// loop to display menu choices and run corresponding program
do while .t.
   // reset the menu array values
   acProg = {}                    // program name
   acDesc = {}                    // program desc
   acSecure = {}                  // program security level flag

   // create lfilter from current menufilter
   lfilter := bldfilter(menufilter)             

   // build current menu array using MENU.DBF with lfilter 
   buildmenu(lfilter)

   // extract the menu title from the array (array item 1)
   // then delete it from the menu array
   menutitle = alltrim(upper(substr(acDesc[1],4)))
   adel(acDesc,1)
   adel(acProg,1)
   adel(acSecure,1)

   // display the current menu using achoice, set menuchoice based on response
   setcolor(colr)
   scrnhead(menutitle)
   // @ 2,0 say padc(menutitle,maxcol())
   @ 22,0 say padc("Drive has " + alltrim(transform(diskspace(),"999,999,999")) + " bytes free", maxcol() )
   text = " Today is " + cdow(date()) + ", " + cmonth(date()) + " " + alltrim(str(day(date()))) + ", " + alltrim(str(year(date()))) + " "
   @ 23, (maxcol()/2)-(len(text)/2) say text
   @ 24,0 say padc(menuhelp, maxcol())
   menuchoice = achoice(4,20,22,60, acDesc, acSecure)

   // adjust the menufilter pattern based on the user's choice
   if menuchoice == 0                  // need to move to a higher menu level
      menufilter= moveup(menufilter)
   else                                // need to move to a lower menu level
      ProgToRun = alltrim(acProg[menuchoice])

      // now run the program if it's there, otherwise ignore it
      if len(ProgToRun) > 0
         // display menu choice at bottom of screen
         @ 24,0 say padc("Menu Choice " + acDesc[menuchoice] , maxcol())
         eval(COMPILE( ProgToRun ))
      else
         menufilter = movedown(menufilter, menuchoice)
      endif
   endif

   // check for final quit, must be time to exit
   if menufilter == "*000" 
      // ZYESNO() asks the question, accepts and returns Y or N
      // ocolor := setcolor()
      // if zyesno(20,"Are you sure you want to exit SCANBOSS?") > 1
      //    setcolor(ocolor)              // all done, get out of here
         exit
      // else                             // stick around
      //    menufilt := "0*00"
      // endif
      // setcolor(ocolor)
   endif

enddo                   // while .t. (menu loop)

// clean up and go home    
clear screen
return

// - //
//       supporting functions                                       //
// - //

static function buildmenu(lfilter)

   // builds the filter for the current menu level

   local CBFilt                     // code block for the filter statement
   local counter                    // just a counter
   
   // open MENU.DBF each time, in case changed by user in another related prg
   // if MENU database not changeable by any program, put USE statement
   // at top of main MENU program, and delete the CLOSE statement at the
   // end of this function
   use menu index menu new
   
   // build the temp menu array based on the lfilter
   CBFilt := {|| &lFilter }         // codeblock it
   set filter to eval(CBFilt)       // evaluate it and set the filter
   go top                           // make sure that we're at the first match

   // get number of records that match the filter
   counter = 0
   do while .not. eof()
      counter ++
      skip
   enddo
   go top
   
   // size the arrays to the number of records matching the filter
   asize(acProg, counter)
   asize(acDesc, counter)
   asize(acSecure, counter)
   
   // build the current menu array from the filtered database
   counter = 1
   do while .not. eof()

      // the program to run for this choice
      acProg[counter] = alltrim(menu->progname)
      
      // the menu description text
      //    put a numbered choice for each menu choice except the first one
      //       which will be made into the menu title
      acDesc[counter] = if(counter <> 0, ;
            ltrim(str(counter-1,2))+". " + ltrim(menu->progdesc), ;
                                       ltrim(menu->progdesc) )
      // set the security flag according to the user's security level
      acSecure[counter] = if(userlevel $ menu->security, .t., .f.)

      counter ++
      skip
   enddo

   // close the database when done
   // if database opened in main MENU routine, delete this line   
   close menu

return NIL

// - //

static function bldfilter(newmenflt)

   // builds the current menu array using the menufilter pattern
   local i  := 0              // just a counter
   local onechar := ""        // one character from the filter

   // clear out the current filter
   lfilter := ""              
   
   // if filter pattern too long, force it to the top menu
   if len(newmenflt) > 4           
      newmenflt = "0*00"
   endif
   
   // build the new menu filter condition
   for i = 1 to len(newmenflt)
      // extract a character from the filter pattern
      onechar = substr(newmenflt,i,1)
      if onechar == "*"       // wildcard, leave it alone
      else                    // build and add to filter expression 
         lfilter += " .and. menu->L" + ltrim(str(i)) + " = " + onechar
      endif
   next i
   
   if len(lfilter) > 0        // gets rid of the first " .and. "
      lfilter = substr(lfilter, 8)
   endif

return (lfilter)

// - //

static function movedown(newmenflt, menuchoice)

   // sets filter pattern one level higher, returns the new filter pattern

   // figure out the current deep position
   local deep := at("*", newmenflt)
   
   // put the choice number at the deep position
   newmenflt = stuff(newmenflt, deep, 1, alltrim(str(menuchoice)))

   // put an asterisk to right of the choice number
   deep ++
   newmenflt = stuff(newmenflt, deep, 1, "*")

return (newmenflt)

// - //

static function moveup(newmenflt)

   // sets filter pattern one level higher, returns the new filter pattern

   // figure out the current deep position
   local deep := at("*", newmenflt)

   // put a "0" at the deep position
   newmenflt = stuff(newmenflt, deep, 1, "0") 

   // put an asterisk to left of the choice number
   deep --
   newmenflt = stuff(newmenflt, deep, 1, "*")

return (newmenflt)


//  //
//                            end of file                            //
//  //
