/* 
        Function: MBDBEDIT()
         Version: 12/09/91 Original
                  12/10/91 Corrected DE_HITTOP & DE_HITBOTTOM (I was 
                           converting to using DBEDIT.CH and got the 
                           constants reversed)
                  03/12/92 Moved "main loop" operations into a sep UDF so VMM 
                           would create a new segment boundry.  This helped
                           to eliminate "not enough memory"  errors on low
                           memory and large browse situations i.e. large app
                           with network drivers, TSRs, etc. when browse
                           display of full screen with VGA in 50 line mode.
                  04/14/92 Correction: When in DE_START mode added ability to
                           DE_ABORT the browse.
                  09/11/92 1. Changed <aheadings> array default init process. 
                              It was possible to get an "out of bounds" error 
                              in certain situations.  
                           2. Corrected display for Memo fields.
                           3. Added column "wrap" feature.

          Author: Mark Butler, Rio Linda (Sacramento), California
   Compuserve ID: 70026,612
           Phone: (916) 991-6230
 Clipper Version: 5.01
    Compile With: /m/n/w  (include /b if you want debugger info included)


BACKGROUND:

MBDBEDIT() IS A DIRECT REPLACEMENT FOR Clippers' DBEDIT().  I created it for
use with all the code I have written that already works fine with DBEDIT(), 
but I wanted a little tweak here and there.  That includes Summer/87 
applications I was converting over.  I wasn't interested in rewriting all of 
my working code as a TBrowse object class, it seemed simplier to rewrite 
DBEDIT() (in TBrowse) and extend its functionality.

MBDBEDIT() should function EXACTLY like Clippers' DBEDIT(), so you can 
replace it with any call to DBEDIT() in your application.  You don't have to 
change anything else, but you will probably want to.  The point is you can 
add refinements to your application at your own pace.

The power of MBDBEDIT() is that it supplies a third parameter to your 
user-defined function.  That is the TBrowse object itself.  With access to 
the browse object you have all the control available over the browse (i.e 
MBDBEDIT()).  Additionally, there are features for a time out; program 
branching in the event of a time out; freezing columns in the display; and
column wrapping effects.  You can even change the browse object prior to 
display (your UDF is called with a mode of "start up" before anything is 
displayed).  You can even supply your UDF as a code block so that it can be 
a STATIC function instead of global.  Like DBEDIT(), MBDBEDIT() is fully 
re-entrant, which means you can make nested calls to it (don't you just love 
those LOCALS).

If you like MBDBEDIT() you are free to use it in your application, if you 
include it in a commercial or for hire application please send me a line or 
two (it is always good to know when your work is appreciated).

I have used MBDBEDIT() in several applications, but if you discover bugs, or
additional enhancements, please let me know and I'll update it.

Check the Clipper manual/Norton Guides for more info on using DBEDIT() and 
it's behavior.  I have included a brief example using MBDBEDIT().

Isn't Clipper wonderful!

Enjoy,
Mark Butler

Thanks to Don Caton for some column:block help.

***********************************************
EXAMPLE:

The following example shows an abreviated MBDBEDIT() call with a 10 minute time 
out and the first column on the left frozen.  Also there is code used during 
the start up mode to; alter the heading/column separators (could have been 
done with the parameters but I did it this way for illustration); set a special
color for 1 specific column; position the browse cursor and browse display to
other than the default (1st left).  Also the User Function is specified as a 
code block so it can be STATIC.  The User-Function accepts a third parameter,
the object, so it can manipulate the TBrowse directly.

STATIC PROCEDURE browse_something
USE something
* init arrays
* my_fields   := // a list of field names
* my_headings := // a list of headings for each field
* my_pictures := // a list of pictures for formating each field

mbdbedit(10,10,20,60,my_fields,{|x,y,z|my_udf(x,y,z)},my_pictures,my_headings,,,,,{ {'TO',600},{'FZ',1} })
CLEAR SCREEN
CLOSE DATABASES
RETURN

STATIC FUNCTION my_udf (de_mode,de_col,de_object)
LOCAL de_return := DE_CONT, col, n

DO CASE
CASE de_mode == DE_START
  * start up mode, this is where further configuration of the object is
  * possible, prior to display.
   de_object:headSep := ''  // re-size heading separator
   de_object:colSep := ''    // and column separator
   * change color for column/field 4
   *     Color Table Index:  1     2     3    4    5
   de_object:colorSpec := 'w/b, gr+/n, r/b, r/w, r/w'
   col := de_object:getColumn(4)
   col:colorBlock := {|x| IIF(VALTYPE(x) = 'N',IIF( x > 0, {4, 5}, {3, 4} ),{3,4})}
   col:defColor := {3, 4}
   de_object:configure()  // force object to reconfigure
   de_object:invalidate() // and redraw
   * move browse display and browse cursor to first column to edit
   FOR n := 1 TO 4
      de_object:panRight()
   NEXT
   de_object:colPos := 5
CASE de_mode == DE_IDLE
  * code to handle idle condition
CASE de_mode == DE_HITTOP
  * code to handle top of file
CASE de_mode == DE_HITBOTTOM
  * code to handle bottom of file
CASE de_mode == DE_EMPTY
  * code to handle an empty file
CASE de_mode == DE_EXCEPT
   * code to handle a key stroke exception
ENDCASE
RETURN de_return
********* END OF EXAMPLE *******
*/

#include 'inkey.ch'
#include 'setcurs.ch'
#include 'dbedit.ch'
#include 'dbstruct.ch'

#define DE_START   5   // MBDBEDIT()() return status message that MBDBEDIT()() has started, user may alter prior to display
#define TRUE      .T.
#define FALSE     .F.

#undef  MB_EXTENSIONS  // compile time directive code in/ex/clusion for some of my other extensions/enhancements


FUNCTION mbdbedit (ntop, nleft, nbottom, nright, afields, buser_udf, apictures, aheadings, ahead_seps, acol_seps, afoot_seps, afootings, aoptions)
/* 
PURPOSE:  Completely substitute Clipper DBEDIT() and enhance with the 
          TBrowse Class.  Enhancements/Features include:
             1. Caller access to the TBrowse object.
             2. User-defined function can be a code block allowing it to be
                a STATIC function instead of global.
             3. Access to internals (via the object) prior to display.
             4. Optional time out and an optional action to perform if a 
                time out occurs (via code block).
             5. Optionally freeze 1 or more columns on the left side of the 
                display.
             6. Optionally create a Wrap effect: when browse cursor is at left
                most or right most columns.

PARAMETERS: (Check the Clipper Manual for more info on using DBEDIT())

   ntop       = Top row of window, default row 0.

   nleft      = Top left column of window, default column 0.

   nbottom    = Bottom row of window, default MAXROW().

   nright     = Bottom right column of window, default MAXCOL()

   afields    = is an array of character expressions containing database field 
                names or expressions to use as column values for each row displayed.
                If this argument is not specified, MBDBEDIT() displays all fields
                in the current work area as columns.

  buser_udf   = is a (character expression name or code block) of a 
                user-defined function (UDF) that executes when an 
                unrecognizable key is pressed or there are no keys pending in 
                the keyboard buffer.  The function can be specified 
                either as:

                1. A character expression without parentheses or arguments.
                   This is the usual Clipper DBEDIT() method.  For example:

                     "my_udf"

                OR

                2. As a code block accepting three parameters.  This is the
                   enhanced MBDBEDIT() method.   For example:

                     {|x,y,z|my_udf(x,y,z)}

                In both cases when the UDF is called, the following arguments
                are automatically passed:
                   1. Mode: passed with the same numeric value as DBEDIT().
                      The exception is at start up, which is after
                      the TBrowse object has been assembled and prior to 
                      display, the mode passed will be (5, #define DE_START).  
                      This gives you the opportunity to alter the TBrowse 
                      object directly before display.
                   2. The index of the current column in <afields> passed as
                      a numeric argument.
                   3. The TBrowse object.  Here is the power, this gives you
                      ALL the access you can get into the object.

  apictures   = is a parallel array of picture clauses to format each column.
                Specifing a character string instead of an array displays all
                columns with the same format.

  aheadings   = is a parallel array of character expressions that define the 
                headings for each column. Specifing a character expression 
                instead of an array gives the same heading for all columns.  
                To display a multi-line heading, embed a semicolon in the 
                heading expression where you want the string to break.  If not 
                specified, column headings are taken from the <afields> array, 
                or the field names in the current work area if the <afields> 
                argument is not specified.

   ahead_seps = is a parallel array of character expressions that define the 
                characters used to draw horizontal lines separating column 
                headings from the field display area. Specifing a character 
                expression instead of an array uses the same heading separator
                for all columns.  If this argument is not specified, the 
                default separator is a double graphics line.

    acol_seps = is a parallel array of character expressions that define the 
                characters used to draw vertical lines separating the columns
                Specifing a character expression instead of an array uses the 
                same separator for all columns.  If this argument is not 
                specified, the default separator is a single graphics line.

   afoot_seps = is a parallel array of character expressions that define the 
                characters used to draw horizontal lines separating column 
                footings from the field display area. Specifing a character 
                expression instead of an array uses the same footing separator
                for all columns.  If this argument is not specified, there is
                no footing separator.

    afootings = is a parallel array of character expressions that define 
                footings for each column.  Specifing a character expression 
                instead of an array gives the same footing for all columns.  
                To display a multi-line heading, embed a semicolon in the 
                footing expression where you want the string to break.  If 
                this argument is not specified, there are no column footings.

     aoptions = is a two-dimensional array of options used to override
                defaults or activate features of MBDBEDIT().  The first column
                of the array contains a character expression used to identify 
                the option, the second column contains any expected value for 
                the option.  For example:

                   To specify a 10 minute time out after no keyboard 
                   activity <aoptions> would literally be:  { {'TO',600} }

               The following lists the feature or overridable (sic) default, 
               its character expression option ID, and any expected value.

               ID      EXPECTED VALUE                        DESCRIPTION
               --   -------------------------   -----------------------------------------------------
               TO   Numeric number of seconds   Provides a time out feature if no user keyboard input.  
                    to wait for a key press     Default is 0 or no time out.  Unless a time out code 
                                                block (TOB) is specified, MBDBEDIT() returns when a
                                                time out occurs.

               TOB  Code block, executed        Provides a time out action if a time out occurs.  The
                    if a time out occurs        code block should return a logical value.  The block
                                                will be called with the TBrowse object as an argument.
                                                If it returns true (.T.), MBDBEDIT() will continue, 
                                                if it returns false (.F.), MBDBEDIT() will return.

               FZ   Numeric, the column number  Provides the ability to freeze 1 or more columns on the
                    to freeze on left side      left side of the display.
                    of the display

               WR   none                        When the cursor is in the left most column and a cursor
                                                left is pressed, the display will WRap to right most column,
                                                likewise, when the cursor is in the right most column and a
                                                cursor right is pressed, the display will WRap to left most 
                                                column.

WARNINGS: Minimal type checking and parameter validation is done!  The type
          checking that is performed is done to duplicate DBEDIT() behavior.
*/

LOCAL brow_obj, column, n, ncurssave, has_udf := FALSE, timeout, timeoutblk
LOCAL freeze_col, cr_row_col := {ROW(),COL()}, db_struct,wrap := FALSE
#ifdef MB_EXTENSIONS
   timeout    := set_kbtout() * 60  // convert current global keyboard time out minutes to seconds
   timeoutblk := set_toblk()        // set time out action to current global time action code block
   timed_out(FALSE)                 // init global timed out
#else
   timeout    := 0                  // no time out default
   timeoutblk := {||FALSE}          // default time out action returns to caller
#endif

* Handle Parameters
IF VALTYPE(ntop) != 'N'
   ntop := 0             // top row default
ENDIF
IF VALTYPE(nleft) != 'N'
   nleft := 0            // top column default
ENDIF
IF VALTYPE(nbottom) != 'N'
   nbottom := MAXROW()   // bottom row default
ENDIF
IF VALTYPE(nright) != 'N'
   nright := MAXCOL()    // bottom column default
ENDIF

* Set fields/columns for MBDBEDIT()
IF VALTYPE(afields) != 'A'
   IF 0 == LEN(db_struct := DBSTRUCT())  // default to all fields in current work area
      * error, nothing to work with
      RETURN NIL
   ENDIF
   afields := {}
   FOR n := 1 TO LEN(db_struct)
      AADD(afields,db_struct[n,DBS_NAME])
   NEXT
ELSE
   * <afields> is an array, but make sure the type contents are ok
   FOR n := 1 to LEN(afields)
      IF VALTYPE(afields[n]) != 'C'
         afields[n] := ''  // default to null string
      ENDIF
   NEXT
ENDIF

* Set column headings
IF VALTYPE(aheadings) == 'C' // same heading specified for all fields
   aheadings := AFILL(ARRAY(LEN(afields)),aheadings)
ELSE
   IF VALTYPE(aheadings) != 'A'
      aheadings := ACLONE(afields)  // default to array afields
   ENDIF
   * it is an array by now, but make sure of the size
   IF LEN(aheadings) < LEN(afields)
      ASIZE(aheadings,LEN(afields))
   ENDIF
   * check type of each element for valid char string
   FOR n := 1 to LEN(aheadings)
      IF VALTYPE(aheadings[n]) != 'C'
         aheadings[n] := afields[n]  // default to field item in parallel array
      ENDIF
   NEXT
ENDIF

* Make new database browse object
brow_obj := TBrowseDB(ntop, nleft, nbottom, nright)

* Add a column to the object for each field specified in the array of fields
FOR n := 1 to LEN(afields)
   * Make the new column
   IF VALTYPE( &(afields[n]) ) == 'M' // testing for memo fields
      column := TBColumnNew(aheadings[n], &( "{|| IIF(EMPTY( " + afields[n] + "),'<memo>','<MEMO>') }") )
   ELSE
      column := TBColumnNew(aheadings[n], &( "{|| " + afields[n] + " }") )
   ENDIF
   brow_obj:addColumn(column)
NEXT

* Set pictures for display formating
IF VALTYPE(apictures) == 'C'  // specified single string for all pictures
   FOR n := 1 TO LEN(afields)
      column := brow_obj:getColumn(n)
      column:block := &( "{|| TRANSFORM( " + afields[n] + ", [" + apictures + "] ) }" )
   NEXT
ELSE
   IF VALTYPE(apictures) == 'A' // specified picture for each column
      FOR n := 1 TO LEN(apictures)
         IF VALTYPE(apictures[n]) == 'C'  // in case not all are to be modified
            column := brow_obj:getColumn(n)
            column:block := &( "{|| TRANSFORM( " + afields[n] + ", [" + apictures[n] + "] ) }" )
         ENDIF
      NEXT
   ENDIF
ENDIF

* Set heading separators
brow_obj:headSep := ""
IF VALTYPE(ahead_seps) == 'C'  // specified single string for all heading separators
   brow_obj:headSep := ahead_seps
ELSE
   IF VALTYPE(ahead_seps) == 'A'  // specified heading separator for each column
      FOR n := 1 TO LEN(ahead_seps)
         IF VALTYPE(ahead_seps[n]) == 'C'  // in case not all are to be modified
            column := brow_obj:getColumn(n)
            column:headSep := ahead_seps[n]
         ENDIF
      NEXT
   ENDIF
ENDIF

* Set column separators
brow_obj:colSep := "  "
IF VALTYPE(acol_seps) == 'C'  // specified single string for all column separators
   brow_obj:colSep := acol_seps
ELSE
   IF VALTYPE(acol_seps) == 'A'  // specified column separator for each column
      FOR n := 1 to LEN(acol_seps)
         IF VALTYPE(acol_seps[n]) == 'C'  // in case not all are to be modified
            column := brow_obj:getColumn(n)
            column:colSep := acol_seps[n]
         ENDIF
      NEXT
   ENDIF
ENDIF

* Set footing separators
IF VALTYPE(afoot_seps) == 'C'  // specified single string for all footing separators
   FOR n := 1 to LEN(afields)
      column := brow_obj:getColumn(n)
      column:footSep := afoot_seps
   NEXT
ELSE
   IF VALTYPE(afoot_seps) == 'A'  // specified footing separator for each column
      FOR n := 1 to LEN(afoot_seps)
         IF VALTYPE(afoot_seps[n]) == 'C'  // in case not all are to be modified
            column := brow_obj:getColumn(n)
            column:footSep := afoot_seps[n]
         ENDIF
      NEXT
   ENDIF
ENDIF

* Set column footings
IF VALTYPE(afootings) == 'C'  // specified single string for all column footings
   FOR n := 1 to LEN(afields)
      column := brow_obj:getColumn(n)
      column:footing := afootings
   NEXT
ELSE
   IF VALTYPE(afootings) == 'A'  // specified column footing for each column
      FOR n := 1 to LEN(afootings)
         IF VALTYPE(afootings[n]) == 'C'  // in case not all are to be modified
            column := brow_obj:getColumn(n)
            column:footing := afootings[n]
         ENDIF
      NEXT
   ENDIF
ENDIF

* Other Special Options
IF VALTYPE(aoptions) == 'A'
   * Check array of options
   FOR n := 1 TO LEN(aoptions)
      IF aoptions[n,1] == 'TO'  // time out option
         timeout := aoptions[n,2]  // time out value in seconds expected in second element
      ENDIF
      IF aoptions[n,1] == 'TOB'  // time out code block action option
         timeoutblk := aoptions[n,2]  // reset time out action to custom time action code block expected in second element
      ENDIF
      IF aoptions[n,1] == 'FZ'  // freeze column specifed
         freeze_col := aoptions[n,2]  // freeze column expected in second element
      ENDIF
      IF aoptions[n,1] == 'WR'  // wrap columns specifed
         wrap := TRUE
      ENDIF
   NEXT
ENDIF

IF VALTYPE(freeze_col) == 'N'
   * freeze leftmost display column setting
   brow_obj:freeze := freeze_col
ENDIF

ncurssave := SETCURSOR(SC_NONE)  // save current cursor and set none

IF VALTYPE(buser_udf) == 'B'
   has_udf := TRUE
ELSE
   IF VALTYPE(buser_udf) == 'C'  // old DBEDIT Style
      buser_udf := &('{|x,y,z|' + buser_udf + '(x,y,z)}')  // turn user UDF into a code block with additional 3rd parm for tbrowse object
      has_udf := TRUE
   ENDIF
ENDIF

* Everthing has been initialized, run the browse object
run_mbdbedit (@brow_obj,@buser_udf,has_udf,timeout,@timeoutblk,wrap)

* done, clean up
SETCURSOR(ncurssave)
DEVPOS(cr_row_col[1],cr_row_col[2]) // put cursor back where it was
RETURN NIL



****************************************************************************
STATIC FUNCTION run_mbdbedit (brow_obj,buser_udf,has_udf,timeout,timeoutblk,wrap)
* PURPOSE:  This UDF actually manages the Browse.  It exists as a separate
*           procedure (from the initializing process) in order to cause the 
*           VMM to place it in a new segment boundry to help eliminate 
*           "not enough memory"  errors on large browse operations.  This UDF
*           contains the minimum number of vars to run the browse.
LOCAL c_row,c_col,user_ret,nkey,bkey_block

* You may want to uncomment the following two lines, it will flush file buffers
* and may give VMM time to do some garbage collection, and possibly free 
* memory in a large TBrowse operations (i.e. full screen MBDBEDIT() VGA 50 l
* line mode in low memory situations).  NOTE: It will slow the start of the
* browse.

* COMMIT
* MEMORY(-1)

IF has_udf
   * We are ready to get started, call user-UDF in mode DE_START, in case
   * caller wants to make other changes to the browse object before displaying it
   IF DE_ABORT == EVAL(buser_udf,DE_START,brow_obj:colPos,brow_obj)
      * caller has requested an abort from DE_START mode
      RETURN NIL
   ENDIF
ENDIF
* main control loop
DO WHILE TRUE
   * Don't allow cursor to move into frozen columns
   IF ( brow_obj:colPos <= brow_obj:freeze )
      brow_obj:colPos := brow_obj:freeze + 1
   ENDIF

   * Stabilize the display
   DO WHILE (NEXTKEY() == 0) .AND. (.NOT. brow_obj:stabilize())
       * abort display stabilize loop if a key is waiting
   ENDDO

   IF brow_obj:stable .AND. has_udf
      c_row := ROW()
      c_col := COL()
      /* Display is stable; check for empty file, hit top or bottom of file,
         or just idle: position screen cursor at browse cursor and call 
         user UDF for next action */
      DEVPOS(c_row,c_col)
      user_ret := EVAL(buser_udf,IIF(LASTREC() = 0,DE_EMPTY,IIF(brow_obj:hitTop,DE_HITTOP,IIF(brow_obj:hitBottom,DE_HITBOTTOM,DE_IDLE))),brow_obj:colPos,brow_obj)
      DO CASE
      CASE user_ret == DE_ABORT     // DBEDIT() return value to quit
         EXIT
      CASE user_ret == DE_CONT      // DBEDIT() return value to continue
         brow_obj:refreshCurrent()
      CASE user_ret == DE_REFRESH   // DBEDIT() return value to repaint the screen
         brow_obj:refreshAll()
      ENDCASE
      * Stabilize the display after return from user UDF
      DO WHILE (NEXTKEY() == 0) .AND. (.NOT. brow_obj:stabilize())
         /* (abort display stabilize loop if a key is waiting) */
      ENDDO
   ENDIF

   * Everything's done; waiting for a key
   nkey := INKEY(timeout)

   IF 0 == nkey
      * timed out
      #ifdef MB_EXTENSIONS
         timed_out(TRUE)  // set global timed out
      #endif
      IF .NOT. EVAL(timeoutblk,brow_obj)  // call time out action block
         * block returned with request to abort MBDBEDIT()
         EXIT
      ENDIF
   ENDIF

   * process key

   IF ((bkey_block := SETKEY(nkey)) != NIL )   // check for a SET KEY keystroke
      * run SET KEY block
      EVAL(bkey_block, PROCNAME(2), PROCLINE(2), READVAR())
   ENDIF

   DO CASE
   CASE  nkey == K_UP
      brow_obj:up()
   CASE  nkey == K_DOWN
      brow_obj:down()
   CASE  nkey == K_LEFT
      IF wrap .AND. brow_obj:colPos == 1
         * wrap option set and in first column, go to last column
         TONE(2000,2)  // audible signal
         brow_obj:panEnd()
         * respect frozen columns setting, if any
         IF ( brow_obj:colPos <= brow_obj:freeze )
            brow_obj:colPos := brow_obj:freeze + 1
         ENDIF
      ELSE
         brow_obj:left()
      ENDIF
   CASE  nkey == K_RIGHT
      IF wrap .AND. brow_obj:colPos == brow_obj:colCount
         * wrap option set and in last column, go to first column
         TONE(2000,2)  // audible signal
         brow_obj:panHome()
         * respect frozen columns setting, if any
         IF ( brow_obj:colPos <= brow_obj:freeze )
            brow_obj:colPos := brow_obj:freeze + 1
         ENDIF
      ELSE
         brow_obj:right()
      ENDIF
   CASE  nkey == K_CTRL_LEFT .OR. nkey == K_SH_TAB
      brow_obj:panLeft()
   CASE  nkey == K_CTRL_RIGHT .OR. nkey == K_TAB
      brow_obj:panRight()
   CASE  nkey == K_HOME
      brow_obj:home()
   CASE  nkey == K_END
      brow_obj:end()
   CASE  nkey == K_CTRL_HOME
      brow_obj:panHome()
   CASE  nkey == K_CTRL_END
      brow_obj:panEnd()
   CASE  nkey == K_PGUP
      brow_obj:pageUp()
   CASE  nkey == K_PGDN
      brow_obj:pageDown()
   CASE  nkey == K_CTRL_PGUP
      brow_obj:goTop()
   CASE  nkey == K_CTRL_PGDN
      brow_obj:goBottom()
   OTHERWISE
      IF .NOT. has_udf
         * no user UDF,  handle exceptions
         IF nkey == K_ENTER .OR. nkey == K_ESC
            EXIT  // terminate
         ENDIF
      ELSE
         * Stabilize the display before calling User UDF
         DO WHILE .NOT. brow_obj:stabilize()
         ENDDO
         user_ret := EVAL(buser_udf,DE_EXCEPT,brow_obj:colPos,brow_obj)
         DO CASE
         CASE user_ret == DE_ABORT     // DBEDIT() return value to quit
            EXIT  // terminate
         CASE user_ret == DE_CONT      // DBEDIT() return value to continue
            brow_obj:refreshCurrent()
         CASE user_ret == DE_REFRESH   // DBEDIT() return value to repaint the screen
            brow_obj:refreshAll()
         ENDCASE
      ENDIF
   ENDCASE
ENDDO
RETURN NIL

*: EOF: MBDBEDIT.PRG