/**********************************************************************
LOCALIZE.PRG

NOTES: Compile with /n/w/a/m or use LOCALIZE.RMK

       Requires the FILEIO.PRG file included in the SOURCE\SAMPLE
       subdirectory of Clipper 5.x.

       Since LOCALIZE invokes the Clipper program directly, a fair amount of
       free RAM (i.e., > 570K) is required.

       Makes use of an undocument feature of AEVAL() in which the ordinal
       number of each element in the array can be passed to the code block
       being AEVAL'ed.


Generates a formatted list of variables that were undeclared within a
Clipper program.  Useful for eliminating Compiler warning messages when
updating Summer '87 code with 5.x.

The output of this program is a text file containing a list of LOCAL
variables in the format:

   LOCAL varname1 ,;
         varname2 ,;
         varname3 ,;
         etc.

The resulting text file can be imported into the top of your .prg file.

This method of eliminating undeclared variables isn't 100% accurate (e.g.,
the program will follow the compiler's lead and assume that .dbf FIELDs not
written using the 'ALIAS->FIELDname' format are MEMVARs).  The program also
assumes that every variable in a program should be declared as LOCAL.

In the end however, I found that even though this program isn't particularly
pretty or flexible, using it sure beat going through 15,000+ lines of code
trying to identify each variable that was previously undeclared.

Needless to say, any comments, improvements, criticisms, etc. are welcome.


Steven J. Serenska
The REEF Development Company, Inc.

CIS: 73607,3517

*****************************************************************************/
*
*
*
*:*********************************************************************
*:
*: Procedure file: C:\CLIPPER\LOCALIZE.PRG
*:
*:         System: Localize
*:         Author: Steven J. Serenska
*:      Copyright (c) 1992, The REEF Development Company, Inc.
*:  Last modified: 02/21/92     11:05
*:
*:  Procs & Fncts: MAIN()
*:
*:          Calls: MAIN()             (function  in LOCALIZE.PRG)
*:
*:      Documented 02/22/92 at 13:47                SNAP!  version 5.00
*:*********************************************************************
*
#command DEFAULT <param> TO <val> [, <paramn> TO <valn> ]          ;
                                                                   ;
         => <param>  := IIF( <param>  == NIL, <val>,  <param>  )   ;
         [; <paramn> := IIF( <paramn> == NIL, <valn>, <paramn> ) ]

*!*********************************************************************
*!
*!       Function: MAIN()
*!
*!      Called by: LOCALIZE.PRG
*!
*!          Calls: FREADLN()          (function  in FILEIO.PRG)
*!               : FEOF()             (function  in FILEIO.PRG)
*!               : FWRITELN()         (function  in FILEIO.PRG)
*!
*!    Other Files: !!_LOCAL.TMP
*!               : COUTFILE
*!
*!*********************************************************************
*
FUNCTION Main( cProgram, cOutFile, cSortIt )
   LOCAL hSourceFile    ,;
         hTargetFile    ,;
         cProgLine      ,;
         nMaxWidth := 0

   LOCAL aLocals := {}

   DEFAULT cOutFile TO "LOCALS.TXT" ,;
           cSortIt  TO ""

   IF cProgram == NIL
      TONE( 100, 1 )
      QOUT( "Parameters: filename.prg [outfile.ext] [/S]" )
      QOUT()
   ELSE
      //
      // The '/s' switch causes the Clipper compiler to perform a Sytax check
      // only; an OBJ file is NOT generated.  All output from the /w switch is
      // sent to the temporary file '!!_LOCAL.TMP'.
      //
      RUN ("Clipper " + cProgram + " /s/w > !!_LOCAL.TMP" )
      *
      hSourceFile := FOPEN( "!!_LOCAL.TMP" )
      hTargetFile := FCREATE( cOutFile )
      cProgLine   := FReadLn( hSourceFile, 1, 251 )
      *
      DO WHILE .NOT. FEof( hSourceFile )
         //
         // Compiler warnings of C1003 or C1004 indicate that a program variable
         // was not properly initialized.
         //
         IF "C1003" $ cProgLine .OR. "C1004" $ cProgLine
            //
            // First, strip everything to the left of the variable name.
            // Second, strip everything to the right and add a comma and semi-
            // colon.
            //
            cProgLine := SUBSTR( cProgLine, AT("'", cProgLine) + 1 )
            cProgLine := LEFT( cProgLine, LEN( cProgLine ) - 3 ) + " ,;"
            *
            //
            // Add the variable to the output array only if it has not been
            // added before.
            //
            IF ASCAN( aLocals, cProgLine ) == 0
               AADD ( aLocals, cProgLine )
               nMaxWidth := MAX( nMaxWidth, LEN( cProgLine ) )
            ENDIF
         ENDIF
         *
         cProgLine := FReadLn( hSourceFile, 1, 251 )
      ENDDO
      *
      IF .NOT. EMPTY( aLocals )
         IF UPPER( cSortIt ) == "/S" .OR. ;
               (PCOUNT() == 2 .AND. UPPER( cOutFile ) == "/S")

            ASORT( aLocals )
         ENDIF
         //
         // AEVAL 1) Go through the variables and space the comma/semi-colon
         //          pair out to the same TAB stop.  Neatness counts!
         //
         // AEVAL 2) Go through the variables and indent each 9 spaces from
         //          the left margin.  NOTE THE USE OF THE UNDOCUMENTED 'num'
         //          ARGUMENT IN THE AEVAL() CODEBLOCK FUNCTION!
         //
         // AEVAL 3) Go through the array and write each element as a line to
         //          the output file.
         //
         *
         AEVAL( aLocals,                                                          ;
                { |element,num| aLocals[num] :=                                   ;
                  STRTRAN( element, ",", SPACE( nMaxWidth - LEN(element) ) + "," );
                }                                                                 ;
              )
         *
         AEVAL( aLocals, { |element,num| aLocals[num] := SPACE(9) + element } )
         *
         aLocals[1]            := STRTRAN( aLocals[1], SPACE(9), "   LOCAL " )
         aLocals[LEN(aLocals)] := TRIM( STRTRAN( ATAIL(aLocals), ",;" ) )
         *
         AEVAL( aLocals, { |element| FWriteLn( hTargetFile, element ) } )
      ENDIF
      *
      //
      // Close up, clean up.
      //
      FCLOSE( hSourceFile )
      FCLOSE( hTargetFile )
      FERASE( "!!_LOCAL.TMP" )
   ENDIF
   *
RETURN NIL
*
*
*
* EOFile: LOCALIZE.PRG
******************************************************************************
